import iam = require('@aws-cdk/aws-iam');
import { PolicyDocument, PolicyStatement } from '@aws-cdk/aws-iam';
import { Construct, IResource, RemovalPolicy, Resource } from '@aws-cdk/core';
import { Alias } from './alias';
/**
 * A KMS Key, either managed by this CDK app, or imported.
 */
export interface IKey extends IResource {
    /**
     * The ARN of the key.
     *
     * @attribute
     */
    readonly keyArn: string;
    /**
     * Defines a new alias for the key.
     */
    addAlias(alias: string): Alias;
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement: PolicyStatement, allowNoOp?: boolean): void;
    /**
     * Grant the indicated permissions on this key to the given principal
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant decryption permisisons using this key to the given principal
     */
    grantDecrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption permisisons using this key to the given principal
     */
    grantEncrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption and decryption permisisons using this key to the given principal
     */
    grantEncryptDecrypt(grantee: iam.IGrantable): iam.Grant;
}
declare abstract class KeyBase extends Resource implements IKey {
    /**
     * The ARN of the key.
     */
    abstract readonly keyArn: string;
    /**
     * Optional policy document that represents the resource policy of this key.
     *
     * If specified, addToResourcePolicy can be used to edit this policy.
     * Otherwise this method will no-op.
     */
    protected abstract readonly policy?: PolicyDocument;
    /**
     * Defines a new alias for the key.
     */
    addAlias(alias: string): Alias;
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement: PolicyStatement, allowNoOp?: boolean): void;
    /**
     * Grant the indicated permissions on this key to the given principal
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant decryption permisisons using this key to the given principal
     */
    grantDecrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption permisisons using this key to the given principal
     */
    grantEncrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption and decryption permisisons using this key to the given principal
     */
    grantEncryptDecrypt(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Construction properties for a KMS Key object
 */
export interface KeyProps {
    /**
     * A description of the key. Use a description that helps your users decide
     * whether the key is appropriate for a particular task.
     *
     * @default - No description.
     */
    readonly description?: string;
    /**
     * Initial alias to add to the key
     *
     * More aliases can be added later by calling `addAlias`.
     *
     * @default - No alias is added for the key.
     */
    readonly alias?: string;
    /**
     * Indicates whether AWS KMS rotates the key.
     *
     * @default false
     */
    readonly enableKeyRotation?: boolean;
    /**
     * Indicates whether the key is available for use.
     *
     * @default - Key is enabled.
     */
    readonly enabled?: boolean;
    /**
     * Custom policy document to attach to the KMS key.
     *
     * @default - A policy document with permissions for the account root to
     * administer the key will be created.
     */
    readonly policy?: PolicyDocument;
    /**
     * Whether the encryption key should be retained when it is removed from the Stack. This is useful when one wants to
     * retain access to data that was encrypted with a key that is being retired.
     *
     * @default RemovalPolicy.Retain
     */
    readonly removalPolicy?: RemovalPolicy;
}
/**
 * Defines a KMS key.
 *
 * @resource AWS::KMS::Key
 */
export declare class Key extends KeyBase {
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope  the construct that will "own" the imported key.
     * @param id     the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     */
    static fromKeyArn(scope: Construct, id: string, keyArn: string): IKey;
    readonly keyArn: string;
    protected readonly policy?: PolicyDocument;
    constructor(scope: Construct, id: string, props?: KeyProps);
    /**
     * Let users from this account admin this key.
     * @link https://aws.amazon.com/premiumsupport/knowledge-center/update-key-policy-future/
     */
    private allowAccountToAdmin;
}
export {};
