"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const alias_1 = require("./alias");
const kms_generated_1 = require("./kms.generated");
class KeyBase extends core_1.Resource {
    /**
     * Defines a new alias for the key.
     */
    addAlias(alias) {
        return new alias_1.Alias(this, `Alias${alias}`, { aliasName: alias, targetKey: this });
    }
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement, allowNoOp = true) {
        const stack = core_1.Stack.of(this);
        if (!this.policy) {
            if (allowNoOp) {
                return;
            }
            throw new Error(`Unable to add statement to IAM resource policy for KMS key: ${JSON.stringify(stack.resolve(this.keyArn))}`);
        }
        this.policy.addStatements(statement);
    }
    /**
     * Grant the indicated permissions on this key to the given principal
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalAndResource({
            grantee,
            actions,
            resourceArns: [this.keyArn],
            resource: this,
            resourceSelfArns: ['*']
        });
    }
    /**
     * Grant decryption permisisons using this key to the given principal
     */
    grantDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt');
    }
    /**
     * Grant encryption permisisons using this key to the given principal
     */
    grantEncrypt(grantee) {
        return this.grant(grantee, 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
    /**
     * Grant encryption and decryption permisisons using this key to the given principal
     */
    grantEncryptDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt', 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
}
/**
 * Defines a KMS key.
 *
 * @resource AWS::KMS::Key
 */
class Key extends KeyBase {
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope  the construct that will "own" the imported key.
     * @param id     the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     */
    static fromKeyArn(scope, id, keyArn) {
        class Import extends KeyBase {
            constructor() {
                super(...arguments);
                this.keyArn = keyArn;
                this.policy = undefined;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.policy) {
            this.policy = props.policy;
        }
        else {
            this.policy = new aws_iam_1.PolicyDocument();
            this.allowAccountToAdmin();
        }
        const resource = new kms_generated_1.CfnKey(this, 'Resource', {
            description: props.description,
            enableKeyRotation: props.enableKeyRotation,
            enabled: props.enabled,
            keyPolicy: this.policy,
        });
        this.keyArn = resource.attrArn;
        resource.applyRemovalPolicy(props.removalPolicy);
        if (props.alias !== undefined) {
            this.addAlias(props.alias);
        }
    }
    /**
     * Let users from this account admin this key.
     * @link https://aws.amazon.com/premiumsupport/knowledge-center/update-key-policy-future/
     */
    allowAccountToAdmin() {
        const actions = [
            "kms:Create*",
            "kms:Describe*",
            "kms:Enable*",
            "kms:List*",
            "kms:Put*",
            "kms:Update*",
            "kms:Revoke*",
            "kms:Disable*",
            "kms:Get*",
            "kms:Delete*",
            "kms:ScheduleKeyDeletion",
            "kms:CancelKeyDeletion",
            "kms:GenerateDataKey"
        ];
        this.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions,
            principals: [new iam.AccountRootPrincipal()]
        }));
    }
}
exports.Key = Key;
//# sourceMappingURL=data:application/json;base64,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