"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const cdk = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const assertions_1 = require("aws-cdk-lib/assertions");
function deployNewFunc(stack) {
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
    };
    return new lib_1.LambdaToDynamoDB(stack, 'test-lambda-dynamodb-stack', props);
}
function useExistingFunc(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.PYTHON_3_6,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const props = {
        existingLambdaObj: new lambda.Function(stack, 'MyExistingFunction', lambdaFunctionProps),
        dynamoTableProps: {
            billingMode: dynamodb.BillingMode.PROVISIONED,
            readCapacity: 3,
            writeCapacity: 3,
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            }
        },
    };
    return new lib_1.LambdaToDynamoDB(stack, 'test-lambda-dynamodb-stack', props);
}
test('check lambda function properties for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testlambdadynamodbstackLambdaFunctionServiceRole758347A1",
                "Arn"
            ]
        },
        Runtime: "nodejs16.x",
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                DDB_TABLE_NAME: {
                    Ref: "testlambdadynamodbstackDynamoTable8138E93B"
                }
            }
        }
    });
});
test('check dynamo table properties for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        }
    });
});
test('check lambda function role for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/lambda/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "LambdaFunctionServiceRolePolicy"
            }
        ]
    });
});
test('check lambda function policy default table permissions', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "dynamodb:BatchGetItem",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator",
                        "dynamodb:Query",
                        "dynamodb:GetItem",
                        "dynamodb:Scan",
                        "dynamodb:ConditionCheckItem",
                        "dynamodb:BatchWriteItem",
                        "dynamodb:PutItem",
                        "dynamodb:UpdateItem",
                        "dynamodb:DeleteItem",
                        "dynamodb:DescribeTable"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testlambdadynamodbstackDynamoTable8138E93B",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check lambda function properties for deploy: false', () => {
    const stack = new cdk.Stack();
    useExistingFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "MyExistingFunctionServiceRoleF9E14BFD",
                "Arn"
            ]
        },
        Runtime: "python3.6"
    });
});
test('check lambda function role for existing function', () => {
    const stack = new cdk.Stack();
    useExistingFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        ManagedPolicyArns: [
            {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                    ]
                ]
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.dynamoTable).toBeDefined();
});
test('check exception for Missing existingObj from props', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.LambdaToDynamoDB(stack, 'test-iot-lambda-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check for no prop', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        }
    };
    new lib_1.LambdaToDynamoDB(stack, 'test-iot-lambda-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testiotlambdastackLambdaFunctionServiceRoleF72A85A9",
                "Arn"
            ]
        },
        Runtime: "nodejs16.x",
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                DDB_TABLE_NAME: {
                    Ref: "testiotlambdastackDynamoTable76858356"
                }
            }
        }
    });
});
test('check lambda function policy ReadOnly table permissions', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        tablePermissions: 'Read'
    };
    new lib_1.LambdaToDynamoDB(stack, 'test-lambda-dynamodb-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "dynamodb:BatchGetItem",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator",
                        "dynamodb:Query",
                        "dynamodb:GetItem",
                        "dynamodb:Scan",
                        "dynamodb:ConditionCheckItem",
                        "dynamodb:DescribeTable"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testlambdadynamodbstackDynamoTable8138E93B",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check lambda function policy WriteOnly table permissions', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        tablePermissions: 'Write'
    };
    new lib_1.LambdaToDynamoDB(stack, 'test-lambda-dynamodb-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "dynamodb:BatchWriteItem",
                        "dynamodb:PutItem",
                        "dynamodb:UpdateItem",
                        "dynamodb:DeleteItem",
                        "dynamodb:DescribeTable"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testlambdadynamodbstackDynamoTable8138E93B",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check lambda function policy ReadWrite table permissions', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        tablePermissions: 'ReadWrite'
    };
    new lib_1.LambdaToDynamoDB(stack, 'test-lambda-dynamodb-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "dynamodb:BatchGetItem",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator",
                        "dynamodb:Query",
                        "dynamodb:GetItem",
                        "dynamodb:Scan",
                        "dynamodb:ConditionCheckItem",
                        "dynamodb:BatchWriteItem",
                        "dynamodb:PutItem",
                        "dynamodb:UpdateItem",
                        "dynamodb:DeleteItem",
                        "dynamodb:DescribeTable"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testlambdadynamodbstackDynamoTable8138E93B",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check lambda function policy All table permissions', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        tablePermissions: 'All'
    };
    new lib_1.LambdaToDynamoDB(stack, 'test-lambda-dynamodb-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "dynamodb:*",
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testlambdadynamodbstackDynamoTable8138E93B",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check lambda function custom environment variable', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        tableEnvironmentVariableName: 'CUSTOM_DYNAMODB_TABLE'
    };
    new lib_1.LambdaToDynamoDB(stack, 'test-lambda-dynamodb-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs16.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_DYNAMODB_TABLE: {
                    Ref: 'testlambdadynamodbstackDynamoTable8138E93B'
                }
            }
        }
    });
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC without vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.LambdaToDynamoDB(stack, "lambda-to-dynamodb-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatodynamodbstackReplaceDefaultSecurityGroupsecuritygroupD5CEDE41",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC w/vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.LambdaToDynamoDB(stack, "lambda-to-dynamodb-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            ipAddresses: ec2.IpAddresses.cidr("192.68.0.0/16"),
        },
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatodynamodbstackReplaceDefaultSecurityGroupsecuritygroupD5CEDE41",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new cdk.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.LambdaToDynamoDB(stack, "lambda-to-dynamodb-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingVpc: testVpc,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatodynamodbstackReplaceDefaultSecurityGroupsecuritygroupD5CEDE41",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC and existing Lambda function not in a VPC
//
// buildLambdaFunction should throw an error if the Lambda function is not
// attached to a VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC and existing Lambda function not in a VPC", () => {
    // Stack
    const stack = new cdk.Stack();
    const testLambdaFunction = new lambda.Function(stack, 'test-lambda', {
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToDynamoDB(stack, "lambda-to-dynamodb-stack", {
            existingLambdaObj: testLambdaFunction,
            existingVpc: testVpc,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
test("Confirm CheckVpcProps is called", () => {
    // Stack
    const stack = new cdk.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToDynamoDB(stack, "lambda-to-dynamodb-stack", {
            lambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_16_X,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            existingVpc: testVpc,
            deployVpc: true,
        });
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test bad table permission', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        tablePermissions: 'Reed',
    };
    const app = () => {
        new lib_1.LambdaToDynamoDB(stack, 'test-lambda-dynamodb-stack', props);
    };
    // Assertion
    expect(app).toThrowError(/Invalid table permission submitted - Reed/);
});
test('Test that CheckDynamoDBProps is getting called', () => {
    const stack = new cdk.Stack();
    const tableName = 'custom-table-name';
    const existingTable = new dynamodb.Table(stack, 'MyTablet', {
        tableName,
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        }
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        existingTableObj: existingTable,
        dynamoTableProps: {
            tableName,
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
        },
    };
    const app = () => {
        new lib_1.LambdaToDynamoDB(stack, 'test-lambda-dynamodb-stack', props);
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide existingTableObj or dynamoTableProps, but not both.\n/);
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToDynamoDB(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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