# buildsys-dateversion

Easy date-based versioning for Python projects.

[![PyPI](https://img.shields.io/pypi/v/buildsys-dateversion)](https://pypi.org/project/buildsys-dateversion/)

<!--TOC-->

- [buildsys-dateversion](#buildsys-dateversion)
  - [Overview](#overview)
  - [Usage](#usage)
    - [Prerequisites](#prerequisites)
    - [Installation (setuptools)](#installation-setuptools)
    - [Installation (flit)](#installation-flit)
    - [Installation (other)](#installation-other)
  - [Configuration reference](#configuration-reference)
  - [Caveats](#caveats)
  - [License](#license)

<!--TOC-->

## Overview

`buildsys-dateversion` is a Python build system extension to automatically
set a version string based on the current date/time as a project builds.
Once enabled, your project's version number while building a source distribution
or wheel will be automatically generated.

At minimum, the version string will contain the current year & month.
If the generated version would clash with a version already released to PyPI,
additional date/time portions are added to the string (i.e. day, hour, minute)
until the clash is resolved.

`buildsys-dateversion` is a [PEP 517](https://peps.python.org/pep-0517/)
compliant build system and can be used with other PEP 517 implementations,
such as `setuptools` and `flit`.

Here's an example of a sequence of version strings which may result
from using `buildsys-dateversion` for a project:

| Release is made at... | Version string    |
| --------------------- | ----------------- |
| 2025-03-25 12:00      | `2025.3`          |
| 2025-03-31 12:00      | `2025.3.31`       |
| 2025-03-31 13:00      | `2025.3.31.13`    |
| 2025-03-31 14:00      | `2025.3.31.14`    |
| 2025-03-31 14:30      | `2025.3.31.14.30` |
| 2025-03-31 17:30      | `2025.3.31.17`    |
| 2025-04-02 12:00      | `2025.4`          |

## Usage

### Prerequisites

Your project must use [pyproject.toml](https://packaging.python.org/en/latest/specifications/declaring-project-metadata/#declaring-project-metadata).

Your project's sources must contain a `.py` file with a line of the form:

```python
__version__ = "0"
```

(The specific value for `__version__` does not matter, as it will be
overwritten during build, but `"0"` is recommended.)

### Installation (setuptools)

Your `pyproject.toml` should look like this:

```toml
[project]
name = "myproject"
dynamic = ["version"]

[tool.setuptools.dynamic]
version = {attr = "myproject.__version__"}

[build-system]
requires = ["buildsys-dateversion"]
build-backend = "buildsys_dateversion"
```

(Note that `buildsys-dateversion` uses
[pip-compatible build-system defaults](https://pip.pypa.io/en/stable/reference/build-system/pyproject-toml/#fallback-behaviour);
this is the reason why it is not necessary to explicitly configure
`setuptools` as a build backend or to list it in `requires`, as
is necessary in the flit example below.)

### Installation (flit)

Your `pyproject.toml` should look like this:

```toml
[project]
name = "myproject"
dynamic = ["version", "description"]

[build-system]
requires = ["buildsys-dateversion", "flit_core >=3.2,<4"]
build-backend = "buildsys_dateversion"

[tool.buildsys-dateversion]
# buildsys-dateversion only handles generating the version number.
# This tells it which build backend should be used for all the
# other build steps.
build-backend = "flit_core.buildapi"
```

### Installation (other)

`buildsys-dateversion` is theoretically compatible with other
PEP 517 build backends (though only setuptools and flit are tested).
The primary requirement is that the build backend must support
a dynamic `version` based on a `__version__` attr in the project,
as both setuptools and flit do.

```toml
[project]
name = "myproject"

# The chosen PEP 517 backend must support a dynamic version.
dynamic = ["version"]

[build-system]
# Require buildsys-dateversion + the build backend.
requires = ["buildsys-dateversion", "some_build_backend"]
build-backend = "buildsys_dateversion"

[tool.buildsys-dateversion]
# This should be a module implementing the PEP 517 hooks.
build-backend = "some_build_backend.hooks"
```

## Configuration reference

The following values can be set in `pyproject.toml`, in the
`tool.buildsys-dateversion` table.

```toml
[tool.buildsys-dateversion]
# Should point at a module implementing PEP 517 hooks.
# If omitted, default is "setuptools.build_meta:__legacy__"
build-backend = "some_build_backend.hooks"

# A comment added on patched __version__ lines.
version-marker = "generated by buildsys-dateversion"

# Path to the file containing __version__.
# If omitted, the file is located automatically.
version-path = "src/mymodule/__init__.py"
```

## Caveats

Take note of the following before using this project:

* The setting of the `__version__` attribute currently relies on temporarily
  patching files in the source tree, which has some implications:
  * The source tree during a build must not be read-only.
  * If the build process is interrupted, the file defining `__version__` might
    not be restored to its original state.
* `buildsys-dateversion` is implemented as a PEP 517 build backend which
  wraps another PEP 517 build backend. However, this wrapping is not perfect
  and might result in subtle behavior differences from the underlying backend.

## License

This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
