import importlib
import logging
import os
import re
import sys
import urllib.error
import urllib.request
from contextlib import contextmanager
from datetime import datetime

LOG = logging.getLogger("buildsys-dateversion")

if os.getenv("BUILDSYS_DATEVERSION_DEBUG"):
    LOG.setLevel(logging.DEBUG)
    LOG.addHandler(logging.FileHandler(os.getenv("BUILDSYS_DATEVERSION_DEBUG")))

VERSION_PATTERN = re.compile(r"^__version__ *=.*$", flags=re.MULTILINE)


def normalized(version: str):
    # PEP 440 version string normalization.
    #
    # (Not a generic function, just 'good enough' to work on our
    # known strftime templates)
    return ".".join([str(int(x)) for x in version.split(".")])


class DateVersion:
    def __init__(self, config_settings=None):
        self.config_settings: dict = (config_settings or {}).copy()
        self.datetime = datetime.utcnow()
        self._pyproject_toml = None
        self._tool_config = None
        LOG.debug("operating with datetime = %s", self.datetime)

    @property
    def pyproject_toml(self) -> dict:
        if self._pyproject_toml is None:
            if sys.version_info >= (3, 11):
                import tomllib
            else:
                import tomli as tomllib

            with open("pyproject.toml", "rb") as f:
                data = tomllib.load(f)

            LOG.debug("Loaded project metadata: %s", data)

            self._pyproject_toml = data

        return self._pyproject_toml

    @property
    def project_config(self) -> dict:
        return self.pyproject_toml["project"]

    @property
    def tool_config(self) -> dict:
        if self._tool_config is None:
            self._tool_config = (self.pyproject_toml.get("tool") or {}).get(
                "buildsys-dateversion"
            ) or {}
            self._tool_config = self._tool_config.copy()
        return self._tool_config

    @property
    def marker(self) -> str:
        return (
            self.tool_config.get("version-marker")
            or "generated by buildsys-dateversion"
        )

    def _find_version_path(self) -> str:
        for dirpath, _, files in os.walk("."):
            for file in files:
                if not file.endswith(".py"):
                    continue

                path = os.path.join(dirpath, file)
                LOG.debug("Checking file %s", path)
                if VERSION_PATTERN.search(open(path).read()):
                    LOG.info("detected __version__ in %s", path)
                    return path

        # TODO: more helpful error string
        raise RuntimeError(
            "Could not locate a file in the source tree defining `__version__`"
        )

    @property
    def version_path(self) -> str:
        if "version-path" not in self.tool_config:
            self.tool_config["version-path"] = self._find_version_path()
        return self.tool_config["version-path"]

    @property
    def build_backend_name(self) -> str:
        return (
            self.tool_config.get("build-backend") or "setuptools.build_meta:__legacy__"
        )

    @property
    def build_backend(self):
        if "dateversion-build-backend-object" not in self.config_settings:
            backend_name = self.build_backend_name

            components = backend_name.split(":")
            module = importlib.import_module(components[0])
            backend = module

            if len(components) == 2:
                backend = getattr(module, components[1])

            LOG.debug("using build backend %s (%s)", backend_name, backend)

            self.config_settings["dateversion-build-backend-object"] = backend

        return self.config_settings["dateversion-build-backend-object"]

    @property
    def index_url(self) -> str:
        out = (
            # TODO: reuse pip or other standard env vars
            self.config_settings.get("dateversion-index-url")
            or "https://pypi.org/simple"
        )
        while out.endswith("/"):
            out = out[:-1]
        return out

    @property
    def distribution_name(self) -> str:
        return self.project_config["name"]

    @property
    def version(self) -> str:
        if "dateversion-distribution-version" not in self.config_settings:
            self.config_settings[
                "dateversion-distribution-version"
            ] = self._get_version()
        return self.config_settings["dateversion-distribution-version"]

    def _get_version(self) -> str:
        formats = [
            "%Y.%m",
            "%Y.%m.%d",
            "%Y.%m.%d.%H",
            "%Y.%m.%d.%H.%M",
        ]

        for format in formats:
            candidate = normalized(self.datetime.strftime(format))
            if not self.version_in_repository(candidate):
                return candidate
            LOG.info(
                "%s-%s is already released, trying next...",
                self.distribution_name,
                candidate,
            )

        # Final fallback
        return normalized(datetime.strftime("%Y.%m.%d.%H.%M.%S"))

    def version_in_repository(self, candidate_version: str) -> bool:
        repo_url = os.path.join(self.index_url, self.distribution_name)

        LOG.info("looking for %s in %s ...", candidate_version, repo_url)

        # TODO: might we need to support custom CA bundles here?
        try:
            with urllib.request.urlopen(repo_url) as response:
                LOG.debug("Response status: %s", response.status)

                body = response.read()
                LOG.debug("Response body: %s", body)

                # Not wanting any real parsing logic here, we'll do a very
                # simple match. There might be some ways this can go wrong.
                return f"{self.distribution_name}-{candidate_version}".encode() in body

        except urllib.error.HTTPError as error:
            if error.getcode() == 404:
                # If repository itself doesn't exist, we'll take that
                # as meaning the distribution has never been released
                # to this index.
                return False
            raise error

    @contextmanager
    def patched_version(self):
        old_version_content = open(self.version_path).read()
        new_version_content = old_version_content

        # Do not patch a version more than once.
        #
        # This is so, for example, if an sdist is prepared and then
        # reused to build a wheel, the version from the sdist 'sticks'
        # rather than being recalculated and potentially changing.
        if self.marker not in old_version_content:
            new_version_content = VERSION_PATTERN.sub(
                f"__version__ = {repr(self.version)}  # {self.marker}",
                old_version_content,
            )
        try:
            # TODO: find a better way which doesn't require patching
            # in place.
            if new_version_content != old_version_content:
                open(self.version_path, "w").write(new_version_content)
            yield
        finally:
            if new_version_content != old_version_content:
                # TODO: restore previous mtime?
                open(self.version_path, "w").write(old_version_content)

    def delegate_build_wheel(
        self, wheel_directory, config_settings, metadata_directory
    ):
        return self.build_backend.build_wheel(
            wheel_directory,
            config_settings=config_settings,
            metadata_directory=metadata_directory,
        )

    def delegate_build_sdist(self, sdist_directory, config_settings):
        return self.build_backend.build_sdist(
            sdist_directory=sdist_directory, config_settings=config_settings
        )

    def delegate_build_editable(
        self, wheel_directory, config_settings, metadata_directory
    ):
        return self.build_backend.build_editable(
            wheel_directory,
            config_settings=config_settings,
            metadata_directory=metadata_directory,
        )

    def get_requires_for_build_editable(self):
        return self.get_requires_for_build_wheel()

    def get_requires_for_build_sdist(self):
        out = []

        # Bootstrap problem on python < 3.11:
        #
        # There is no standard tomllib yet.
        # We're about to ask for it to be installed.
        #
        # However, that hasn't happened yet, therefore we can't read pyproject.toml,
        # therefore we can't know what the build backend name is & whether the
        # setuptools default logic should be applied.
        #
        # To resolve this we're just going to assume that you might want setuptools.
        # Don't like it? Upgrade to python 3.11.
        if sys.version_info < (3, 11) or self.build_backend_name.startswith(
            "setuptools."
        ):
            # apply the same default behavior as pip
            out.append("setuptools>=40.8.0")

        if sys.version_info < (3, 11):
            out.append("tomli>=2.0.1")

        LOG.info("sdist requires: %s", out)

        return out

    def get_requires_for_build_wheel(self):
        out = self.get_requires_for_build_sdist()
        # About the 3.11 check, see comment in get_requires_for_build_sdist
        if sys.version_info < (3, 11) or self.build_backend_name.startswith(
            "setuptools."
        ):
            # apply the same default behavior as pip
            out.append("wheel")
        return out


def build_wheel(wheel_directory, config_settings=None, metadata_directory=None):
    LOG.debug(
        "build_wheel %s %s %s", wheel_directory, config_settings, metadata_directory
    )

    helper = DateVersion(config_settings=config_settings)
    with helper.patched_version():
        return helper.delegate_build_wheel(
            wheel_directory,
            config_settings=config_settings,
            metadata_directory=metadata_directory,
        )


def build_sdist(sdist_directory, config_settings=None):
    LOG.debug("build_sdist %s %s", sdist_directory, config_settings)

    helper = DateVersion(config_settings=config_settings)
    with helper.patched_version():
        return helper.delegate_build_sdist(sdist_directory, config_settings)


# FIXME: I only want to define build_editable here if the delegate backend
# also defines that. But, how can I know which delegate is being used
# at import time?
#
# - Hooks aren't being invoked yet, so there's no `config_settings`
#
# - Although PEP 517 says hooks are invoked with the working directory
#   being equal to a project's root, this only speaks of the environment
#   at hook execution time and not at import time. So we can't guarantee
#   the working directory at this point, meaning we can't reliably locate
#   pyproject.toml to look up settings there.
#
# We'll just define a build_editable always, and expect an error if
# the delegate backend doesn't have it.
def build_editable(wheel_directory, config_settings=None, metadata_directory=None):
    LOG.debug(
        "build_editable %s %s %s", wheel_directory, config_settings, metadata_directory
    )
    helper = DateVersion(config_settings=config_settings)
    # Should there be version patching for editable mode or not??
    with helper.patched_version():
        return helper.delegate_build_editable(
            wheel_directory, config_settings, metadata_directory
        )


def get_requires_for_build_wheel(config_settings=None):
    LOG.debug("get_requires_for_build_wheel %s", config_settings)
    helper = DateVersion(config_settings=config_settings)
    return helper.get_requires_for_build_wheel()


def get_requires_for_build_sdist(config_settings=None):
    LOG.debug("get_requires_for_build_sdist %s", config_settings)

    helper = DateVersion(config_settings=config_settings)
    return helper.get_requires_for_build_sdist()


def get_requires_for_build_editable(config_settings=None):
    LOG.debug("get_requires_for_build_editable %s", config_settings)
    helper = DateVersion(config_settings=config_settings)
    return helper.get_requires_for_build_editable()


# TODO: is it necessary to implement prepare_metadata_for_build_wheel
# as well? Noting that PEP 517 says build_wheel must respect metadata
# earlier prepared by prepare_metadata_for_build_wheel (if any), so in
# that case, it'd be too late to patch the version.
