"""
Intelligent analyzer for CodeDocGen using NLP and AST analysis.

Combines natural language processing with abstract syntax tree analysis
to generate context-aware, intelligent function descriptions.
"""

import re
import ast
import nltk
from typing import List, Dict, Optional, Tuple
from nltk.tokenize import word_tokenize
from nltk.corpus import wordnet
from nltk import pos_tag
from .models import Function, Parameter, FunctionBody, FunctionException
from .config import Config
from .ast_analyzer import ASTAnalyzer
from .ai_analyzer import AIAnalyzer
import logging


class IntelligentAnalyzer:
    """Intelligent analyzer that uses AI, NLTK, and regex-based analysis."""
    
    def __init__(self, config: Config):
        self.config = config
        self.logger = logging.getLogger(__name__)
        
        # Initialize AI analyzer if enabled
        self.ai_analyzer = AIAnalyzer(config) if config.get_ai_config().get('enabled', False) else None
        
        # Initialize AST analyzer
        self.ast_analyzer = ASTAnalyzer()
        
        # Ensure NLTK resources are available
        self._ensure_nltk_resources()
        
        # Patterns for function name analysis
        self.camel_case_pattern = re.compile(r'([A-Z][a-z0-9]+)')
        self.snake_case_pattern = re.compile(r'_([a-z0-9])')
    
    def _ensure_nltk_resources(self) -> None:
        """Download required NLTK resources for intelligent analysis."""
        resources = ['punkt', 'averaged_perceptron_tagger', 'wordnet']
        for resource in resources:
            try:
                nltk.data.find(resource)
            except LookupError:
                nltk.download(resource, quiet=True)
    
    def analyze_function(self, function: Function, language: str = "python") -> None:
        """
        Analyze a function and generate documentation.
        
        Args:
            function: Function to analyze
            language: Programming language for AI analysis
        """
        # Check if function already has documentation using enhanced detection
        if self._has_existing_documentation(function, language) or self.has_existing_documentation(function.source_code, language):
            self.logger.debug(f"Function {function.name} already has documentation, skipping")
            return  # Skip if already documented
        
        self.logger.debug(f"=== Analyzing function: {function.name} ===")
        
        # Try AI analysis first if enabled
        if self.ai_analyzer:
            ai_comment = self.ai_analyzer.analyze_function(function, language)
            
            if ai_comment:
                self.logger.debug(f"AI generated comment for {function.name}: {ai_comment}")
                function.brief_description = ai_comment
                return
        
        self.logger.debug(f"AI analysis failed for {function.name}, falling back to NLTK")
        
        # Fallback to NLTK analysis
        try:
            # Generate description using NLTK
            description = self._generate_intelligent_description(function)
            if description:
                # Store the raw description - let the documentation generator format it
                function.brief_description = description
                self.logger.debug(f"NLTK generated description for {function.name}: {description}")
            
            # Generate parameter descriptions
            for param in function.parameters:
                param_desc = self._generate_parameter_description(param)
                if param_desc:
                    param.description = param_desc
                    self.logger.debug(f"NLTK generated param description for {param.name}: {param_desc}")
                    
        except Exception as e:
            self.logger.warning(f"NLTK analysis failed for function {function.name}: {e}")
            
            # Final fallback to basic description
            try:
                basic_comment = f"Function {function.name}"
                function.brief_description = basic_comment
                self.logger.debug(f"Basic comment for {function.name}: {basic_comment}")
            except Exception as e2:
                self.logger.warning(f"Basic analysis also failed for function {function.name}: {e2}")
    
    def _generate_intelligent_description(self, function: Function) -> str:
        """
        Generate context-aware function description using NLP and AST analysis.
        
        Args:
            function: Function to analyze
            
        Returns:
            Intelligent description string
        """
        # 1. Semantic analysis of function name
        verb, obj = self._parse_function_name(function.name)
        
        # 2. AST analysis for key characteristics
        characteristics = self._analyze_function_ast(function)
        
        # 3. Detect special patterns
        pattern = self._detect_special_pattern(function)
        
        # 4. Construct natural language description
        description = self._construct_description(verb, obj, characteristics, pattern)
        
        return description.capitalize()
    
    def _has_existing_documentation(self, function: Function, language: str) -> bool:
        """
        Check if a function already has documentation.
        
        Args:
            function: Function to check
            language: Programming language
            
        Returns:
            True if function already has documentation, False otherwise
        """
        if not function.source_code:
            return False
        
        # Get the lines before the function definition
        lines = function.source_code.split('\n')
        function_line_idx = -1
        
        # Find the function definition line
        for i, line in enumerate(lines):
            if function.name in line and ('def ' in line or 'function ' in line or 
                                        language in ['c++', 'java'] and ('(' in line and ')' in line)):
                function_line_idx = i
                break
        
        if function_line_idx == -1:
            return False
        
        # Check for documentation in the lines before the function
        # Only check the immediate lines before the function (not file-level docstrings)
        start_check = max(0, function_line_idx - 3)  # Only check 3 lines before
        
        for i in range(start_check, function_line_idx):
            line = lines[i].strip()
            
            if language == 'python':
                if line.startswith('"""') or line.startswith("'''") or line.startswith('#'):
                    # Check if this is a file-level docstring (at the very beginning)
                    if i < 5:  # If it's in the first few lines, it might be file-level
                        # Check if there are any other functions before this one
                        has_other_functions = False
                        for j in range(i):
                            if 'def ' in lines[j]:
                                has_other_functions = True
                                break
                        if not has_other_functions:
                            # This is likely a file-level docstring, not function documentation
                            continue
                    return True
            else:
                if (line.startswith('/**') or line.startswith('/*') or 
                    line.startswith('///') or line.startswith('*')):
                    return True
                # Don't treat simple // comments as documentation unless they're documentation-specific
                if line.startswith('//') and any(keyword in line.lower() for keyword in ['@brief', '@param', '@return', 'brief', 'param', 'return']):
                    return True
        
        return False
    
    def has_existing_documentation(self, source: str, language: str) -> bool:
        """
        Enhanced method to detect existing documentation using comprehensive patterns.
        
        Args:
            source: Source code string
            language: Programming language
            
        Returns:
            True if documentation exists, False otherwise
        """
        patterns = []
        
        if language == 'python':
            patterns = [
                r'^\s*""".*?"""',     # Python docstrings
                r'^\s*\'\'\'.*?\'\'\'', # Python docstrings (single quotes)
                r'^\s*#\s*@',         # Python decorator docs
                r'^\s*##\s+',         # Common comment format
            ]
        else:  # C++/Java
            patterns = [
                r'^\s*/\*\*.*?\*/',   # C++/Java doc comments
                r'^\s*///',           # C++ single-line docs
                r'^\s*\* @',          # JSDoc-style
                r'^\s*<!--.*?-->',    # XML comments
                r'^\s*##\s+',         # Common comment format
            ]
        
        return any(re.search(p, source, re.DOTALL | re.MULTILINE) for p in patterns)
    
    def _parse_function_name(self, name: str) -> Tuple[str, str]:
        """
        Extract semantic meaning from function name using NLP.
        
        Args:
            name: Function name to parse
            
        Returns:
            Tuple of (verb, object)
        """
        # Split snake_case or camelCase
        words = re.split(r'_|(?=[A-Z])', name)
        if not words:
            return "processes", "data"
        
        try:
            # Use NLTK for POS tagging
            tagged = pos_tag(words)
            
            # Find best verb candidate with more specific mapping
            verbs = [word for word, pos in tagged if pos.startswith('VB')]
            if verbs:
                verb = verbs[0]
            else:
                # Infer verb from function name with specific actions
                name_lower = name.lower()
                if any(word in name_lower for word in ['add', 'sum']):
                    verb = "adds"
                elif any(word in name_lower for word in ['calculate', 'compute', 'factorial']):
                    verb = "calculates"
                elif any(word in name_lower for word in ['get', 'fetch', 'retrieve']):
                    verb = "retrieves"
                elif any(word in name_lower for word in ['save', 'write', 'store']):
                    verb = "saves"
                elif any(word in name_lower for word in ['filter', 'select']):
                    verb = "filters"
                elif any(word in name_lower for word in ['extract', 'parse']):
                    verb = "extracts"
                elif any(word in name_lower for word in ['format', 'transform', 'convert']):
                    verb = "formats"
                elif any(word in name_lower for word in ['count']):
                    verb = "counts"
                elif any(word in name_lower for word in ['process', 'handle']):
                    verb = "processes"
                elif any(word in name_lower for word in ['divide', 'safe_divide']):
                    verb = "divides"
                elif any(word in name_lower for word in ['validate', 'check']):
                    verb = "validates"
                elif any(word in name_lower for word in ['read', 'load']):
                    verb = "reads"
                else:
                    verb = "processes"
            
            # Find best object candidate with more specific mapping
            nouns = [word for word, pos in tagged if pos.startswith('NN')]
            if nouns:
                obj = " ".join(nouns)
            else:
                # Infer object from function name
                obj = " ".join(words[1:]) if len(words) > 1 else words[0]
            
            # Make object more specific based on context
            obj_lower = obj.lower()
            if any(word in obj_lower for word in ['numbers', 'num']):
                obj = "numbers"
            elif any(word in obj_lower for word in ['data', 'api']):
                obj = "data"
            elif any(word in obj_lower for word in ['file', 'filename']):
                obj = "file"
            elif any(word in obj_lower for word in ['email', 'emails']):
                obj = "emails"
            elif any(word in obj_lower for word in ['user', 'name']):
                obj = "user name"
            elif any(word in obj_lower for word in ['word', 'frequency']):
                obj = "word frequency"
            elif any(word in obj_lower for word in ['deviation', 'std']):
                obj = "standard deviation"
            
            return verb, obj
            
        except (LookupError, OSError, ImportError):
            # Fallback to simple analysis with better defaults
            if len(words) >= 2:
                return words[0], " ".join(words[1:])
            else:
                return "processes", words[0] if words else "data"
    
    def _analyze_function_ast(self, function: Function) -> List[str]:
        """
        Analyze function AST for key characteristics.
        
        Args:
            function: Function to analyze
            
        Returns:
            List of behavior characteristics
        """
        if not function.ast_node:
            return []
        
        return self.ast_analyzer.analyze(function.ast_node, function.name)
    
    def _detect_special_pattern(self, function: Function) -> Optional[str]:
        """
        Detect special programming patterns.
        
        Args:
            function: Function to analyze
            
        Returns:
            Pattern description or None
        """
        # Recursion detection - use AST analysis instead of naive string matching
        if function.body and function.body.has_recursion:
            return "recursively"
        
        # Decorator pattern detection
        if function.source_code and "@" in function.source_code.split("\n")[0]:
            return "decorated"
        
        # API call detection
        if "requests.get" in function.source_code or "requests.post" in function.source_code:
            return "API call"
        
        # Main function detection
        if function.name.lower() == 'main':
            return "entry point"
        
        return None
    
    def _construct_description(self, verb: str, obj: str, characteristics: List[str], pattern: Optional[str]) -> str:
        """
        Construct natural language description.
        
        Args:
            verb: Action verb
            obj: Object being acted upon
            characteristics: List of behavior characteristics
            pattern: Special pattern if any
            
        Returns:
            Natural language description
        """
        # Base description
        if pattern:
            description = f"{pattern} {verb} the {obj}"
        else:
            description = f"{verb} the {obj}"
        
        # Add characteristics
        if characteristics:
            if len(characteristics) == 1:
                description += f" by {characteristics[0]}"
            else:
                description += f" by {', '.join(characteristics[:-1])} and {characteristics[-1]}"
        
        return description
    
    def _generate_detailed_description(self, function: Function) -> str:
        """
        Generate a detailed description of the function.
        
        Args:
            function: Function to analyze
            
        Returns:
            Detailed description string
        """
        parts = []
        
        # Start with brief description
        if function.brief_description:
            parts.append(function.brief_description)
        
        # Add parameter information
        if function.has_parameters():
            param_desc = self._describe_parameters(function.parameters)
            if param_desc:
                parts.append(param_desc)
        
        # Add return information
        if function.return_type.lower() != 'void':
            return_desc = self._describe_return_type(function)
            if return_desc:
                parts.append(return_desc)
        
        return " ".join(parts)
    
    def _extract_noun_from_name(self, name: str) -> str:
        """
        Extract a noun from a function name.
        
        Args:
            name: Function name
            
        Returns:
            Extracted noun
        """
        if not name:
            return "value"
        
        # Convert camelCase to words
        words = self.camel_case_pattern.findall(name)
        if words:
            # Use the first word as the noun
            return words[0].lower()
        
        # Convert snake_case to words
        words = name.split('_')
        if words:
            # For snake_case, take the second part if it exists
            if len(words) > 1:
                return words[1].lower()
            else:
                return words[0].lower()
        
        return name.lower()
    
    def _name_to_sentence(self, name: str) -> str:
        """
        Convert a function name to a readable sentence using NLTK.
        
        Args:
            name: Function name
            
        Returns:
            Sentence describing the function
        """
        # Special case for main function
        if name.lower() == 'main':
            return "Entry point of the program."
        
        # Convert camelCase to words
        words = self.camel_case_pattern.findall(name)
        if not words:
            # Convert snake_case to words
            words = name.split('_')
        
        if not words:
            return f"Performs {name} operation."
        
        # Use NLTK for intelligent analysis
        try:
            # Tokenize the words
            tokens = nltk.word_tokenize(' '.join(words))
            pos_tags = nltk.pos_tag(tokens)
            
            # Analyze the structure
            verbs = [word for word, tag in pos_tags if tag.startswith('VB')]
            nouns = [word for word, tag in pos_tags if tag.startswith('NN')]
            adjectives = [word for word, tag in pos_tags if tag.startswith('JJ')]
            
            # Handle different patterns
            if len(tokens) == 1:
                word = tokens[0].lower()
                tag = pos_tags[0][1]
                
                if tag.startswith('VB'):
                    # Single verb - describe the action
                    return f"Executes {word} operation."
                elif tag.startswith('NN'):
                    # Single noun - describe what it processes
                    if word in ['dfs', 'bfs', 'dijkstra', 'kruskal', 'prim']:
                        return f"Implements {word.upper()} algorithm."
                    elif word in ['sort', 'filter', 'map', 'reduce', 'search']:
                        return f"Applies {word} operation to data."
                    elif word in ['init', 'setup', 'start']:
                        return f"Initializes the system or component."
                    elif word in ['cleanup', 'teardown', 'stop']:
                        return f"Cleans up resources and terminates."
                    else:
                        return f"Processes {word} data or operations."
                else:
                    return f"Handles {word} operations."
            
            elif len(tokens) == 2:
                # Two-word patterns
                word1, tag1 = pos_tags[0]
                word2, tag2 = pos_tags[1]
                
                if tag1.startswith('VB') and tag2.startswith('NN'):
                    # Verb + Noun: "getUser", "setValue"
                    return f"{word1.capitalize()} the {word2}."
                elif tag1.startswith('JJ') and tag2.startswith('NN'):
                    # Adjective + Noun: "quickSort", "binarySearch"
                    return f"Implements {word1} {word2} algorithm."
                elif tag1.startswith('NN') and tag2.startswith('NN'):
                    # Noun + Noun: "userManager", "dataProcessor"
                    return f"Manages {word1} {word2} operations."
            
            elif len(tokens) >= 3:
                # Multi-word patterns - use NLTK to understand structure
                if verbs and nouns:
                    verb = verbs[0].lower()
                    noun = nouns[0].lower()
                    return f"{verb.capitalize()} the {noun}."
                elif verbs:
                    verb = verbs[0].lower()
                    return f"{verb.capitalize()} the specified data or values."
                elif nouns:
                    noun = nouns[0].lower()
                    return f"Processes {noun} data or operations."
                else:
                    # Fallback for complex names
                    return f"Handles {name.lower()} operations."
            
            else:
                return f"Processes {name.lower()} operations."
                
        except (LookupError, OSError, ImportError) as e:
            # Fallback to intelligent pattern matching
            word = name.lower()
            
            # Common algorithm patterns
            if word in ['dfs', 'bfs', 'dijkstra', 'kruskal', 'prim', 'bellmanford']:
                return f"Implements {word.upper()} algorithm."
            elif word in ['quicksort', 'mergesort', 'heapsort', 'bubblesort']:
                return f"Implements {word} sorting algorithm."
            elif word in ['binarysearch', 'linearsearch']:
                return f"Implements {word} search algorithm."
            elif word in ['main', 'init', 'setup']:
                return "Initializes or starts the program."
            elif word in ['cleanup', 'teardown', 'destroy']:
                return "Cleans up resources and terminates."
            else:
                return f"Processes {word} operations."
    
    def _fill_template(self, template: str, function: Function) -> str:
        """
        Fill a template with function information.
        
        Args:
            template: Template string with placeholders
            function: Function to use for filling
            
        Returns:
            Filled template string
        """
        # Extract noun from function name
        noun = self._extract_noun_from_name(function.name)
        
        # Get parameter names
        params = function.get_parameter_names()
        params_str = ', '.join(params) if params else "input"
        
        # Replace placeholders
        result = template.replace("{noun}", noun)
        result = result.replace("{params}", params_str)
        result = result.replace("{name}", function.name)
        
        return result
    
    def _describe_parameters(self, parameters: List[Parameter]) -> str:
        """
        Generate a description of function parameters.
        
        Args:
            parameters: List of parameters
            
        Returns:
            Parameter description string
        """
        if not parameters:
            return ""
        
        param_names = [p.name for p in parameters]
        if len(param_names) == 1:
            return f"Takes {param_names[0]} as input."
        else:
            return f"Takes {', '.join(param_names[:-1])} and {param_names[-1]} as input."
    
    def _describe_return_type(self, function: Function) -> str:
        """
        Generate intelligent return type description using NLP.
        
        Args:
            function: Function to analyze
            
        Returns:
            Return type description string
        """
        return_type = function.return_type.lower()
        func_name = function.name.lower()
        
        try:
            # Use NLTK to understand function purpose and return type
            tokens = word_tokenize(func_name)
            pos_tags = pos_tag(tokens)
            
            verbs = [word for word, tag in pos_tags if tag.startswith('VB')]
            nouns = [word for word, tag in pos_tags if tag.startswith('NN')]
            
            # Understand return type based on function purpose
            if return_type == 'bool':
                if verbs and any(v in ['validate', 'check', 'verify', 'is', 'has'] for v in verbs):
                    return "Returns true if the condition is met, false otherwise."
                else:
                    return "Returns a boolean result of the operation."
            
            elif return_type in ['int', 'integer']:
                if verbs and any(v in ['calculate', 'compute', 'count', 'sum'] for v in verbs):
                    return "Returns the calculated integer result."
                elif nouns and any(n in ['index', 'position', 'size', 'length'] for n in nouns):
                    return "Returns the position, index, or count value."
                elif 'factorial' in func_name:
                    return "Returns the factorial value."
                elif 'add' in func_name:
                    return "Returns the sum of the input numbers."
                else:
                    return "Returns an integer result."
            
            elif return_type in ['float', 'double']:
                if verbs and any(v in ['calculate', 'compute', 'average', 'mean'] for v in verbs):
                    return "Returns the calculated floating-point result."
                elif 'deviation' in func_name:
                    return "Returns the standard deviation value."
                else:
                    return "Returns a floating-point result."
            
            elif return_type in ['string', 'str']:
                if verbs and any(v in ['format', 'convert', 'toString'] for v in verbs):
                    return "Returns the formatted or converted string."
                elif verbs and any(v in ['get', 'retrieve', 'fetch'] for v in verbs):
                    return "Returns the retrieved string data."
                elif 'api' in func_name:
                    return "Returns the API response data."
                else:
                    return "Returns a string result."
            
            elif return_type in ['list', 'array']:
                if verbs and any(v in ['get', 'retrieve', 'fetch'] for v in verbs):
                    return "Returns the retrieved list of items."
                elif verbs and any(v in ['process', 'filter', 'transform'] for v in verbs):
                    return "Returns the processed list of results."
                elif 'filter' in func_name and 'even' in func_name:
                    return "Returns a list containing only the even numbers."
                elif 'emails' in func_name:
                    return "Returns a list of extracted email addresses."
                else:
                    return "Returns a list of processed values."
            
            elif return_type in ['dict', 'map']:
                if verbs and any(v in ['get', 'retrieve'] for v in verbs):
                    return "Returns the retrieved data as key-value pairs."
                elif 'frequency' in func_name:
                    return "Returns a dictionary mapping words to their frequency counts."
                else:
                    return "Returns a dictionary containing structured data."
            
            elif return_type == 'void':
                if verbs and any(v in ['traverse', 'visit', 'explore'] for v in verbs):
                    return "Returns nothing (performs traversal operations)."
                elif verbs and any(v in ['set', 'update', 'modify'] for v in verbs):
                    return "Returns nothing (modifies state or data)."
                else:
                    return "Returns nothing (performs side effects only)."
            
            else:
                return f"Returns a {return_type} value from the operation."
                
        except (LookupError, OSError, ImportError):
            # Fallback to basic type descriptions
            if return_type == 'bool':
                return "Returns true or false based on the operation result."
            elif return_type in ['int', 'integer']:
                return "Returns an integer value from the calculation."
            elif return_type in ['float', 'double']:
                return "Returns a floating-point value from the calculation."
            elif return_type in ['string', 'str']:
                return "Returns a string value or formatted text."
            elif return_type in ['list', 'array']:
                return "Returns a list or array of processed values."
            elif return_type in ['dict', 'map']:
                return "Returns a dictionary containing key-value pairs."
            elif return_type == 'void':
                return "Returns nothing (performs side effects only)."
            else:
                return f"Returns a {return_type} value from the operation."
    
    def _analyze_function_body(self, body: FunctionBody) -> None:
        """
        Analyze function body for patterns and behaviors.
        
        Args:
            body: Function body to analyze
        """
        # This method would analyze the actual function body
        # For now, we'll set some default behaviors based on common patterns
        # In a real implementation, this would parse the AST and detect patterns
        
        # Example analysis (this would be done by the parsers)
        # body.has_loops = self._detect_loops(ast_node)
        # body.has_conditionals = self._detect_conditionals(ast_node)
        # body.has_exceptions = self._detect_exceptions(ast_node)
        # etc.
        pass
    
    def analyze_parameter(self, parameter: Parameter) -> None:
        """
        Analyze a parameter and generate description.
        
        Args:
            parameter: Parameter to analyze
        """
        if not parameter.description:
            parameter.description = self._generate_parameter_description(parameter)
    
    def _generate_parameter_description(self, parameter: Parameter) -> str:
        """
        Generate intelligent parameter description using NLP.
        
        Args:
            parameter: Parameter to describe
            
        Returns:
            Parameter description string
        """
        param_name = parameter.name.lower()
        param_type = parameter.type.lower()
        
        try:
            # Use NLTK to understand parameter meaning
            tokens = word_tokenize(param_name)
            pos_tags = pos_tag(tokens)
            
            # Analyze parameter name structure
            if len(tokens) == 1:
                word = tokens[0]
                tag = pos_tags[0][1]
                
                if tag.startswith('NN'):
                    # Noun - understand what it represents
                    if word in ['node', 'vertex']:
                        return f"The {word} to process in graph/tree structure."
                    elif word in ['data', 'input', 'value']:
                        return f"The {word} to be processed."
                    elif word in ['result', 'output']:
                        return f"The {word} to store results."
                    elif word in ['list', 'array', 'collection']:
                        return f"The {word} of items to process."
                    elif word in ['count', 'number', 'size']:
                        return f"The {word} value for calculations."
                    elif word in ['id', 'key', 'index']:
                        return f"The {word} for identification."
                    elif word in ['url', 'link', 'address']:
                        return f"The {word} to fetch data from."
                    elif word in ['filename', 'file', 'path']:
                        return f"The {word} to read/write."
                    elif word in ['content', 'text', 'data']:
                        return f"The {word} to process."
                    elif word in ['a', 'b', 'c']:
                        # Handle single letter parameters intelligently
                        if word == 'a':
                            return f"The first number of type {param_type}."
                        elif word == 'b':
                            return f"The second number of type {param_type}."
                        elif word == 'c':
                            return f"The third number of type {param_type}."
                    elif word in ['n', 'num', 'number']:
                        return f"The number to calculate for."
                    elif word in ['x', 'y', 'z']:
                        return f"The {word}-coordinate value."
                    elif word in ['i', 'j', 'k']:
                        return f"The {word} index value."
                    else:
                        # For other words, use a more intelligent approach
                        # Only use wordnet for longer, meaningful words
                        if len(word) > 2:
                            try:
                                synsets = wordnet.synsets(word)
                                if synsets and not any(synset.name().startswith('b.') or synset.name().startswith('n.') for synset in synsets):
                                    meaning = synsets[0].lemmas()[0].name().replace('_', ' ')
                                    return f"The {meaning} of type {param_type}."
                            except:
                                pass
                        return f"The {word} parameter."
                
                elif tag.startswith('JJ'):
                    # Adjective - describe the type
                    return f"The {word} parameter."
                
                else:
                    # Other parts of speech
                    return f"The {word} parameter."
            
            elif len(tokens) >= 2:
                # Multi-word parameter names
                if any(tag.startswith('NN') for _, tag in pos_tags):
                    # Contains nouns - understand the object
                    nouns = [word for word, tag in pos_tags if tag.startswith('NN')]
                    if 'id' in nouns or 'identifier' in nouns:
                        return f"The identifier for the {' '.join(nouns)}."
                    elif 'list' in nouns or 'array' in nouns:
                        return f"The {' '.join(nouns)} to process."
                    else:
                        return f"The {' '.join(nouns)} parameter."
                else:
                    return f"The {param_name} parameter."
            
            else:
                return f"The {param_name} parameter."
                
        except (LookupError, OSError, ImportError):
            # Fallback to type-based description
            type_desc = self._get_type_description(parameter.type)
            return f"The {parameter.name} {type_desc}."
    
    def _get_type_description(self, type_name: str) -> str:
        """
        Get a human-readable description of a type.
        
        Args:
            type_name: Type name
            
        Returns:
            Type description string
        """
        type_mapping = {
            'int': 'integer value',
            'float': 'floating-point number',
            'double': 'double-precision floating-point number',
            'char': 'character',
            'string': 'string value',
            'str': 'string value',
            'bool': 'boolean value',
            'boolean': 'boolean value',
            'void': 'void value',
            'list': 'list of values',
            'array': 'array of values',
            'dict': 'dictionary of key-value pairs',
            'map': 'map of key-value pairs',
            'tuple': 'tuple of values',
            'set': 'set of unique values',
            'bytes': 'bytes data',
            'object': 'object instance',
            'None': 'None value'
        }
        
        return type_mapping.get(type_name.lower(), f"value of type {type_name}")
    
    def analyze_exception(self, exception: Exception) -> None:
        """
        Analyze an exception and generate description.
        
        Args:
            exception: Exception to analyze
        """
        if not exception.description:
            exception.description = self._generate_exception_description(exception)
    
    def _generate_exception_description(self, exception: Exception) -> str:
        """
        Generate a description for an exception.
        
        Args:
            exception: Exception to describe
            
        Returns:
            Exception description string
        """
        return f"Thrown when {exception.name.lower()} occurs." 