import { Construct } from 'constructs';
import { IAction, ActionBindOptions } from './action';
import { Event } from './event';
import { Expression } from './expression';
import { CfnDetectorModel } from 'aws-cdk-lib/aws-iotevents';
/**
 * Properties for options of state transition.
 */
export interface TransitionOptions {
    /**
     * The name of the event.
     *
     * @default string combining the names of the States as `${originStateName}_to_${targetStateName}`
     */
    readonly eventName?: string;
    /**
     * The condition that is used to determine to cause the state transition and the actions.
     * When this was evaluated to `true`, the state transition and the actions are triggered.
     */
    readonly when: Expression;
    /**
     * The actions to be performed with the transition.
     *
     * @default - no actions will be performed
     */
    readonly executing?: IAction[];
}
/**
 * Properties for defining a state of a detector.
 */
export interface StateProps {
    /**
     * The name of the state.
     */
    readonly stateName: string;
    /**
     * Specifies the events on enter. The conditions of the events will be evaluated when entering this state.
     * If the condition of the event evaluates to `true`, the actions of the event will be executed.
     *
     * @default - no events will trigger on entering this state
     */
    readonly onEnter?: Event[];
    /**
     * Specifies the events on input. The conditions of the events will be evaluated when any input is received.
     * If the condition of the event evaluates to `true`, the actions of the event will be executed.
     *
     * @default - no events will trigger on input in this state
     */
    readonly onInput?: Event[];
    /**
     * Specifies the events on exit. The conditions of the events are evaluated when an exiting this state.
     * If the condition evaluates to `true`, the actions of the event will be executed.
     *
     * @default - no events will trigger on exiting this state
     */
    readonly onExit?: Event[];
}
/**
 * Defines a state of a detector.
 */
export declare class State {
    private readonly props;
    /**
     * The name of the state.
     */
    readonly stateName: string;
    private readonly transitionEvents;
    constructor(props: StateProps);
    /**
     * Add a transition event to the state.
     * The transition event will be triggered if condition is evaluated to `true`.
     *
     * @param targetState the state that will be transit to when the event triggered
     * @param options transition options including the condition that causes the state transition
     */
    transitionTo(targetState: State, options: TransitionOptions): void;
    /**
     * Collect states in dependency graph that constructed by state transitions,
     * and return the JSONs of the states.
     * This function is called recursively and collect the states.
     *
     * @internal
     */
    _collectStateJsons(scope: Construct, actionBindOptions: ActionBindOptions, collectedStates: Set<State>): CfnDetectorModel.StateProperty[];
    /**
     * Returns true if this state has at least one condition via events.
     *
     * @internal
     */
    _onEnterEventsHaveAtLeastOneCondition(): boolean;
    private toStateJson;
}
