"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const events = require("@aws-cdk/aws-events");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const destinations = require("../lib");
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
});
const lambdaProps = {
    code: new lambda.InlineCode('foo'),
    handler: 'index.handler',
    runtime: lambda.Runtime.NODEJS_10_X,
};
test('event bus as destination', () => {
    // GIVEN
    const eventBus = new events.EventBus(stack, 'EventBus');
    // WHEN
    new lambda.Function(stack, 'Function', {
        ...lambdaProps,
        onSuccess: new destinations.EventBridgeDestination(eventBus)
    });
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    'Fn::GetAtt': [
                        'EventBus7B8748AA',
                        'Arn'
                    ]
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'events:PutEvents',
                    Effect: 'Allow',
                    Resource: '*'
                }
            ],
            Version: '2012-10-17'
        }
    });
});
test('event bus as destination defaults to default event bus', () => {
    // WHEN
    new lambda.Function(stack, 'Function', {
        ...lambdaProps,
        onSuccess: new destinations.EventBridgeDestination()
    });
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition'
                            },
                            ':events:',
                            {
                                Ref: 'AWS::Region'
                            },
                            ':',
                            {
                                Ref: 'AWS::AccountId'
                            },
                            ':event-bus/default'
                        ]
                    ]
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'events:PutEvents',
                    Effect: 'Allow',
                    Resource: '*'
                }
            ],
            Version: '2012-10-17'
        }
    });
});
test('lambda as destination', () => {
    // GIVEN
    const successLambda = new lambda.Function(stack, 'SuccessFunction', lambdaProps);
    // WHEN
    new lambda.Function(stack, 'Function', {
        ...lambdaProps,
        onSuccess: new destinations.LambdaDestination(successLambda)
    });
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    'Fn::GetAtt': [
                        'SuccessFunction93C61D39',
                        'Arn'
                    ]
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'lambda:InvokeFunction',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'SuccessFunction93C61D39',
                            'Arn'
                        ]
                    }
                }
            ],
            Version: '2012-10-17'
        }
    });
});
test('lambda payload as destination', () => {
    // GIVEN
    const successLambda = new lambda.Function(stack, 'SuccessFunction', lambdaProps);
    const failureLambda = new lambda.Function(stack, 'FailureFunction', lambdaProps);
    // WHEN
    new lambda.Function(stack, 'Function', {
        ...lambdaProps,
        onSuccess: new destinations.LambdaDestination(successLambda, { responseOnly: true }),
        onFailure: new destinations.LambdaDestination(failureLambda, { responseOnly: true }),
    });
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition'
                            },
                            ':events:',
                            {
                                Ref: 'AWS::Region'
                            },
                            ':',
                            {
                                Ref: 'AWS::AccountId'
                            },
                            ':event-bus/default'
                        ]
                    ]
                }
            },
            OnFailure: {
                Destination: {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition'
                            },
                            ':events:',
                            {
                                Ref: 'AWS::Region'
                            },
                            ':',
                            {
                                Ref: 'AWS::AccountId'
                            },
                            ':event-bus/default'
                        ]
                    ]
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'events:PutEvents',
                    Effect: 'Allow',
                    Resource: '*'
                }
            ],
            Version: '2012-10-17'
        }
    });
    expect(stack).toHaveResource('AWS::Events::Rule', {
        EventPattern: {
            'detail-type': [
                'Lambda Function Invocation Result - Success'
            ],
            'resources': [
                {
                    'Fn::Join': [
                        '',
                        [
                            {
                                'Fn::GetAtt': [
                                    'Function76856677',
                                    'Arn'
                                ]
                            },
                            ':$LATEST'
                        ]
                    ]
                }
            ],
            'source': [
                'lambda'
            ]
        },
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': [
                        'SuccessFunction93C61D39',
                        'Arn'
                    ]
                },
                Id: 'Target0',
                InputPath: '$.detail.responsePayload'
            }
        ]
    });
    expect(stack).toHaveResource('AWS::Events::Rule', {
        EventPattern: {
            'detail-type': [
                'Lambda Function Invocation Result - Failure'
            ],
            'resources': [
                {
                    'Fn::Join': [
                        '',
                        [
                            {
                                'Fn::GetAtt': [
                                    'Function76856677',
                                    'Arn'
                                ]
                            },
                            ':$LATEST'
                        ]
                    ]
                }
            ],
            'source': [
                'lambda'
            ]
        },
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': [
                        'FailureFunctionE917A574',
                        'Arn'
                    ]
                },
                Id: 'Target0',
                InputPath: '$.detail.responsePayload'
            }
        ]
    });
});
test('sns as destination', () => {
    // GIVEN
    const topic = new sns.Topic(stack, 'Topic');
    // WHEN
    new lambda.Function(stack, 'Function', {
        ...lambdaProps,
        onSuccess: new destinations.SnsDestination(topic)
    });
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    Ref: 'TopicBFC7AF6E'
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'sns:Publish',
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'TopicBFC7AF6E'
                    }
                }
            ],
            Version: '2012-10-17'
        }
    });
});
test('sqs as destination', () => {
    // GIVEN
    const queue = new sqs.Queue(stack, 'Queue');
    // WHEN
    new lambda.Function(stack, 'Function', {
        ...lambdaProps,
        onSuccess: new destinations.SqsDestination(queue)
    });
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::EventInvokeConfig', {
        DestinationConfig: {
            OnSuccess: {
                Destination: {
                    'Fn::GetAtt': [
                        'Queue4A7E3555',
                        'Arn'
                    ]
                }
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'sqs:SendMessage',
                        'sqs:GetQueueAttributes',
                        'sqs:GetQueueUrl'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Queue4A7E3555',
                            'Arn'
                        ]
                    }
                }
            ],
            Version: '2012-10-17'
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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