"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PostgresqlDatabase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const triggers_1 = require("aws-cdk-lib/triggers");
const base_database_1 = require("./base-database");
const HANDLER_PATH = path.join(__dirname, "../../assets/rds/triggers/pgsql");
/**
 * A PostgreSQL database.
 */
class PostgresqlDatabase extends base_database_1.BaseDatabase {
    /**
     * Creates a new PostgresqlDatabase.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param props - The configuration properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.ownerSecrets = [];
        this.readerSecrets = [];
        this.unprivilegedSecrets = [];
        const { adminSecret, vpc, ownerSecret, schemaName, encoding = "UTF8", } = props;
        const environment = {
            AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
            ADMIN_SECRET_ARN: adminSecret.secretArn,
            OWNER_SECRET_ARN: ownerSecret.secretArn,
            OWNER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.ownerSecrets.map((s) => s.secretArn)),
            }),
            READER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.readerSecrets.map((s) => s.secretArn)),
            }),
            UNPRIVILEGED_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.unprivilegedSecrets.map((s) => s.secretArn)),
            }),
            DB_NAME: this.databaseName,
            SCHEMA_NAME: schemaName || "",
            DB_ENCODING: encoding,
        };
        if (props.locale) {
            environment.DB_LOCALE = props.locale;
        }
        if (props.certificateAuthoritiesUrl) {
            environment.CA_CERTS_URL = props.certificateAuthoritiesUrl;
        }
        this.lambdaFunction = new aws_lambda_1.Function(this, "Function", {
            runtime: aws_lambda_1.Runtime.NODEJS_20_X,
            code: aws_lambda_1.Code.fromAsset(HANDLER_PATH),
            handler: "index.handler",
            description: "Creates a schema and possibly some secondary users",
            vpc,
            vpcSubnets: this.subnetSelection,
            securityGroups: [this.securityGroup],
            environment,
            timeout: aws_cdk_lib_1.Duration.minutes(2),
        });
        adminSecret.grantRead(this.lambdaFunction);
        this.trigger = new triggers_1.Trigger(this, "Trigger", {
            handler: this.lambdaFunction,
        });
        this.trigger.executeAfter(this.lambdaFunction);
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseCluster.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forCluster(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new PostgresqlDatabase(scope, id, {
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            vpc: cluster.vpc,
            ...props,
        });
        database.trigger.executeAfter(cluster);
        return database;
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forClusterFromSnapshot(scope, id, cluster, options) {
        // The DatabaseClusterFromSnapshot type is equivalent to DatabaseCluster.
        return PostgresqlDatabase.forCluster(scope, id, cluster, options);
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseCluster.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessCluster(scope, id, cluster, options) {
        // The ServerlessClusterFromSnapshot type is a subset of ServerlessCluster.
        return PostgresqlDatabase.forServerlessClusterFromSnapshot(scope, id, cluster, options);
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessClusterFromSnapshot(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new PostgresqlDatabase(scope, id, {
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            ...props,
        });
        database.trigger.executeAfter(cluster);
        return database;
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseInstance.
     *
     * This method automatically adds the instance to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstance(scope, id, instance, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || instance.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new PostgresqlDatabase(scope, id, {
            target: instance,
            endpoint: instance.instanceEndpoint,
            adminSecret: clusterSecret,
            vpc: instance.vpc,
            ...props,
        });
        database.trigger.executeAfter(instance);
        return database;
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseInstanceFromSnapshot.
     *
     * This method automatically adds the instance to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstanceFromSnapshot(scope, id, instance, options) {
        // The DatabaseInstanceFromSnapshot type is equivalent to DatabaseInstance.
        return PostgresqlDatabase.forInstance(scope, id, instance, options);
    }
    addUserAsOwner(secret) {
        this.ownerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsReader(secret) {
        this.readerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsUnprivileged(secret) {
        this.unprivilegedSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
}
exports.PostgresqlDatabase = PostgresqlDatabase;
_a = JSII_RTTI_SYMBOL_1;
PostgresqlDatabase[_a] = { fqn: "shady-island.PostgresqlDatabase", version: "0.1.6" };
//# sourceMappingURL=data:application/json;base64,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