"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const https = require("https");
// eslint-disable-next-line import/no-extraneous-dependencies
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
// eslint-disable-next-line import/no-extraneous-dependencies
const db_provision_pgsql_1 = require("@libreworks/db-provision-pgsql");
// eslint-disable-next-line import/no-extraneous-dependencies
const pg_1 = require("pg");
const util_1 = require("./util");
const adminSecretArn = process.env.ADMIN_SECRET_ARN;
const ownerSecretArn = process.env.OWNER_SECRET_ARN;
const databaseName = process.env.DB_NAME;
const encoding = process.env.DB_ENCODING || "UTF8";
const locale = process.env.DB_LOCALE;
const schemaName = process.env.SCHEMA_NAME;
const secretsManagerClient = new client_secrets_manager_1.SecretsManagerClient({});
const ownerSecretArns = util_1.parseJsonArrayFromEnv("OWNER_SECRETS");
const readerSecretArns = util_1.parseJsonArrayFromEnv("READER_SECRETS");
const unprivilegedSecretArns = util_1.parseJsonArrayFromEnv("UNPRIVILEGED_SECRETS");
/**
 * Reads an HTTPS resource into a string.
 *
 * We need this function since newer RDS CA certificates aren't in Lambda.
 *
 * @see https://github.com/aws/aws-lambda-base-images/issues/123
 */
function readRemote(url, options) {
    return new Promise((resolve, reject) => {
        https
            .get(url, options || {}, (res) => {
            const data = [];
            res.on("data", (d) => data.push(d));
            res.on("end", () => resolve(Buffer.concat(data)));
        })
            .on("error", (e) => reject(e));
    });
}
const handler = async () => {
    // Here's the first network request:
    // Load the admin secret, needed to create the catalog and its owner.
    const adminSecret = await util_1.fetchSecret(adminSecretArn, secretsManagerClient);
    // Here's the second network request:
    const region = process.env.AWS_REGION || process.env.AWS_DEFAULT_REGION;
    const caBundle = await readRemote(process.env.CA_CERTS_URL ||
        `https://truststore.pki.rds.amazonaws.com/${region}/${region}-bundle.pem`);
    // First we need to connect to the "postgres" database.
    const clientDefaults = {
        host: adminSecret.host,
        port: adminSecret.port,
        user: adminSecret.username,
        password: adminSecret.password,
        connectionTimeoutMillis: 40000,
        ssl: { ca: caBundle },
    };
    const client = new pg_1.Client({
        ...clientDefaults,
        database: "postgres",
    });
    const serverClient = new db_provision_pgsql_1.ServerClient(client);
    // Here's our second network request.
    console.log("About to open a connection to database: postgres");
    await client.connect();
    const catalog = new db_provision_pgsql_1.Catalog(databaseName, encoding, locale);
    console.log(`About to create database: ${databaseName}`);
    await serverClient.createDatabase(catalog);
    console.log(`Database created: ${databaseName}`);
    // Here are the next set of network requests:
    // Load all other Secrets Manager secrets in one go.
    const [schemaOwnerSecret, ownerSecrets, readerSecrets, unprivilegedSecrets] = await Promise.all([
        util_1.fetchSecret(ownerSecretArn, secretsManagerClient),
        util_1.fetchAllSecrets(ownerSecretArns, secretsManagerClient),
        util_1.fetchAllSecrets(readerSecretArns, secretsManagerClient),
        util_1.fetchAllSecrets(unprivilegedSecretArns, secretsManagerClient),
    ]);
    // Map the secrets to the Logins for the sake of performance.
    const loginMap = new Map();
    // Create a Login object for each secret we have. Store it in the map.
    const loginSecrets = [
        schemaOwnerSecret,
        ...ownerSecrets,
        ...readerSecrets,
        ...unprivilegedSecrets,
    ];
    for (const secret of loginSecrets) {
        const login = new db_provision_pgsql_1.Login(secret.username, secret.password);
        loginMap.set(secret, login);
    }
    // Here are the next set of network requests:
    // Let's do this serially until we determine that we can use Promise.all.
    for (const login of loginMap.values()) {
        console.log(`About to create user: ${login.username}`);
        await serverClient.createRole(login);
        console.log(`User created: ${login.username}`);
    }
    const adminRole = new db_provision_pgsql_1.Role(`${databaseName}_${schemaName}_adm`);
    const readerRole = new db_provision_pgsql_1.Role(`${databaseName}_${schemaName}_ro`);
    // Here is the next network request:
    // Create roles for our admin users and read-only users.
    for (const role of [adminRole, readerRole]) {
        console.log(`About to create role: ${role.name}`);
        await serverClient.createRole(role);
        console.log(`Role created: ${role.name}`);
    }
    const grants = [];
    // Create a Grant object for the schema owner.
    let owner = loginMap.get(schemaOwnerSecret);
    grants.push(catalog.grant(owner, "CONNECT", "CREATE", "TEMP"));
    // Create a Grant object for schema admins and all readers.
    for (const secret of [...ownerSecrets, ...readerSecrets]) {
        grants.push(catalog.grant(loginMap.get(secret), "CONNECT", "TEMP"));
    }
    // Create a Grant object for the users with no privileges.
    for (const secret of unprivilegedSecrets) {
        grants.push(catalog.grant(loginMap.get(secret), "CONNECT"));
    }
    // Here are the next set of network requests:
    // Let's do this serially until we determine that we can use Promise.all
    for (const grant of grants) {
        console.log(`About to grant privileges: ${grant.toSql()}`);
        await serverClient.createGrant(grant);
        console.log("Privileges granted");
    }
    console.log("About to close the connection");
    await client.end();
    console.log("Connection closed");
    const schemaClient = new pg_1.Client({
        ...clientDefaults,
        user: owner.name,
        password: owner.password,
        database: databaseName,
    });
    const databaseClient = new db_provision_pgsql_1.DatabaseClient(schemaClient);
    // Here is the next network request:
    console.log(`About to open a connection to database: ${databaseName}`);
    await schemaClient.connect();
    // Here is the next network request:
    const schema = catalog.createSchema(schemaName || owner.username, owner);
    console.log(`About to create schema: ${schema.name} (if it does not already exist)`);
    await schemaClient.query(schema.toSql());
    console.log(`Schema created: ${schema.name}`);
    console.log(`About to alter schema ${schema.name} owner to: ${owner.name}`);
    await schemaClient.query(schema.changeOwner(owner).toSql());
    console.log(`Schema altered: ${schema.name}`);
    const admins = ownerSecrets.map((secret) => loginMap.get(secret));
    const readers = readerSecrets.map((secret) => loginMap.get(secret));
    // Here are the next set of network requests:
    // Create the grants and default permissions for the owner secrets.
    console.log(`About to create schema grants for admins`);
    await databaseClient.createAdminGrants(schema, adminRole, admins);
    // Create the grants and default permissions for the reader secrets.
    console.log(`About to create schema grants for readers`);
    await databaseClient.createReaderGrants(schema, readerRole, readers);
    console.log("About to close the connection");
    await schemaClient.end();
    console.log("Connection closed");
};
module.exports = { handler };
//# sourceMappingURL=data:application/json;base64,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