"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.requestQuotas = exports.getCurrentQuotaValue = void 0;
const client_service_quotas_1 = require("@aws-sdk/client-service-quotas");
const getCurrentQuotaValue = async (region, service, quota) => {
    const client = new client_service_quotas_1.ServiceQuotasClient({
        region,
    });
    const res = await client.send(new client_service_quotas_1.GetServiceQuotaCommand({
        ServiceCode: service,
        QuotaCode: quota,
    }));
    if (typeof res.Quota?.Value === 'undefined') {
        throw new Error(`Invalid response for ${service}/${quota}`);
    }
    return res.Quota.Value;
};
exports.getCurrentQuotaValue = getCurrentQuotaValue;
const getRequestedQuotaChanges = async (region, service, quota) => {
    const client = new client_service_quotas_1.ServiceQuotasClient({
        region,
    });
    const changes = (0, client_service_quotas_1.paginateListRequestedServiceQuotaChangeHistoryByQuota)({
        client,
    }, {
        ServiceCode: service,
        QuotaCode: quota,
    });
    const agg = [];
    for await (const code of changes) {
        agg.push(...(code.RequestedQuotas ?? []));
    }
    return agg;
};
const getPendingQuotaValue = async (region, service, quota) => {
    const requestedChanges = await getRequestedQuotaChanges(region, service, quota);
    console.log('requestedChanges', requestedChanges);
    const requests = requestedChanges.map((value) => {
        if (!value.DesiredValue || !value.Created) {
            throw new Error('invalid quota change request');
        }
        return {
            status: value.Status,
            value: value.DesiredValue,
            createdAt: value.Created,
        };
    });
    return requests;
};
var Status;
(function (Status) {
    Status[Status["GRANTED"] = 0] = "GRANTED";
    Status[Status["REQUESTED"] = 1] = "REQUESTED";
    Status[Status["DENIED"] = 2] = "DENIED";
})(Status || (Status = {}));
const deniedStatuses = [
    client_service_quotas_1.RequestStatus.DENIED,
    client_service_quotas_1.RequestStatus.NOT_APPROVED,
    client_service_quotas_1.RequestStatus.CASE_CLOSED,
    client_service_quotas_1.RequestStatus.INVALID_REQUEST,
];
const pendingStatuses = [client_service_quotas_1.RequestStatus.CASE_OPENED, client_service_quotas_1.RequestStatus.PENDING, client_service_quotas_1.RequestStatus.APPROVED];
const makeRequest = async (region, service, quota, desiredValue) => {
    const client = new client_service_quotas_1.ServiceQuotasClient({
        region,
    });
    const res = await client.send(new client_service_quotas_1.RequestServiceQuotaIncreaseCommand({
        DesiredValue: desiredValue,
        QuotaCode: quota,
        ServiceCode: service,
    }));
    const req = res.RequestedQuota;
    if (!req) {
        throw new Error('bad response from RequestServiceQuotaIncreaseCommand');
    }
    return req;
};
const getStatus = (status) => {
    const isDenied = deniedStatuses.includes(status);
    if (isDenied) {
        return Status.DENIED;
    }
    const isPending = pendingStatuses.includes(status);
    if (isPending) {
        return Status.REQUESTED;
    }
    return Status.GRANTED;
};
const createdAtSort = (a, b) => {
    return new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime();
};
const dqvCache = {};
const clearDQVCache = () => {
    Object.keys(dqvCache).forEach((key) => {
        delete dqvCache[key];
    });
};
const getDefaultQuotaValues = async (region, service) => {
    const key = [region, service].join(':');
    if (!dqvCache[key]) {
        const agg = [];
        const client = new client_service_quotas_1.ServiceQuotasClient({
            region,
        });
        const defaultQuotas = (0, client_service_quotas_1.paginateListAWSDefaultServiceQuotas)({
            client,
        }, {
            ServiceCode: service,
        });
        for await (const quota of defaultQuotas) {
            agg.push(...(quota.Quotas ?? []));
        }
        dqvCache[key] = agg;
    }
    return dqvCache[key];
};
const getDefaultQuotaValue = async (region, service, quota) => {
    const agg = await getDefaultQuotaValues(region, service);
    return agg.find((sq) => sq.QuotaCode === quota)?.Value;
};
const getCurrentOrDefaultQuotaValue = async (region, service, quota) => {
    let currentValue;
    try {
        currentValue = await (0, exports.getCurrentQuotaValue)(region, service, quota);
    }
    catch (e) {
        currentValue = await getDefaultQuotaValue(region, service, quota);
    }
    if (typeof currentValue === 'undefined') {
        throw new Error(`unable to get current or default value of quota: ${service}/${quota}`);
    }
    return currentValue;
};
const requestQuota = async ({ desiredValue, quota, region, service }, { rerequestWhenDenied }) => {
    if (typeof desiredValue === 'string') {
        desiredValue = parseInt(desiredValue, 10);
    }
    const currentValue = await getCurrentOrDefaultQuotaValue(region, service, quota);
    if (desiredValue >= currentValue) {
        const pending = await getPendingQuotaValue(region, service, quota);
        const greaterThan = pending.filter((req) => {
            return req.value >= desiredValue;
        });
        greaterThan.sort(createdAtSort);
        if (greaterThan.length) {
            const [mostRecent] = greaterThan;
            const status = getStatus(mostRecent.status);
            if (rerequestWhenDenied) {
                if (status !== Status.DENIED) {
                    return status;
                }
            }
            else {
                return status;
            }
        }
        const requested = await makeRequest(region, service, quota, desiredValue);
        const status = requested.Status;
        if (!status) {
            throw new Error('no status returned from request');
        }
        return getStatus(status);
    }
    return Status.GRANTED;
};
const requestQuotas = async (quotas, config) => {
    const results = [];
    for (const quota of quotas) {
        const status = await requestQuota(quota, config);
        results.push({
            quota,
            status,
        });
    }
    console.table(results.map(({ quota, status }) => ({
        ...quota,
        status,
    })));
    if (config.failIfNotGranted) {
        const notGranted = results.filter((result) => result.status !== Status.GRANTED);
        if (notGranted.length) {
            throw new Error(`config.failIfNotGranted is true, and ${notGranted.length} quotas are not yet granted`);
        }
    }
    clearDQVCache();
    return results;
};
exports.requestQuotas = requestQuotas;
//# sourceMappingURL=data:application/json;base64,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