import pandas as pd

def select_model(df_output, model_name):
    """
    Selects and returns the scores of a specified model from a DataFrame of model scores.

    This function searches the index of a DataFrame, typically generated by the show_all() function,
    for a model matching the name being provided. If found, it returns the corresponding row 
    of scores. Otherwise, it issues a "Model Not Found" message and a list of available models.

    Parameters
    ----------
    df_output : DataFrame
        A DataFrame containing scores of various models, usually outputted by show_all().
    model_name : str
        The name of the model to search for in the DataFrame.

    Returns
    -------
    Series or str
        The row from the DataFrame corresponding to the specified model.
        Otherwise, returns a "Model Not Found" message.

    Raises
    ------
    TypeError
        If df_output is not a pandas DataFrame or model_name is not a string.
    ValueError
        If df_output DataFrame is empty.

    Examples
    --------
    >>> from autopredictor.select_model import select_model
    >>> df_scores = pd.DataFrame({'MAE': [2.5, 3.6],
                                  'MSE': [10.1, 20.3]},
                                 index=['Linear Regression', 'Random Forest'])
    >>> select_model(df_scores, 'Linear Regression')
                       MAE   MSE
    Linear Regression  2.5  10.1
    
    >>> select_model(df_scores, 'Support Vector Machine')
    "Model 'Support Vector Machine' not found. Here is the list of the models available: Linear Regression, Random Forest."
    """
    if not isinstance(df_output, pd.DataFrame):
        raise TypeError("df_output must be a pandas DataFrame.")

    if not isinstance(model_name, str):
        raise TypeError("model_name must be a string.")
    model_name = model_name.strip()

    if df_output.empty:
        raise ValueError("df_output DataFrame is empty.")
    
    if model_name in df_output.index:
        return df_output.loc[[model_name]]
    else:
        available_models = df_output.index.tolist()
        available_models_string = ", ".join(available_models)
        return f"Model '{model_name}' not found. Here is the list of the models available: {available_models_string}."
