"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DotNetFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const lambda = require("aws-cdk-lib/aws-lambda");
const bundling_1 = require("./private/bundling");
const util_1 = require("./private/util");
/**
 * A .NET Lambda function
 */
class DotNetFunction extends lambda.Function {
    constructor(scope, id, props) {
        const { projectDir } = props;
        // Find aws-lambda-tools-defaults.json in project directory
        const lambdaToolsDefaults = (0, util_1.getLambdaToolsDefaults)(projectDir);
        // Resolve Lambda runtime from properties, aws-lambda-tools-defaults.json or default to PROVIDED_AL2
        let runtime = props.runtime;
        if (!runtime) {
            const toolsRuntime = lambdaToolsDefaults?.['function-runtime'];
            if (toolsRuntime &&
                (toolsRuntime.startsWith('dotnet') ||
                    toolsRuntime.startsWith('provided'))) {
                const family = toolsRuntime.startsWith('dotnet')
                    ? lambda.RuntimeFamily.DOTNET_CORE
                    : lambda.RuntimeFamily.OTHER;
                runtime = new lambda.Runtime(toolsRuntime, family);
            }
            else {
                runtime = lambda.Runtime.DOTNET_6;
            }
        }
        if (runtime.family !== lambda.RuntimeFamily.DOTNET_CORE &&
            runtime.family !== lambda.RuntimeFamily.OTHER) {
            throw new Error(`Unsupported runtime '${runtime.name}'. Only '.NET' and 'provided' runtimes are supported.`);
        }
        // Resolve architecture from properties, aws-lambda-tools-defaults.json, get current processor architect for provided runtimes
        // or default to X86_64.
        let architecture = props.architecture;
        if (!architecture) {
            const toolsArchitecture = lambdaToolsDefaults?.['function-architecture'];
            toolsArchitecture;
            if (toolsArchitecture) {
                architecture = lambdaArchitecture(toolsArchitecture);
            }
            else if (runtime.family === lambda.RuntimeFamily.OTHER) {
                architecture = lambdaArchitecture(process.arch);
            }
            else {
                architecture = lambda.Architecture.X86_64;
            }
        }
        if (runtime.family === lambda.RuntimeFamily.OTHER &&
            architecture?.name !== lambdaArchitecture(process.arch).name) {
            throw new Error(`Unsupported architecture '${runtime.name}'. Only '${process.arch}' architure is supported for provided runtimes.`);
        }
        // Resolve solution directory from property.
        // If is file, than use the folder else find traverse parent folders of projectDir to find solution file.
        let solutionDir;
        if (props.solutionDir) {
            solutionDir = props.solutionDir;
            const parsedSolutionDir = path.parse(props.solutionDir);
            if (parsedSolutionDir.ext && parsedSolutionDir.ext === '.sln') {
                if (!fs.existsSync(props.solutionDir)) {
                    throw new Error(`Solutions file at ${props.solutionDir} doesn't exist`);
                }
            }
            else if (!fs.readdirSync(solutionDir).find((file) => file.endsWith('.sln'))) {
                throw new Error(`Solution file (.sln) at ${props.solutionDir} doesn't exist`);
            }
        }
        else {
            const solutionFile = (0, util_1.findUp)('.sln', projectDir);
            solutionDir = path.resolve(solutionFile ? path.dirname(solutionFile) : projectDir);
        }
        // Resolve handler based on project name
        let handler;
        if (props.handler) {
            handler = props.handler;
        }
        else if (lambdaToolsDefaults && lambdaToolsDefaults['function-handler']) {
            handler = lambdaToolsDefaults['function-handler'];
        }
        else {
            const projectFile = fs
                .readdirSync(projectDir)
                .find((file) => file.endsWith('.csproj'));
            if (!projectFile) {
                throw new Error(`.csproj file at ${props.projectDir} doesn't exist`);
            }
            const projectName = projectFile.replace('.csproj', '');
            handler = projectName;
        }
        super(scope, id, {
            ...props,
            runtime,
            architecture,
            code: bundling_1.Bundling.bundle({
                ...(props.bundling ?? {}),
                runtime,
                architecture,
                projectDir,
                solutionDir,
            }),
            handler,
        });
    }
}
exports.DotNetFunction = DotNetFunction;
_a = JSII_RTTI_SYMBOL_1;
DotNetFunction[_a] = { fqn: "@aws-cdk/aws-lambda-dotnet.DotNetFunction", version: "0.0.3" };
function lambdaArchitecture(architecture) {
    return architecture === 'arm64'
        ? lambda.Architecture.ARM_64
        : lambda.Architecture.X86_64;
}
//# sourceMappingURL=data:application/json;base64,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