"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Packaging = exports.DependenciesFile = void 0;
const fs = require("fs");
const path = require("path");
var DependenciesFile;
(function (DependenciesFile) {
    DependenciesFile["PIP"] = "requirements.txt";
    DependenciesFile["POETRY"] = "poetry.lock";
    DependenciesFile["PIPENV"] = "Pipfile.lock";
    DependenciesFile["NONE"] = "";
})(DependenciesFile || (exports.DependenciesFile = DependenciesFile = {}));
class Packaging {
    /**
     * Standard packaging with `pip`.
     */
    static withPip() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIP,
        });
    }
    /**
     * Packaging with `pipenv`.
     */
    static withPipenv() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIPENV,
            // By default, pipenv creates a virtualenv in `/.local`, so we force it to create one in the package directory.
            // At the end, we remove the virtualenv to avoid creating a duplicate copy in the Lambda package.
            exportCommand: `PIPENV_VENV_IN_PROJECT=1 pipenv requirements > ${DependenciesFile.PIP} && rm -rf .venv`,
        });
    }
    /**
     * Packaging with `poetry`.
     */
    static withPoetry(props) {
        return new Packaging({
            dependenciesFile: DependenciesFile.POETRY,
            // Export dependencies with credentials available in the bundling image.
            exportCommand: [
                'poetry', 'export',
                ...props?.poetryIncludeHashes ? [] : ['--without-hashes'],
                ...props?.poetryWithoutUrls ? ['--without-urls'] : [],
                '--with-credentials',
                '--format', DependenciesFile.PIP,
                '--output', DependenciesFile.PIP,
            ].join(' '),
        });
    }
    /**
     * No dependencies or packaging.
     */
    static withNoPackaging() {
        return new Packaging({ dependenciesFile: DependenciesFile.NONE });
    }
    static fromEntry(entry, poetryIncludeHashes, poetryWithoutUrls) {
        if (fs.existsSync(path.join(entry, DependenciesFile.PIPENV))) {
            return this.withPipenv();
        }
        if (fs.existsSync(path.join(entry, DependenciesFile.POETRY))) {
            return this.withPoetry({ poetryIncludeHashes, poetryWithoutUrls });
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.PIP))) {
            return this.withPip();
        }
        else {
            return this.withNoPackaging();
        }
    }
    constructor(props) {
        this.dependenciesFile = props.dependenciesFile;
        this.exportCommand = props.exportCommand;
    }
}
exports.Packaging = Packaging;
//# sourceMappingURL=data:application/json;base64,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