# Copyright 2022, OpenVoiceOS.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from os import environ, listdir, path

import datetime
import os
import tempfile
from lingua_franca.format import get_date_strings
from mycroft.skills.api import SkillApi
from mycroft.skills.core import (MycroftSkill, intent_file_handler,
                                 resting_screen_handler)
from mycroft_bus_client import Message
from ovos_skills_manager.utils import get_skills_examples
from ovos_utils import classproperty
from ovos_utils.log import LOG
from ovos_utils.process_utils import RuntimeRequirements

from .skill import (DashboardHandler, CardGenerator)


class OVOSHomescreenSkill(MycroftSkill):
    # The constructor of the skill, which calls MycroftSkill's constructor
    def __init__(self):
        super(OVOSHomescreenSkill, self).__init__(name="OVOSHomescreen")
        self.notifications_storage_model = []
        self.def_wallpaper_folder = path.dirname(__file__) + '/ui/wallpapers/'
        self.loc_wallpaper_folder = None
        self.selected_wallpaper_path = None
        self.selected_wallpaper = None
        self.default_provider_set = False
        self.wallpaper_collection = []
        self.rtlMode = None  # Get from config after __init__ is done

        # Populate skill IDs to use for data sources
        self.datetime_skill_id = None  # Get from config after __init__ is done
        self.examples_skill_id = None  # Get from config after __init__ is done
        self.datetime_api = None
        self.skill_info_api = None

        # A variable to turn on/off the example text
        self.examples_enabled = True

        # Display Configuration Variables
        self.dashboard_handler = None

        # Media State Tracking For Widget
        # Needed for setting qml button state
        self.media_widget_player_state = None

        # Offline / Online State
        self.system_connectivity = None

    @classproperty
    def runtime_requirements(self):
        return RuntimeRequirements(internet_before_load=False,
                                   network_before_load=False,
                                   gui_before_load=True,
                                   requires_internet=False,
                                   requires_network=False,
                                   requires_gui=True,
                                   no_internet_fallback=True,
                                   no_network_fallback=True,
                                   no_gui_fallback=False)

    def initialize(self):
        self.dashboard_handler = DashboardHandler(self.file_system.path,
                                                  path.dirname(__file__))
        self.card_generator = CardGenerator(self.file_system.path, self.bus,
                                            path.dirname(__file__))
        self.datetime_api = None
        self.loc_wallpaper_folder = self.file_system.path + '/wallpapers/'
        self.rtlMode = 1 if self.config_core.get("rtl", False) else 0

        self.datetime_skill_id = self.settings.get("datetime_skill_id")
        self.examples_enabled = 1 if self.settings.get(
            "examples_enabled", True) else 0

        if self.examples_enabled:
            self.examples_skill_id = self.settings.get("examples_skill")

        now = datetime.datetime.now()
        callback_time = datetime.datetime(
            now.year, now.month, now.day, now.hour, now.minute
        ) + datetime.timedelta(seconds=60)
        self.schedule_repeating_event(self.update_dt, callback_time, 10)

        # Handler Registration For Notifications
        self.add_event("homescreen.wallpaper.set",
                       self.handle_set_wallpaper)
        self.add_event("ovos.notification.update_counter",
                       self.handle_notification_widget_update)
        self.add_event("ovos.notification.update_storage_model",
                       self.handle_notification_storage_model_update)
        self.gui.register_handler("homescreen.swipe.change.wallpaper",
                                  self.change_wallpaper)
        self.add_event("mycroft.ready", self.handle_mycroft_ready)

        # Handler Registration For Widgets
        self.add_event("ovos.widgets.timer.update",
                       self.handle_timer_widget_manager)
        self.add_event("ovos.widgets.timer.display",
                       self.handle_timer_widget_manager)
        self.add_event("ovos.widgets.timer.remove",
                       self.handle_timer_widget_manager)

        self.add_event("ovos.widgets.alarm.update",
                       self.handle_alarm_widget_manager)
        self.add_event("ovos.widgets.alarm.display",
                       self.handle_alarm_widget_manager)
        self.add_event("ovos.widgets.alarm.remove",
                       self.handle_alarm_widget_manager)

        # Handler Registration For Dashboard
        self.add_event("ovos.homescreen.dashboard.add.card",
                       self.add_dashboard_card)
        self.add_event("ovos.homescreen.dashboard.generate.card",
                       self.generate_dashboard_card)
        self.gui.register_handler("ovos.homescreen.dashboard.generate.card",
                                  self.generate_dashboard_card)
        self.gui.register_handler("ovos.homescreen.dashboard.remove.card",
                                  self.remove_dashboard_card)

        if not self.file_system.exists("wallpapers"):
            os.mkdir(path.join(self.file_system.path, "wallpapers"))

        # Handler For Weather Response
        self.bus.on("skill-ovos-weather.openvoiceos.weather.response", self.update_weather_response)

        # Handler For OCP Player State Tracking
        self.bus.on("gui.player.media.service.sync.status",
                    self.handle_media_player_state_update)
        self.bus.on("ovos.common_play.track_info.response",
                    self.handle_media_player_widget_update)

        # Handler For Offline Widget
        self.bus.on("mycroft.network.connected", self.on_network_connected)
        self.bus.on("mycroft.internet.connected", self.on_internet_connected)
        self.bus.on("enclosure.notify.no_internet", self.on_no_internet)

        # Handle Screenshot Response
        self.bus.on("ovos.display.screenshot.get.response",
                    self.screenshot_taken)

        self.collect_wallpapers()
        self._load_skill_apis()

        self.schedule_repeating_event(self.update_weather, callback_time, 900)
        self.schedule_repeating_event(self.update_examples, callback_time, 900)

        self.bus.on("ovos.wallpaper.manager.loaded",
                    self.register_homescreen_wallpaper_provider)
        
        self.bus.on(f"{self.skill_id}.get.wallpaper.collection",
                    self.supply_wallpaper_collection)
        
        self.bus.on("ovos.wallpaper.manager.setup.default.provider.response",
                    self.handle_default_provider_response)
        
        # We can't depend on loading order, so send a registration request
        # Regardless on startup
        self.register_homescreen_wallpaper_provider()

        # Get / Set the default wallpaper
        # self.selected_wallpaper = self.settings.get(
        #     "wallpaper") or "default.jpg"

        self.bus.emit(Message("mycroft.device.show.idle"))

    #####################################################################
    # Homescreen Registration & Handling

    @resting_screen_handler("OVOSHomescreen")
    def handle_idle(self, message):
        self._load_skill_apis()
        LOG.debug('Activating OVOSHomescreen')
        self.gui['wallpaper_path'] = self.selected_wallpaper_path
        self.gui['selected_wallpaper'] = self.selected_wallpaper
        self.gui['notification'] = {}
        self.gui["notification_model"] = self.notifications_storage_model
        self.gui["system_connectivity"] = "offline"
        self.gui["applications_model"] = self.build_voice_applications_model()
        self.gui["dashboard_model"] = self.get_dashboard_cards()
        self.gui["persistent_menu_hint"] = self.settings.get("persistent_menu_hint", False)

        try:
            self.update_dt()
            self.update_weather()
            self.update_examples()
        except Exception as e:
            LOG.error(e)

        self.gui['rtl_mode'] = self.rtlMode
        self.gui['dateFormat'] = self.config_core.get("date_format") or "DMY"
        self.gui.show_page("idle.qml")

    def update_examples(self):
        """
        Loads or updates skill examples via the skill_info_api.
        """
        if self.skill_info_api:
            self.gui['skill_examples'] = {"examples": self.skill_info_api.skill_info_examples()}
        else:
            skill_examples = get_skills_examples(randomize=self.settings.get("randomize_examples", True))
            self.gui['skill_examples'] = {"examples": skill_examples}

        self.gui['skill_info_enabled'] = self.examples_enabled
        self.gui['skill_info_prefix'] = self.settings.get("examples_prefix",
                                                          True)

    def update_dt(self):
        """
        Loads or updates date/time via the datetime_api.
        """
        if not self.datetime_api and self.datetime_skill_id:
            LOG.debug("Requested update before datetime API loaded")
            self._load_skill_apis()
        if self.datetime_api:
            time_string = self.datetime_api.get_display_current_time()
            date_string = self.datetime_api.get_display_date()
            weekday_string = self.datetime_api.get_weekday()
            day_string, month_string = self._split_month_string(self.datetime_api.get_month_date())
            year_string = self.datetime_api.get_year()
        else:
            date_string_object = get_date_strings(date_format=self.config_core.get("date_format", "MDY"), 
                                                  time_format=self.config_core.get("time_format", "full"),
                                                  lang=self.lang)
            time_string = date_string_object.get("time_string")
            date_string = date_string_object.get("date_string")
            weekday_string = date_string_object.get("weekday_string")
            day_string = date_string_object.get("day_string")
            month_string = date_string_object.get("month_string")
            year_string = date_string_object.get("year_string")

        self.gui["time_string"] = time_string
        self.gui["date_string"] = date_string
        self.gui["weekday_string"] = weekday_string
        self.gui['day_string'] = day_string
        self.gui["month_string"] = month_string
        self.gui["year_string"] = year_string

    def update_weather(self):
        """
        Loads or updates weather via the weather_api.
        """
        self.bus.emit(Message("skill-ovos-weather.openvoiceos.weather.request"))

    def update_weather_response(self, message=None):
        """
        Weather Update Response
        """
        current_weather_report = message.data.get("report")
        if current_weather_report:
            self.gui["weather_api_enabled"] = True
            self.gui["weather_code"] = current_weather_report.get("weather_code")
            self.gui["weather_temp"] = current_weather_report.get("weather_temp")
        else:
            self.gui["weather_api_enabled"] = False

    def on_network_connected(self, message):
        self.system_connectivity = "network"
        self.gui["system_connectivity"] = self.system_connectivity

    def on_internet_connected(self, message):
        self.system_connectivity = "online"
        self.gui["system_connectivity"] = self.system_connectivity

    def on_no_internet(self, message):
        self.system_connectivity = "offline"
        self.gui["system_connectivity"] = self.system_connectivity

    #####################################################################
    # Homescreen Wallpaper Provider and Consumer Handling
    # Follows OVOS PHAL Wallpaper Manager API

    def collect_wallpapers(self):
        def_wallpaper_collection, loc_wallpaper_collection = None, None
        for dirname, dirnames, filenames in os.walk(self.def_wallpaper_folder):
            def_wallpaper_collection = filenames
            def_wallpaper_collection = [os.path.join(dirname, wallpaper) for wallpaper in def_wallpaper_collection]

        for dirname, dirnames, filenames in os.walk(self.loc_wallpaper_folder):
            loc_wallpaper_collection = filenames
            loc_wallpaper_collection = [os.path.join(dirname, wallpaper) for wallpaper in loc_wallpaper_collection]

        self.wallpaper_collection = def_wallpaper_collection + loc_wallpaper_collection
        
    def register_homescreen_wallpaper_provider(self, message=None):
        self.bus.emit(Message("ovos.wallpaper.manager.register.provider", {
            "provider_name": self.skill_id,
            "provider_display_name": "OVOSHomescreen"
        }))

    def supply_wallpaper_collection(self, message):
        self.bus.emit(Message("ovos.wallpaper.manager.collect.collection.response", {
            "provider_name": self.skill_id,
            "wallpaper_collection": self.wallpaper_collection
        }))
        # We need to call this here as we know wallpaper collection is ready
        if not self.default_provider_set:
            self.setup_default_provider()
        
    def setup_default_provider(self):
        self.bus.emit(Message("ovos.wallpaper.manager.setup.default.provider", {
            "provider_name": self.skill_id,
            "default_wallpaper_name": self.settings.get("wallpaper", "default.jpg")
        }))
    
    def handle_default_provider_response(self, message):
        self.default_provider_set = True
        url = message.data.get("url")
        self.selected_wallpaper_path = self.extract_wallpaper_info(url)[0] 
        self.selected_wallpaper = self.extract_wallpaper_info(url)[1]
        self.gui['wallpaper_path'] = self.selected_wallpaper_path
        self.gui['selected_wallpaper'] = self.selected_wallpaper

    @intent_file_handler("change.wallpaper.intent")
    def change_wallpaper(self, _):
        self.bus.emit(Message("ovos.wallpaper.manager.change.wallpaper"))

    def get_wallpaper_idx(self, filename):
        try:
            index_element = self.wallpaper_collection.index(filename)
            return index_element
        except ValueError:
            return None

    def handle_set_wallpaper(self, message):
        url = message.data.get("url")
        self.selected_wallpaper_path = self.extract_wallpaper_info(url)[0] 
        self.selected_wallpaper = self.extract_wallpaper_info(url)[1]
        self.gui['wallpaper_path'] = self.selected_wallpaper_path
        self.gui['selected_wallpaper'] = self.selected_wallpaper

    def extract_wallpaper_info(self, wallpaper):
        wallpaper_split = wallpaper.rsplit('/', 1)
        wallpaper_path = wallpaper_split[0] + "/"
        wallpaper_filename = wallpaper_split[1]
        return wallpaper_path, wallpaper_filename

    #####################################################################
    # Manage notifications widget

    def handle_notification_widget_update(self, message):
        # Receives notification counter update
        # Emits request to update storage model on counter update
        notifcation_count = message.data.get("notification_counter", "")
        self.gui["notifcation_counter"] = notifcation_count
        self.bus.emit(Message("ovos.notification.api.request.storage.model"))

    def handle_notification_storage_model_update(self, message):
        # Receives updated storage model and forwards it to widget
        self.notifications_storage_model = message.data.get("notification_model", "")
        self.gui["notification_model"] = self.notifications_storage_model

    #####################################################################
    # Misc
    def shutdown(self):
        self.cancel_all_repeating_events()

    def handle_mycroft_ready(self, message):
        self._load_skill_apis()

    def _load_skill_apis(self):
        """
        Loads weather, date/time, and examples skill APIs
        """
        # Import Date Time Skill As Date Time Provider if configured (default LF)
        try:
            if not self.datetime_api and self.datetime_skill_id:
                self.datetime_api = SkillApi.get(self.datetime_skill_id)
        except Exception as e:
            LOG.error(f"Failed to import DateTime Skill: {e}")

        # Import Skill Info Skill if configured (default OSM)
        if not self.skill_info_api and self.examples_skill_id:
            try:
                self.skill_info_api = SkillApi.get(self.examples_skill_id)
            except Exception as e:
                LOG.error(f"Failed to import Info Skill: {e}")

    def _split_month_string(self, month_date: str) -> list:
        """
        Splits a month+date string into month and date (i.e. "August 06" -> ["August", "06"])
        :param month_date: formatted month and day of month ("August 06" or "06 August")
        :return: [day, month]
        """
        month_string = month_date.split(" ")
        if self.config_core.get('date_format') == 'MDY':
            day_string = month_string[1]
            month_string = month_string[0]
        else:
            day_string = month_string[0]
            month_string = month_string[1]

        return [day_string, month_string]

    #####################################################################
    # Build Voice Applications Model

    def find_icon_full_path(self, icon_name):
        localuser = environ.get('USER')
        folder_search_paths = ["/usr/share/icons/", "/usr/local/share/icons/",
                               f"/home/{localuser}/.local/share/icons/"]
        for folder_search_path in folder_search_paths:
            # SVG extension
            icon_full_path = folder_search_path + icon_name + ".svg"
            if path.exists(icon_full_path):
                return icon_full_path
            # PNG extension
            icon_full_path = folder_search_path + icon_name + ".png"
            if path.exists(icon_full_path):
                return icon_full_path
            # JPEG extension
            icon_full_path = folder_search_path + icon_name + ".jpg"
            if path.exists(icon_full_path):
                return icon_full_path

    def parse_desktop_file(self, file_path):
        if path.isfile(file_path) and path.splitext(file_path)[1] == ".desktop":

            if path.isfile(file_path) and path.isfile(file_path) and path.getsize(file_path) > 0:

                with open(file_path, "r") as f:
                    file_contents = f.read()

                    name_start = file_contents.find("Name=")
                    name_end = file_contents.find("\n", name_start)
                    name = file_contents[name_start + 5:name_end]

                    icon_start = file_contents.find("Icon=")
                    icon_end = file_contents.find("\n", icon_start)
                    icon_name = file_contents[icon_start + 5:icon_end]
                    icon = self.find_icon_full_path(icon_name)

                    exec_start = file_contents.find("Exec=")
                    exec_end = file_contents.find("\n", exec_start)
                    exec_line = file_contents[exec_start + 5:exec_end]
                    exec_array = exec_line.split(" ")
                    for arg in exec_array:
                        if arg.find("--skill=") == 0:
                            skill_name = arg.split("=")[1]
                            break
                        else:
                            skill_name = "None"
                    exec_path = skill_name

                    categories_start = file_contents.find("Categories=")
                    categories_end = file_contents.find("\n", categories_start)
                    categories = file_contents[categories_start +
                                               11:categories_end]

                    categories_list = categories.split(";")

                    if "VoiceApp" in categories_list:
                        app_entry = {
                            "name": name,
                            "thumbnail": icon,
                            "action": exec_path
                        }
                        return app_entry
                    else:
                        return None
            else:
                return None
        else:
            return None

    def build_voice_applications_model(self):
        voiceApplicationsList = []
        localuser = environ.get('USER')
        file_list = ["/usr/share/applications/", "/usr/local/share/applications/",
                     f"/home/{localuser}/.local/share/applications/"]
        for file_path in file_list:
            if os.path.isdir(file_path):
                files = listdir(file_path)
                for file in files:
                    app_dict = self.parse_desktop_file(file_path + file)
                    if app_dict is not None:
                        voiceApplicationsList.append(app_dict)

        try:
            sort_on = "name"
            decorated = [(dict_[sort_on], dict_)
                         for dict_ in voiceApplicationsList]
            decorated.sort()
            return [dict_ for (key, dict_) in decorated]

        except Exception:
            return voiceApplicationsList

    #####################################################################
    # Handle Widgets

    def handle_timer_widget_manager(self, message):
        timerWidget = message.data.get("widget", {})
        self.gui.send_event("ovos.timer.widget.manager.update", timerWidget)

    def handle_alarm_widget_manager(self, message):
        alarmWidget = message.data.get("widget", {})
        self.gui.send_event("ovos.alarm.widget.manager.update", alarmWidget)

    #### Media Player Widget UI Handling - Replaces Examples UI Bar ####
    def handle_media_player_state_update(self, message):
        """
        Handles OCP State Updates
        """
        player_state = message.data.get("state")
        if player_state == 1:
            self.bus.emit(Message("ovos.common_play.track_info"))
            self.media_widget_player_state = "playing"
            self.gui.send_event("ovos.media.widget.manager.update", {
                "enabled": True,
                "widget": {},
                "state": "playing"
            })
        elif player_state == 0:
            self.media_widget_player_state = "stopped"
            self.gui.send_event("ovos.media.widget.manager.update", {
                "enabled": False,
                "widget": {},
                "state": "stopped"
            })
        elif player_state == 2:
            self.bus.emit(Message("ovos.common_play.track_info"))
            self.media_widget_player_state = "paused"
            self.gui.send_event("ovos.media.widget.manager.update", {
                "enabled": True,
                "widget": {},
                "state": "paused"
            })

    def handle_media_player_widget_update(self, message=None):
        self.gui.send_event("ovos.media.widget.manager.update", {
            "enabled": True,
            "widget": message.data,
            "state": self.media_widget_player_state
        })

    ######################################################################
    # Handle Dashboard

    def generate_dashboard_card(self, message=None):
        """
        Generate a custom dashboard card from the UI
        """
        if message is None:
            return
        card = message.data.get("card", {})
        self.card_generator.generate(card)

    def add_dashboard_card(self, message=None):
        """
        Adds a card to the dashboard from external source
        """
        if message is not None:
            card = message.data.get("card", {})
            self.dashboard_handler.add_item(card)

        self.gui['dashboard_model'] = self.get_dashboard_cards()

    def remove_dashboard_card(self, message=None):
        """
        Removes a card from the dashboard from external source
        """
        if message is not None:
            self.log.info(f"Removing card: {message.data.get('card_id', {})}")
            card_id = message.data.get("card_id", None)
            self.dashboard_handler.remove_item(card_id)

        self.gui['dashboard_model'] = self.get_dashboard_cards()

    def get_dashboard_cards(self, message=None):
        """
        Returns the current dashboard cards
        """
        # Dump the model to a stringified JSON object
        cards = self.dashboard_handler.get_collection()
        collection = {"collection": cards}
        return collection

    ######################################################################
    # Handle Screenshot

    @intent_file_handler("take.screenshot.intent")
    def take_screenshot(self, message):
        folder_path = self.settings.get("screenshot_folder", "")

        if not folder_path:
            folder_path = os.path.expanduser('~') + "/Pictures"

        if not os.path.exists(folder_path):
            try:
                os.makedirs(folder_path, exist_ok=True)
            except OSError as e:
                LOG.error("Could not create screenshot folder: " + str(e))
                folder_path = tempfile.gettempdir()

        self.bus.emit(Message("ovos.display.screenshot.get", {"folderpath": folder_path}))

    def screenshot_taken(self, message):
        result = message.data.get("result")
        display_message = f"Screenshot saved to {result}"
        self.gui.show_notification(display_message)


def create_skill():
    return OVOSHomescreenSkill()
