"""The module that defines the ``InvalidOptionsException`` model.

SPDX-License-Identifier: AGPL-3.0-only OR BSD-3-Clause-Clear
"""

from __future__ import annotations

import typing as t
from dataclasses import dataclass, field

import cg_request_args as rqa
from httpx import Response

from ..utils import to_dict
from .base_error import BaseError


@dataclass
class InvalidOptionsException(BaseError):
    """The exception raised when invalid plagiarism options are present."""

    #: The list of invalid options.
    invalid_options: t.Sequence[str]

    raw_data: t.Optional[t.Dict[str, t.Any]] = field(init=False, repr=False)

    data_parser: t.ClassVar = rqa.Lazy(
        lambda: BaseError.data_parser.parser.combine(
            rqa.FixedMapping(
                rqa.RequiredArgument(
                    "invalid_options",
                    rqa.List(rqa.SimpleValue.str),
                    doc="The list of invalid options.",
                ),
            )
        ).use_readable_describe(True)
    )

    def to_dict(self) -> t.Dict[str, t.Any]:
        res: t.Dict[str, t.Any] = {
            "invalid_options": to_dict(self.invalid_options),
            "message": to_dict(self.message),
            "description": to_dict(self.description),
            "code": to_dict(self.code),
            "request_id": to_dict(self.request_id),
        }
        return res

    @classmethod
    def from_dict(
        cls: t.Type[InvalidOptionsException],
        d: t.Dict[str, t.Any],
        response: t.Optional[Response] = None,
    ) -> InvalidOptionsException:
        parsed = cls.data_parser.try_parse(d)

        res = cls(
            invalid_options=parsed.invalid_options,
            message=parsed.message,
            description=parsed.description,
            code=parsed.code,
            request_id=parsed.request_id,
            response=response,
        )
        res.raw_data = d
        return res


import os

if os.getenv("CG_GENERATING_DOCS", "False").lower() in ("", "true"):
    from .api_codes import APICodes
