"""The module that defines the ``RubricAnalyticsDataRow`` model.

SPDX-License-Identifier: AGPL-3.0-only OR BSD-3-Clause-Clear
"""

from __future__ import annotations

import typing as t
from dataclasses import dataclass, field

import cg_request_args as rqa

from ..utils import to_dict


@dataclass
class RubricAnalyticsDataRow:
    """The analytics for a single rubric row."""

    #: The item that is selected.
    item_id: int
    #: The multiplier that it has.
    multiplier: float

    raw_data: t.Optional[t.Dict[str, t.Any]] = field(init=False, repr=False)

    data_parser: t.ClassVar = rqa.Lazy(
        lambda: rqa.FixedMapping(
            rqa.RequiredArgument(
                "item_id",
                rqa.SimpleValue.int,
                doc="The item that is selected.",
            ),
            rqa.RequiredArgument(
                "multiplier",
                rqa.SimpleValue.float,
                doc="The multiplier that it has.",
            ),
        ).use_readable_describe(True)
    )

    def to_dict(self) -> t.Dict[str, t.Any]:
        res: t.Dict[str, t.Any] = {
            "item_id": to_dict(self.item_id),
            "multiplier": to_dict(self.multiplier),
        }
        return res

    @classmethod
    def from_dict(
        cls: t.Type[RubricAnalyticsDataRow], d: t.Dict[str, t.Any]
    ) -> RubricAnalyticsDataRow:
        parsed = cls.data_parser.try_parse(d)

        res = cls(
            item_id=parsed.item_id,
            multiplier=parsed.multiplier,
        )
        res.raw_data = d
        return res
