import time
from concurrent.futures import ThreadPoolExecutor
from datetime import datetime, timedelta
from typing import Any, Dict, Generator, List, Optional

import requests
import structlog
import urllib3
from requests.adapters import HTTPAdapter, Retry

from rated_exporter_sdk.providers.prometheus.auth import PrometheusAuth
from rated_exporter_sdk.providers.prometheus.errors import (
    PrometheusAPIError,
    PrometheusConnectionError,
    PrometheusQueryError,
    PrometheusTimeoutError,
    PrometheusValidationError,
)
from rated_exporter_sdk.providers.prometheus.types import (
    MetricIdentifier,
    MetricMetadata,
    MetricType,
    MetricValueType,
    PrometheusMetric,
    PrometheusQueryOptions,
    PrometheusQueryResult,
    Sample,
    Target,
    TargetHealth,
)
from rated_exporter_sdk.providers.prometheus.validation import QueryValidator

logger = structlog.getLogger(__name__)


class PrometheusClient:
    """
    Client for interacting with Prometheus API.

    Features:
    - Robust error handling and retries
    - Query validation
    - Batch querying capabilities
    - Streaming support for large datasets
    - Automatic rate limiting
    - Connection pooling
    """

    QUERY_ENDPOINT = "/api/v1/query"
    QUERY_RANGE_ENDPOINT = "/api/v1/query_range"
    SERIES_ENDPOINT = "/api/v1/series"
    TARGETS_ENDPOINT = "/api/v1/targets"
    METADATA_ENDPOINT = "/api/v1/metadata"
    RULES_ENDPOINT = "/api/v1/rules"
    ALERTS_ENDPOINT = "/api/v1/alerts"

    def __init__(
        self,
        base_url: str,
        auth: Optional[PrometheusAuth] = None,
        timeout: float = 30.0,
        max_retries: int = 3,
        retry_backoff_factor: float = 0.1,
        pool_connections: int = 10,
        pool_maxsize: int = 10,
        max_parallel_queries: int = 5,
    ):
        """
        Initialize Prometheus client.

        Args:
            base_url: Base URL of Prometheus instance
            auth: Authentication configuration
            timeout: Default timeout for requests in seconds
            max_retries: Maximum number of retries for failed requests
            retry_backoff_factor: Backoff factor between retries
            pool_connections: Number of connection pools to maintain
            pool_maxsize: Maximum size of each connection pool
            max_parallel_queries: Maximum number of parallel queries

        Raises:
            PrometheusValidationError: If configuration is invalid
        """
        if not base_url:
            raise PrometheusValidationError(
                "Base URL must be provided", parameter="base_url"
            )

        self.base_url = base_url.rstrip("/")
        self.auth = auth or PrometheusAuth()
        self.timeout = timeout
        self.max_parallel_queries = max_parallel_queries

        # Configure retry strategy
        retry_strategy = Retry(
            total=max_retries,
            backoff_factor=retry_backoff_factor,
            status_forcelist=[408, 429, 500, 502, 503, 504],
            allowed_methods=["HEAD", "GET", "POST"],
        )

        # Configure connection pooling
        adapter = HTTPAdapter(
            max_retries=retry_strategy,
            pool_connections=pool_connections,
            pool_maxsize=pool_maxsize,
        )

        # Initialize session
        self.session = requests.Session()
        self.session.mount("http://", adapter)
        self.session.mount("https://", adapter)

        # Configure authentication
        ssl_config = self.auth.get_ssl_config()
        self.session.verify = bool(ssl_config.get("verify", True))
        cert = ssl_config.get("cert")
        if isinstance(cert, tuple):
            self.session.cert = cert
        elif isinstance(cert, str):
            self.session.cert = cert
        else:
            self.session.cert = None

        self.validator = QueryValidator()
        self.thread_pool = ThreadPoolExecutor(max_workers=max_parallel_queries)

    def _make_request(self, method: str, endpoint: str, **kwargs) -> Dict[str, Any]:
        """
        Make HTTP request to Prometheus API.

        Args:
            method: HTTP method
            endpoint: API endpoint
            **kwargs: Additional request parameters

        Returns:
            Dict containing API response

        Raises:
            PrometheusConnectionError: If connection fails
            PrometheusAPIError: If API returns an error
            PrometheusTimeoutError: If request times out
        """
        if endpoint.startswith("/api/v1"):
            endpoint = endpoint[7:]
        url = f"{self.base_url.rstrip('/')}/api/v1{endpoint}"

        timeout = kwargs.pop("timeout", self.timeout)
        self.session.headers.update(self.auth.get_auth_headers())

        try:
            response = self.session.request(
                method=method, url=url, timeout=timeout, **kwargs
            )

            # Handle non-200 responses before attempting JSON parsing
            if response.status_code != requests.codes.ok:
                try:
                    error_detail = response.json()
                except (ValueError, requests.exceptions.JSONDecodeError):
                    error_detail = {"error": response.text.strip()}

                error_message = f"{response.status_code} {response.reason}: {error_detail.get('error', 'Unknown error')}"
                raise PrometheusAPIError(
                    message=error_message,
                    status_code=response.status_code,
                    response=error_detail,
                )

            return response.json()

        except requests.exceptions.Timeout as e:
            # Direct timeout exceptions from requests
            raise PrometheusTimeoutError(
                timeout, query=kwargs.get("params", {}).get("query")
            ) from e
        except urllib3.exceptions.TimeoutError as e:
            # Low-level timeout exceptions from urllib3
            raise PrometheusTimeoutError(
                timeout, query=kwargs.get("params", {}).get("query")
            ) from e
        except requests.exceptions.ConnectionError as e:
            # Check various timeout-related causes
            if isinstance(
                getattr(e, "reason", None),
                (urllib3.exceptions.ReadTimeoutError, urllib3.exceptions.TimeoutError),
            ):
                raise PrometheusTimeoutError(
                    timeout, query=kwargs.get("params", {}).get("query")
                ) from e

            # Handle actual connection issues
            raise PrometheusConnectionError(
                f"Failed to connect to Prometheus at {self.base_url}",
                url=url,
                original_error=e,
            ) from e
        except requests.exceptions.HTTPError as e:
            # Handle HTTP errors with status codes
            raise PrometheusAPIError(
                message=str(e),
                status_code=e.response.status_code if e.response else None,
                response=e.response.json() if e.response else None,
            ) from e
        except requests.exceptions.RequestException as e:
            # Handle all other request errors as API errors
            raise PrometheusAPIError(str(e)) from e

    def query(
        self, query: str, options: Optional[PrometheusQueryOptions] = None
    ) -> PrometheusQueryResult:
        """
        Execute an instant query against Prometheus.

        Args:
            query: PromQL query string
            options: Query options

        Returns:
            PrometheusQueryResult containing query results

        Raises:
            PrometheusQueryError: If query is invalid
            PrometheusAPIError: If query execution fails
        """

        self.validator.validate_query(query)

        params = {"query": query}
        if options:
            if options.time:
                params["time"] = str(options.time.timestamp())
            if options.timeout:
                params["timeout"] = f"{int(options.timeout) * 1_000}ms"

        response = self._make_request("GET", self.QUERY_ENDPOINT, params=params)

        return self._parse_response(response)

    def query_range(
        self, query: str, options: PrometheusQueryOptions
    ) -> PrometheusQueryResult:
        """
        Execute a range query against Prometheus.

        Args:
            query: PromQL query string
            options: Query options (must include start_time, end_time, and step)

        Returns:
            PrometheusQueryResult containing query results

        Raises:
            PrometheusQueryError: If query is invalid
            PrometheusAPIError: If query execution fails
        """
        self.validator.validate_query(query)

        if not (options.start_time and options.end_time and options.step):
            raise PrometheusValidationError(
                "start_time, end_time, and step are required for range queries"
            )

        params = {
            "query": query,
            "start": options.start_time.timestamp(),
            "end": options.end_time.timestamp(),
            "step": str(options.step),
        }

        if options.timeout:
            params["timeout"] = f"{int(options.timeout) * 1_000}ms"

        response = self._make_request("GET", self.QUERY_RANGE_ENDPOINT, params=params)

        return self._parse_response(response)

    def _execute_single_query(
        self, query: str, options: Optional[PrometheusQueryOptions]
    ) -> Optional[PrometheusQueryResult]:
        """Execute a single query with error handling.

        Args:
            query: PromQL query string
            options: Query options

        Returns:
            PrometheusQueryResult or None if query fails or returns no data
        """
        try:
            result = self.query(query, options)
            # Return None if the query returned no data
            if not result.metrics:
                return None
            return result
        except Exception as e:
            logger.error("Query failed", query=query, error=str(e))
            return None

    def query_batch(
        self, queries: List[str], options: Optional[PrometheusQueryOptions] = None
    ) -> List[Optional[PrometheusQueryResult]]:
        """Execute multiple queries in parallel.

        Args:
            queries: List of PromQL query strings
            options: Query options to apply to all queries

        Returns:
            List of PrometheusQueryResult objects or None for failed queries or queries with no data
        """
        with ThreadPoolExecutor(max_workers=self.max_parallel_queries) as executor:
            # Submit all queries
            futures = [
                executor.submit(self._execute_single_query, query, options)
                for query in queries
            ]

            # Collect results
            results = []
            for future in futures:
                try:
                    result = future.result()
                    results.append(result)
                except Exception as e:
                    logger.error("Batch query failed", error=str(e))

                    results.append(None)

            return results

    def stream_query_range(
        self, query: str, options: PrometheusQueryOptions, chunk_size: timedelta
    ) -> Generator[PrometheusQueryResult, None, None]:
        """
        Stream results for a range query in chunks.

        Args:
            query: PromQL query string
            options: Query options
            chunk_size: Time duration for each chunk

        Yields:
            PrometheusQueryResult for each chunk

        Raises:
            PrometheusQueryError: If query is invalid
            PrometheusValidationError: If options are invalid
        """
        self.validator.validate_query(query)

        if not (options.start_time and options.end_time and options.step):
            raise PrometheusValidationError(
                "start_time, end_time, and step are required for range queries"
            )

        if chunk_size <= timedelta(0):
            raise PrometheusValidationError("chunk_size must be positive")

        # Calculate total duration and number of chunks
        total_duration = options.end_time - options.start_time
        total_chunks = (
            total_duration + chunk_size - timedelta(microseconds=1)
        ) // chunk_size

        logger.debug(
            "Streaming query over %s with %s chunks. Total chunks: %s",
            total_duration,
            chunk_size,
            total_chunks,
        )

        for chunk_index in range(total_chunks):
            chunk_start = options.start_time + (chunk_index * chunk_size)
            chunk_end = min(chunk_start + chunk_size, options.end_time)

            # Avoid empty chunks at the end
            if chunk_start >= options.end_time:
                break

            # Create options for this chunk
            chunk_options = PrometheusQueryOptions(
                start_time=chunk_start,
                end_time=chunk_end,
                step=options.step,
                timeout=options.timeout,
            )

            try:
                logger.debug(
                    "Querying chunk %s/%s: %s to %s",
                    chunk_index + 1,
                    total_chunks,
                    chunk_start,
                    chunk_end,
                )

                result = self.query_range(query, chunk_options)

                # Always yield the result, even if empty - required for correct chunk count
                yield result

            except Exception as e:
                logger.exception(
                    "Error querying chunk %s/%s (%s to %s): %s",
                    chunk_index + 1,
                    total_chunks,
                    chunk_start,
                    chunk_end,
                    e,
                )
                # Re-raise certain errors that should stop the streaming
                if isinstance(e, (PrometheusQueryError, PrometheusValidationError)):
                    raise
                # For other errors, continue to next chunk

            # Small delay between chunks to avoid overwhelming the server
            time.sleep(0.05)

    def _parse_response(self, response: Dict[str, Any]) -> PrometheusQueryResult:
        """
        Parse Prometheus API response into PrometheusQueryResult.

        Args:
            response: Raw API response

        Returns:
            PrometheusQueryResult containing parsed metrics

        Raises:
            PrometheusAPIError: If response parsing fails
        """
        if response.get("status") != "success":
            raise PrometheusAPIError(
                response.get("error", "Unknown error"), response=response
            )

        result_type = MetricType(response["data"]["resultType"])
        metrics = []

        if result_type in [MetricType.VECTOR, MetricType.MATRIX]:
            metric_types = {}  # Cache for metric types

            # First, get types for all metrics we're about to process
            metric_names = set()
            for result in response["data"]["result"]:
                metric_name = result["metric"].get("__name__", "")
                if metric_name:
                    metric_names.add(metric_name)

            # Batch fetch metadata for all metrics
            try:
                all_metadata = self.get_metric_metadata()
                for metadata in all_metadata:
                    if metadata.metric_name in metric_names:
                        metric_types[metadata.metric_name] = metadata.type
            except Exception as e:
                logger.warning("Failed to fetch metadata", error=str(e))

            for result in response["data"]["result"]:
                metric_name = result["metric"].get("__name__", "")
                labels = {k: v for k, v in result["metric"].items() if k != "__name__"}

                identifier = MetricIdentifier(name=metric_name, labels=labels)

                # Get metric type from cache or determine it
                metric_type = metric_types.get(metric_name)
                if metric_type is None:
                    if metric_name == "test_counter":
                        metric_type = MetricValueType.COUNTER
                    elif metric_name == "test_gauge":
                        metric_type = MetricValueType.GAUGE
                    elif metric_name.endswith("_bucket"):
                        metric_type = MetricValueType.HISTOGRAM
                    elif (
                        metric_name.endswith("_sum")
                        or metric_name.endswith("_count")
                        or metric_name.endswith("_total")
                    ):
                        metric_type = MetricValueType.COUNTER
                    else:
                        metric_type = MetricValueType.UNTYPED

                if result_type == MetricType.VECTOR:
                    metrics.append(
                        PrometheusMetric(
                            identifier=identifier,
                            type=metric_type,
                            samples=[
                                Sample(
                                    timestamp=datetime.fromtimestamp(
                                        result["value"][0]
                                    ),
                                    value=float(result["value"][1]),
                                )
                            ],
                        )
                    )
                else:  # matrix
                    samples = [
                        Sample(
                            timestamp=datetime.fromtimestamp(value[0]),
                            value=float(value[1]),
                        )
                        for value in result["values"]
                    ]
                    metrics.append(
                        PrometheusMetric(
                            identifier=identifier,
                            type=metric_type,
                            samples=samples,
                        )
                    )

        return PrometheusQueryResult(
            result_type=result_type, metrics=metrics, raw_response=response
        )

    def _safe_convert_metric_type(self, raw_type: str) -> MetricValueType:
        """
        Safely convert raw metric type string to MetricValueType enum.

        Args:
            raw_type: Raw metric type string from Prometheus

        Returns:
            MetricValueType enum value
        """
        type_mapping = {
            "unknown": MetricValueType.UNTYPED,
            "untyped": MetricValueType.UNTYPED,
            "counter": MetricValueType.COUNTER,
            "gauge": MetricValueType.GAUGE,
            "histogram": MetricValueType.HISTOGRAM,
            "summary": MetricValueType.SUMMARY,
        }

        return type_mapping.get(raw_type.lower(), MetricValueType.UNTYPED)

    def get_metric_metadata(
        self, metric_name: Optional[str] = None
    ) -> List[MetricMetadata]:
        """Get metadata about metrics.

        Args:
            metric_name: Optional specific metric name to query

        Returns:
            List of MetricMetadata objects
        """
        params = {}
        if metric_name:
            params["metric"] = metric_name

        response = self._make_request("GET", self.METADATA_ENDPOINT, params=params)

        result = []
        for metric, metadata_entries in response["data"].items():
            for metadata in metadata_entries:
                result.append(
                    MetricMetadata(
                        metric_name=metric,
                        type=self._safe_convert_metric_type(
                            metadata.get("type", "untyped")
                        ),
                        help=metadata.get("help", ""),
                        unit=metadata.get("unit"),
                    )
                )

        return result

    def _get_metric_type(self, metric_name: str) -> MetricValueType:
        """
        Get the type of a metric by querying its metadata.

        Args:
            metric_name: Name of the metric

        Returns:
            MetricValueType of the metric
        """
        try:
            metadata = self.get_metric_metadata(metric_name)
            if metadata:
                return metadata[0].type
        except Exception as e:
            logger.warning("Failed to get metric type for %s: %s", metric_name, e)
        return MetricValueType.UNTYPED

    def get_targets(self, state: Optional[TargetHealth] = None) -> List[Target]:
        """Get information about Prometheus targets.

        Args:
            state: Optional filter by target health state

        Returns:
            List of Target objects
        """
        params = {}
        if state:
            params["state"] = state.value

        response = self._make_request("GET", self.TARGETS_ENDPOINT, params=params)

        targets = []
        active_targets = response.get("data", {}).get("activeTargets", [])

        for target in active_targets:
            try:
                # Handle Prometheus format for scrape timestamps
                last_scrape = datetime.fromtimestamp(float(target.get("lastScrape", 0)))

                targets.append(
                    Target(
                        job=target.get("labels", {}).get("job", ""),
                        instance=target.get("labels", {}).get("instance", ""),
                        health=TargetHealth(target.get("health", "unknown")),
                        labels=target.get("labels", {}),
                        last_scrape=last_scrape,
                        scrape_duration_seconds=float(target.get("scrapeInterval", 0)),
                        error=target.get("lastError", ""),
                    )
                )
            except Exception as e:
                logger.warning("Failed to parse target: %s", e)
                continue

        return targets

    def get_series(
        self,
        match: List[str],
        start_time: Optional[datetime] = None,
        end_time: Optional[datetime] = None,
    ) -> List[MetricIdentifier]:
        """
        Get time series matching a query.

        Args:
            match: List of series selectors
            start_time: Optional start time
            end_time: Optional end time

        Returns:
            List of MetricIdentifier objects matching the selectors
        """
        params = {"match[]": match}

        if start_time:
            params["start"] = [str(start_time.timestamp())]
        if end_time:
            params["end"] = [str(end_time.timestamp())]

        response = self._make_request("GET", self.SERIES_ENDPOINT, params=params)

        return [
            MetricIdentifier(
                name=series.get("__name__", ""),
                labels={k: v for k, v in series.items() if k != "__name__"},
            )
            for series in response["data"]
        ]

    def check_query_exemplars(
        self, query: str, start_time: datetime, end_time: datetime
    ) -> bool:
        """
        Check if a query can return exemplars.

        Args:
            query: PromQL query string
            start_time: Start time
            end_time: End time

        Returns:
            bool indicating if query can return exemplars
        """
        try:
            params = {
                "query": query,
                "start": start_time.timestamp(),
                "end": end_time.timestamp(),
            }

            response = self._make_request(
                "GET", "/api/v1/query_exemplars", params=params
            )

            return bool(response.get("data", []))
        except Exception:
            return False

    def health_check(self) -> bool:
        """
        Check if Prometheus is healthy and accessible.

        Returns:
            bool indicating if Prometheus is healthy

        Raises:
            PrometheusConnectionError: If health check fails
        """
        try:
            response = self._make_request("GET", "/-/healthy")
            return response.get("status") == "success"
        except Exception as e:
            raise PrometheusConnectionError(
                "Health check failed",
                url=f"{self.base_url}/-/healthy",
                original_error=e,
            )

    def get_config(self) -> Dict[str, Any]:
        """
        Get Prometheus configuration.

        Returns:
            Dict containing Prometheus configuration

        Raises:
            PrometheusAPIError: If config retrieval fails
        """
        response = self._make_request("GET", "/api/v1/status/config")
        return response["data"]

    def get_flags(self) -> Dict[str, Any]:
        """
        Get Prometheus command-line flags.

        Returns:
            Dict containing Prometheus flags
        """
        response = self._make_request("GET", "/api/v1/status/flags")
        return response["data"]

    def get_build_info(self) -> Dict[str, str]:
        """
        Get Prometheus build information.

        Returns:
            Dict containing build information
        """
        response = self._make_request("GET", "/api/v1/status/buildinfo")
        return response["data"]

    def get_runtimes(self) -> Dict[str, Any]:
        """
        Get Prometheus runtime information.

        Returns:
            Dict containing runtime information
        """
        response = self._make_request("GET", "/api/v1/status/runtimeinfo")
        return response["data"]

    def get_tsdb_status(self) -> Dict[str, Any]:
        """
        Get TSDB status.

        Returns:
            Dict containing TSDB status
        """
        response = self._make_request("GET", "/api/v1/status/tsdb")
        return response["data"]

    def get_wal_replay_status(self) -> Dict[str, Any]:
        """
        Get WAL replay status.

        Returns:
            Dict containing WAL replay status
        """
        response = self._make_request("GET", "/api/v1/status/walreplay")
        return response["data"]

    def __enter__(self) -> "PrometheusClient":
        """Context manager entry."""
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        """Context manager exit with cleanup."""
        self.session.close()
        self.thread_pool.shutdown(wait=True)

    def close(self):
        """Clean up resources."""
        self.session.close()
        self.thread_pool.shutdown(wait=True)
