"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * @deprecated use `fromBucket`
     */
    static bucket(bucket, key, objectVersion) {
        return this.fromBucket(bucket, key, objectVersion);
    }
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * @deprecated use `fromInline`
     */
    static inline(code) {
        return this.fromInline(code);
    }
    /**
     * Loads the function code from a local disk asset.
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static fromAsset(path, options) {
        return new AssetCode(path, options);
    }
    /**
     * @deprecated use `fromAsset`
     */
    static asset(path) {
        return this.fromAsset(path);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @returns a new instance of `CfnParametersCode`
     * @param props optional construction properties of {@link CfnParametersCode}
     */
    static fromCfnParameters(props) {
        return new CfnParametersCode(props);
    }
    /**
     * @deprecated use `fromCfnParameters`
     */
    static cfnParameters(props) {
        return this.fromCfnParameters(props);
    }
    /**
     * Called after the CFN function resource has been created to allow the code
     * class to bind to it. Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindToResource(_resource, _options) {
        return;
    }
}
exports.Code = Code;
/**
 * Lambda code from an S3 archive.
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion
            }
        };
    }
}
exports.S3Code = S3Code;
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        this.isInline = true;
        if (code.length === 0) {
            throw new Error(`Lambda inline code cannot be empty`);
        }
        if (code.length > 4096) {
            throw new Error("Lambda source is too large, must be <= 4096 but is " + code.length);
        }
    }
    bind(_scope) {
        return {
            inlineCode: this.code
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * Lambda code from a local directory.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        this.isInline = false;
    }
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', Object.assign({ path: this.path }, this.options));
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey
            }
        };
    }
    bindToResource(resource, options = {}) {
        if (!this.asset) {
            throw new Error(`bindToResource() must be called after bind()`);
        }
        const resourceProperty = options.resourceProperty || 'Code';
        // https://github.com/aws/aws-cdk/issues/1432
        this.asset.addResourceMetadata(resource, resourceProperty);
    }
}
exports.AssetCode = AssetCode;
/**
 * Lambda code defined using 2 CloudFormation parameters.
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
class CfnParametersCode extends Code {
    constructor(props = {}) {
        super();
        this.isInline = false;
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    bind(scope) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(scope, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(scope, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
        return {
            s3Location: {
                bucketName: this._bucketNameParam.valueAsString,
                objectKey: this._objectKeyParam.valueAsString,
            }
        };
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code
     */
    assign(location) {
        const ret = {};
        ret[this.bucketNameParam] = location.bucketName;
        ret[this.objectKeyParam] = location.objectKey;
        return ret;
    }
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
//# sourceMappingURL=data:application/json;base64,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