"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const lambda_generated_1 = require("./lambda.generated");
/**
 * Defines a Lambda EventSourceMapping resource.
 *
 * Usually, you won't need to define the mapping yourself. This will usually be done by
 * event sources. For example, to add an SQS event source to a function:
 *
 *    import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
 *    lambda.addEventSource(new SqsEventSource(sqs));
 *
 * The `SqsEventSource` class will automatically create the mapping, and will also
 * modify the Lambda's execution role so it can consume messages from the queue.
 */
class EventSourceMapping extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.maxBatchingWindow && props.maxBatchingWindow.toSeconds() > 300) {
            throw new Error(`maxBatchingWindow cannot be over 300 seconds, got ${props.maxBatchingWindow.toSeconds()}`);
        }
        new lambda_generated_1.CfnEventSourceMapping(this, 'Resource', {
            batchSize: props.batchSize,
            enabled: props.enabled,
            eventSourceArn: props.eventSourceArn,
            functionName: props.target.functionName,
            startingPosition: props.startingPosition,
            maximumBatchingWindowInSeconds: props.maxBatchingWindow && props.maxBatchingWindow.toSeconds(),
        });
    }
}
exports.EventSourceMapping = EventSourceMapping;
/**
 * The position in the DynamoDB or Kinesis stream where AWS Lambda should start
 * reading.
 */
var StartingPosition;
(function (StartingPosition) {
    /**
     * Start reading at the last untrimmed record in the shard in the system,
     * which is the oldest data record in the shard.
     */
    StartingPosition["TRIM_HORIZON"] = "TRIM_HORIZON";
    /**
     * Start reading just after the most recent record in the shard, so that you
     * always read the most recent data in the shard
     */
    StartingPosition["LATEST"] = "LATEST";
})(StartingPosition = exports.StartingPosition || (exports.StartingPosition = {}));
//# sourceMappingURL=data:application/json;base64,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