"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const lambda = require("../lib");
module.exports = {
    'version and aliases'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        assert_1.expect(stack).to(assert_1.beASupersetOfTemplate({
            MyLambdaVersion16CDE3C40: {
                Type: "AWS::Lambda::Version",
                Properties: {
                    FunctionName: { Ref: "MyLambdaCCE802FB" }
                }
            },
            Alias325C5727: {
                Type: "AWS::Lambda::Alias",
                Properties: {
                    FunctionName: { Ref: "MyLambdaCCE802FB" },
                    FunctionVersion: stack.resolve(version.version),
                    Name: "prod"
                }
            }
        }));
        test.done();
    },
    'can create an alias to $LATEST'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'latest',
            version: fn.latestVersion,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
            FunctionName: { Ref: "MyLambdaCCE802FB" },
            FunctionVersion: '$LATEST',
            Name: 'latest',
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Version'));
        test.done();
    },
    'can use newVersion to create a new Version'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('NewVersion');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Version', {
            FunctionName: { Ref: "MyLambdaCCE802FB" },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
            FunctionName: { Ref: "MyLambdaCCE802FB" },
            Name: "prod"
        }));
        test.done();
    },
    'can add additional versions to alias'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version1 = fn.addVersion('1');
        const version2 = fn.addVersion('2');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version: version1,
            additionalVersions: [{ version: version2, weight: 0.1 }]
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
            FunctionVersion: stack.resolve(version1.version),
            RoutingConfig: {
                AdditionalVersionWeights: [
                    {
                        FunctionVersion: stack.resolve(version2.version),
                        FunctionWeight: 0.1
                    }
                ]
            }
        }));
        test.done();
    },
    'sanity checks on version weights'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        // WHEN: Individual weight too high
        test.throws(() => {
            new lambda.Alias(stack, 'Alias1', {
                aliasName: 'prod', version,
                additionalVersions: [{ version, weight: 5 }]
            });
        });
        // WHEN: Sum too high
        test.throws(() => {
            new lambda.Alias(stack, 'Alias2', {
                aliasName: 'prod', version,
                additionalVersions: [{ version, weight: 0.5 }, { version, weight: 0.6 }]
            });
        });
        test.done();
    },
    'metric adds Resource: aliasArn to dimensions'(test) {
        const stack = new core_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // WHEN
        new cloudwatch.Alarm(stack, 'Alarm', {
            metric: alias.metric('Test'),
            alarmName: 'Test',
            threshold: 1,
            evaluationPeriods: 1
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            Dimensions: [{
                    Name: "FunctionName",
                    Value: {
                        Ref: "MyLambdaCCE802FB"
                    }
                }, {
                    Name: "Resource",
                    Value: {
                        'Fn::Join': [
                            '',
                            [
                                { Ref: "MyLambdaCCE802FB" },
                                ':prod'
                            ]
                        ]
                    }
                }]
        }));
        test.done();
    },
    'alias exposes real Lambdas role'(test) {
        const stack = new core_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // THEN
        test.equals(alias.role, fn.role);
        test.done();
    },
    'functionName is derived from the aliasArn so that dependencies are sound'(test) {
        const stack = new core_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // WHEN
        test.deepEqual(stack.resolve(alias.functionName), {
            "Fn::Join": [
                "",
                [
                    {
                        "Fn::Select": [
                            6,
                            {
                                "Fn::Split": [
                                    ":",
                                    {
                                        Ref: "Alias325C5727"
                                    }
                                ]
                            }
                        ]
                    },
                    ":prod"
                ]
            ]
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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