"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const path = require("path");
const lambda = require("../lib");
function newTestLambda(scope) {
    return new lambda.Function(scope, 'MyLambda', {
        code: new lambda.InlineCode('foo'),
        handler: 'bar',
        runtime: lambda.Runtime.PYTHON_2_7
    });
}
module.exports = {
    'default function'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        assert_1.expect(stack).toMatch({
            Resources: {
                MyLambdaServiceRole4539ECB6: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: "lambda.amazonaws.com" }
                                }],
                            Version: '2012-10-17'
                        },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    }
                },
                MyLambdaCCE802FB: {
                    Type: 'AWS::Lambda::Function',
                    Properties: {
                        Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs8.10'
                    },
                    DependsOn: ['MyLambdaServiceRole4539ECB6']
                }
            }
        });
        test.done();
    },
    'adds policy permissions'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            initialPolicy: [new iam.PolicyStatement({ actions: ["*"], resources: ["*"] })]
        });
        assert_1.expect(stack).toMatch({
            Resources: {
                MyLambdaServiceRole4539ECB6: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: "lambda.amazonaws.com" }
                                }],
                            Version: '2012-10-17'
                        },
                        ManagedPolicyArns: 
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    }
                },
                MyLambdaServiceRoleDefaultPolicy5BBC6F68: {
                    Type: "AWS::IAM::Policy",
                    Properties: {
                        PolicyDocument: {
                            Statement: [
                                {
                                    Action: "*",
                                    Effect: "Allow",
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        },
                        PolicyName: "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        Roles: [
                            {
                                Ref: "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                MyLambdaCCE802FB: {
                    Type: 'AWS::Lambda::Function',
                    Properties: {
                        Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs8.10'
                    },
                    DependsOn: ['MyLambdaServiceRoleDefaultPolicy5BBC6F68', 'MyLambdaServiceRole4539ECB6']
                }
            }
        });
        test.done();
    },
    'fails if inline code is used for an invalid runtime'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'bar',
            runtime: lambda.Runtime.DOTNET_CORE_2
        }));
        test.done();
    },
    'addToResourcePolicy': {
        'can be used to add permissions to the Lambda function'(test) {
            const stack = new cdk.Stack();
            const fn = newTestLambda(stack);
            fn.addPermission('S3Permission', {
                action: 'lambda:*',
                principal: new iam.ServicePrincipal('s3.amazonaws.com'),
                sourceAccount: stack.account,
                sourceArn: 'arn:aws:s3:::my_bucket'
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyLambdaServiceRole4539ECB6": {
                        "Type": "AWS::IAM::Role",
                        "Properties": {
                            "AssumeRolePolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sts:AssumeRole",
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": "lambda.amazonaws.com"
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "ManagedPolicyArns": 
                            // tslint:disable-next-line:max-line-length
                            [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                        }
                    },
                    "MyLambdaCCE802FB": {
                        "Type": "AWS::Lambda::Function",
                        "Properties": {
                            "Code": {
                                "ZipFile": "foo"
                            },
                            "Handler": "bar",
                            "Role": {
                                "Fn::GetAtt": [
                                    "MyLambdaServiceRole4539ECB6",
                                    "Arn"
                                ]
                            },
                            "Runtime": "python2.7"
                        },
                        "DependsOn": [
                            "MyLambdaServiceRole4539ECB6"
                        ]
                    },
                    "MyLambdaS3Permission99D0EA08": {
                        "Type": "AWS::Lambda::Permission",
                        "Properties": {
                            "Action": "lambda:*",
                            "FunctionName": {
                                "Fn::GetAtt": [
                                    "MyLambdaCCE802FB",
                                    "Arn"
                                ]
                            },
                            "Principal": "s3.amazonaws.com",
                            "SourceAccount": {
                                "Ref": "AWS::AccountId"
                            },
                            "SourceArn": "arn:aws:s3:::my_bucket"
                        }
                    }
                }
            });
            test.done();
        },
        'fails if the principal is not a service, account or arn principal'(test) {
            const stack = new cdk.Stack();
            const fn = newTestLambda(stack);
            test.throws(() => fn.addPermission('F1', { principal: new iam.OrganizationPrincipal('org') }), /Invalid principal type for Lambda permission statement/);
            fn.addPermission('S1', { principal: new iam.ServicePrincipal('my-service') });
            fn.addPermission('S2', { principal: new iam.AccountPrincipal('account') });
            fn.addPermission('S3', { principal: new iam.ArnPrincipal('my:arn') });
            test.done();
        },
        'BYORole'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const role = new iam.Role(stack, 'SomeRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            role.addToPolicy(new iam.PolicyStatement({ actions: ['confirm:itsthesame'] }));
            // WHEN
            const fn = new lambda.Function(stack, 'Function', {
                code: new lambda.InlineCode('test'),
                runtime: lambda.Runtime.PYTHON_3_6,
                handler: 'index.test',
                role,
                initialPolicy: [
                    new iam.PolicyStatement({ actions: ['inline:inline'] })
                ]
            });
            fn.addToRolePolicy(new iam.PolicyStatement({ actions: ['explicit:explicit'] }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Version": "2012-10-17",
                    "Statement": [
                        { "Action": "confirm:itsthesame", "Effect": "Allow" },
                        { "Action": "inline:inline", "Effect": "Allow" },
                        { "Action": "explicit:explicit", "Effect": "Allow" }
                    ],
                },
            }));
            test.done();
        }
    },
    'fromFunctionArn'(test) {
        // GIVEN
        const stack2 = new cdk.Stack();
        // WHEN
        const imported = lambda.Function.fromFunctionArn(stack2, 'Imported', 'arn:aws:lambda:us-east-1:123456789012:function:ProcessKinesisRecords');
        // Can call addPermission() but it won't do anything
        imported.addPermission('Hello', {
            principal: new iam.ServicePrincipal('harry')
        });
        // THEN
        test.deepEqual(imported.functionArn, 'arn:aws:lambda:us-east-1:123456789012:function:ProcessKinesisRecords');
        test.deepEqual(imported.functionName, 'ProcessKinesisRecords');
        test.done();
    },
    'Lambda code can be read from a local directory via an asset'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
            handler: 'index.handler',
            runtime: lambda.Runtime.PYTHON_3_6
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Code": {
                "S3Bucket": {
                    "Ref": "AssetParameters9678c34eca93259d11f2d714177347afd66c50116e1e08996eff893d3ca81232S3Bucket1354C645"
                },
                "S3Key": {
                    "Fn::Join": ["", [
                            { "Fn::Select": [0, { "Fn::Split": ["||", { "Ref": "AssetParameters9678c34eca93259d11f2d714177347afd66c50116e1e08996eff893d3ca81232S3VersionKey5D873FAC" }] }] },
                            { "Fn::Select": [1, { "Fn::Split": ["||", { "Ref": "AssetParameters9678c34eca93259d11f2d714177347afd66c50116e1e08996eff893d3ca81232S3VersionKey5D873FAC" }] }] },
                        ]]
                }
            },
            "Handler": "index.handler",
            "Role": {
                "Fn::GetAtt": [
                    "MyLambdaServiceRole4539ECB6",
                    "Arn"
                ]
            },
            "Runtime": "python3.6"
        }));
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to true and functionName defined by client'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            functionName: 'OneFunctionToRuleThemAll',
            deadLetterQueueEnabled: true
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": "lambda.amazonaws.com"
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "MyLambdaDeadLetterQueue399EEA2D",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaDeadLetterQueue399EEA2D": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "MessageRetentionPeriod": 1209600
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "MyLambdaDeadLetterQueue399EEA2D",
                                    "Arn"
                                ]
                            }
                        },
                        "FunctionName": "OneFunctionToRuleThemAll"
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to true and functionName not defined by client'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            deadLetterQueueEnabled: true,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": "lambda.amazonaws.com"
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "MyLambdaDeadLetterQueue399EEA2D",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaDeadLetterQueue399EEA2D": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "MessageRetentionPeriod": 1209600
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "MyLambdaDeadLetterQueue399EEA2D",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to false'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            deadLetterQueueEnabled: false,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": "lambda.amazonaws.com"
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10"
                    },
                    "DependsOn": [
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client provides Queue to be used as DLQ'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriod: cdk.Duration.days(14)
        });
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            deadLetterQueue: dlQueue,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": "lambda.amazonaws.com"
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "DeadLetterQueue9F481546",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "DeadLetterQueue9F481546",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'default function with SQS DLQ when client provides Queue to be used as DLQ and deadLetterQueueEnabled set to true'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriod: cdk.Duration.days(14)
        });
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            deadLetterQueueEnabled: true,
            deadLetterQueue: dlQueue,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": "lambda.amazonaws.com"
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "DeadLetterQueue9F481546",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "DeadLetterQueue9F481546",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6",
                    ]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'error when default function with SQS DLQ when client provides Queue to be used as DLQ and deadLetterQueueEnabled set to false'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriod: cdk.Duration.days(14),
        });
        test.throws(() => new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            deadLetterQueueEnabled: false,
            deadLetterQueue: dlQueue,
        }), /deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false/);
        test.done();
    },
    'default function with Active tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            tracing: lambda.Tracing.ACTIVE
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs8.10",
                "TracingConfig": {
                    "Mode": "Active"
                }
            },
            "DependsOn": [
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                "MyLambdaServiceRole4539ECB6",
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default function with PassThrough tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            tracing: lambda.Tracing.PASS_THROUGH
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs8.10",
                "TracingConfig": {
                    "Mode": "PassThrough"
                }
            },
            "DependsOn": [
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                "MyLambdaServiceRole4539ECB6",
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default function with Disabled tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            tracing: lambda.Tracing.DISABLED
        });
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs8.10"
            },
            "DependsOn": [
                "MyLambdaServiceRole4539ECB6"
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'grantInvoke adds iam:InvokeFunction'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.AccountPrincipal('1234'),
        });
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        fn.grantInvoke(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                        Resource: { "Fn::GetAtt": ["Function76856677", "Arn"] }
                    }
                ]
            }
        }));
        test.done();
    },
    'grantInvoke with a service principal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const service = new iam.ServicePrincipal('apigateway.amazonaws.com');
        // WHEN
        fn.grantInvoke(service);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: {
                'Fn::GetAtt': [
                    'Function76856677',
                    'Arn'
                ]
            },
            Principal: 'apigateway.amazonaws.com'
        }));
        test.done();
    },
    'grantInvoke with an account principal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const account = new iam.AccountPrincipal('123456789012');
        // WHEN
        fn.grantInvoke(account);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: {
                'Fn::GetAtt': [
                    'Function76856677',
                    'Arn'
                ]
            },
            Principal: '123456789012'
        }));
        test.done();
    },
    'grantInvoke with an arn principal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const account = new iam.ArnPrincipal('arn:aws:iam::123456789012:role/someRole');
        // WHEN
        fn.grantInvoke(account);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: {
                'Fn::GetAtt': [
                    'Function76856677',
                    'Arn'
                ]
            },
            Principal: 'arn:aws:iam::123456789012:role/someRole'
        }));
        test.done();
    },
    'Can use metricErrors on a lambda Function'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // THEN
        test.deepEqual(stack.resolve(fn.metricErrors()), {
            dimensions: { FunctionName: { Ref: 'Function76856677' } },
            namespace: 'AWS/Lambda',
            metricName: 'Errors',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
        });
        test.done();
    },
    'addEventSource calls bind'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        let bindTarget;
        class EventSourceMock {
            bind(target) {
                bindTarget = target;
            }
        }
        // WHEN
        fn.addEventSource(new EventSourceMock());
        // THEN
        test.same(bindTarget, fn);
        test.done();
    },
    'support inline code for Ruby runtime'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.RUBY_2_5,
        });
        assert_1.expect(stack).toMatch({
            Resources: {
                MyLambdaServiceRole4539ECB6: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: "lambda.amazonaws.com" }
                                }],
                            Version: '2012-10-17'
                        },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    }
                },
                MyLambdaCCE802FB: {
                    Type: 'AWS::Lambda::Function',
                    Properties: {
                        Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'ruby2.5'
                    },
                    DependsOn: ['MyLambdaServiceRole4539ECB6']
                }
            }
        });
        test.done();
    },
    'using an incompatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const layer = lambda.LayerVersion.fromLayerVersionAttributes(stack, 'TestLayer', {
            layerVersionArn: 'arn:aws:...',
            compatibleRuntimes: [lambda.Runtime.NODEJS_8_10],
        });
        // THEN
        test.throws(() => new lambda.Function(stack, 'Function', {
            layers: [layer],
            runtime: lambda.Runtime.NODEJS_6_10,
            code: lambda.Code.fromInline('exports.main = function() { console.log("DONE"); }'),
            handler: 'index.main'
        }), /nodejs6.10 is not in \[nodejs8.10\]/);
        test.done();
    },
    'using more than 5 layers'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const layers = new Array(6).fill(lambda.LayerVersion.fromLayerVersionAttributes(stack, 'TestLayer', {
            layerVersionArn: 'arn:aws:...',
            compatibleRuntimes: [lambda.Runtime.NODEJS_10_X],
        }));
        // THEN
        test.throws(() => new lambda.Function(stack, 'Function', {
            layers,
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('exports.main = function() { console.log("DONE"); }'),
            handler: 'index.main'
        }), /Unable to add layer:/);
        test.done();
    },
    'environment variables are prohibited in China'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
        // WHEN
        test.throws(() => {
            new lambda.Function(stack, 'MyLambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS,
                environment: {
                    SOME: 'Variable'
                }
            });
        }, /Environment variables are not supported/);
        test.done();
    },
    'environment variables work in an unspecified region'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS,
            environment: {
                SOME: 'Variable'
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Environment: {
                Variables: {
                    SOME: "Variable"
                }
            }
        }));
        test.done();
    },
    'support reserved concurrent executions'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS,
            reservedConcurrentExecutions: 10
        });
        assert_1.expect(stack).toMatch({
            Resources: {
                MyLambdaServiceRole4539ECB6: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: "lambda.amazonaws.com" }
                                }],
                            Version: '2012-10-17'
                        },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    }
                },
                MyLambdaCCE802FB: {
                    Type: 'AWS::Lambda::Function',
                    Properties: {
                        Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        ReservedConcurrentExecutions: 10,
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs'
                    },
                    DependsOn: ['MyLambdaServiceRole4539ECB6']
                }
            }
        });
        test.done();
    },
    'its possible to specify event sources upon creation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        let bindCount = 0;
        class EventSource {
            bind(_) {
                bindCount++;
            }
        }
        // WHEN
        new lambda.Function(stack, 'fn', {
            code: lambda.Code.fromInline('boom'),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.bam',
            events: [
                new EventSource(),
                new EventSource(),
            ]
        });
        // THEN
        test.deepEqual(bindCount, 2);
        test.done();
    },
    'Provided Runtime returns the right values'(test) {
        const rt = lambda.Runtime.PROVIDED;
        test.equal(rt.name, 'provided');
        test.equal(rt.supportsInlineCode, false);
        test.done();
    },
    'specify log retention'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS,
            logRetention: logs.RetentionDays.ONE_MONTH
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            'LogGroupName': {
                'Fn::Join': [
                    '',
                    [
                        '/aws/lambda/',
                        {
                            Ref: 'MyLambdaCCE802FB'
                        }
                    ]
                ]
            },
            'RetentionInDays': 30
        }));
        test.done();
    },
    'imported lambda with imported security group and allowAllOutbound set to false'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = lambda.Function.fromFunctionAttributes(stack, 'fn', {
            functionArn: 'arn:aws:lambda:us-east-1:123456789012:function:my-function',
            securityGroup: ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        // WHEN
        fn.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'sg-123456789',
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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