"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const lambda = require("../lib");
/**
 * Use a class as test fixture
 *
 * setUp() will be mapped to the (synchronous) constructor. tearDown(cb) will be called if available.
 */
function classFixture(klass) {
    let fixture;
    const ret = {
        setUp(cb) {
            fixture = new klass();
            cb();
        },
        tearDown(cb) {
            if (fixture.tearDown) {
                fixture.tearDown(cb);
            }
            else {
                cb();
            }
        }
    };
    const testNames = Reflect.ownKeys(klass.prototype).filter(m => m !== 'tearDown' && m !== 'constructor');
    for (const testName of testNames) {
        ret[testName] = (test) => fixture[testName](test);
    }
    return ret;
}
class SomethingConnectable {
    constructor(connections) {
        this.connections = connections;
    }
}
module.exports = {
    'lambda in a VPC': classFixture(class Henk {
        constructor() {
            // GIVEN
            this.app = new cdk.App();
            this.stack = new cdk.Stack(this.app, 'stack');
            this.vpc = new ec2.Vpc(this.stack, 'VPC');
            // WHEN
            this.lambda = new lambda.Function(this.stack, 'Lambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_10_X,
                vpc: this.vpc,
                allowAllOutbound: false
            });
        }
        'has subnet and securitygroup'(test) {
            // THEN
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        { "Fn::GetAtt": ["LambdaSecurityGroupE74659A1", "GroupId"] }
                    ],
                    SubnetIds: [
                        { Ref: "VPCPrivateSubnet1Subnet8BCA10E0" },
                        { Ref: "VPCPrivateSubnet2SubnetCFCDAA7A" },
                    ]
                }
            }));
            test.done();
        }
        'participates in Connections objects'(test) {
            // GIVEN
            const securityGroup = new ec2.SecurityGroup(this.stack, 'SomeSecurityGroup', { vpc: this.vpc });
            const somethingConnectable = new SomethingConnectable(new ec2.Connections({ securityGroups: [securityGroup] }));
            // WHEN
            this.lambda.connections.allowTo(somethingConnectable, ec2.Port.allTcp(), 'Lambda can call connectable');
            // THEN: Lambda can connect to SomeSecurityGroup
            assert_1.expect(this.stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupEgress", {
                GroupId: { "Fn::GetAtt": ["LambdaSecurityGroupE74659A1", "GroupId"] },
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                DestinationSecurityGroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
                FromPort: 0,
                ToPort: 65535
            }));
            // THEN: SomeSecurityGroup accepts connections from Lambda
            assert_1.expect(this.stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupIngress", {
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                FromPort: 0,
                GroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
                SourceSecurityGroupId: { "Fn::GetAtt": ["LambdaSecurityGroupE74659A1", "GroupId"] },
                ToPort: 65535
            }));
            test.done();
        }
        'can still make Connections after export/import'(test) {
            // GIVEN
            const stack2 = new cdk.Stack(this.app, 'stack2');
            const securityGroup = new ec2.SecurityGroup(stack2, 'SomeSecurityGroup', { vpc: this.vpc });
            const somethingConnectable = new SomethingConnectable(new ec2.Connections({ securityGroups: [securityGroup] }));
            // WHEN
            somethingConnectable.connections.allowFrom(this.lambda.connections, ec2.Port.allTcp(), 'Lambda can call connectable');
            // THEN: SomeSecurityGroup accepts connections from Lambda
            assert_1.expect(stack2).to(assert_1.haveResource("AWS::EC2::SecurityGroupEgress", {
                GroupId: {
                    "Fn::ImportValue": "stack:ExportsOutputFnGetAttLambdaSecurityGroupE74659A1GroupId8F3EC6F1"
                },
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                DestinationSecurityGroupId: {
                    "Fn::GetAtt": [
                        "SomeSecurityGroupEF219AD6",
                        "GroupId"
                    ]
                },
                FromPort: 0,
                ToPort: 65535
            }));
            // THEN: Lambda can connect to SomeSecurityGroup
            assert_1.expect(stack2).to(assert_1.haveResource("AWS::EC2::SecurityGroupIngress", {
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                FromPort: 0,
                GroupId: {
                    "Fn::GetAtt": [
                        "SomeSecurityGroupEF219AD6",
                        "GroupId"
                    ]
                },
                SourceSecurityGroupId: {
                    "Fn::ImportValue": "stack:ExportsOutputFnGetAttLambdaSecurityGroupE74659A1GroupId8F3EC6F1"
                },
                ToPort: 65535
            }));
            test.done();
        }
    }),
    'lambda without VPC throws Error upon accessing connections'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const lambdaFn = new lambda.Function(stack, 'Lambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        test.throws(() => {
            lambdaFn.connections.allowToAnyIpv4(ec2.Port.allTcp(), 'Reach for the world Lambda!');
        });
        test.done();
    },
    'picking public subnets is not allowed'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        test.throws(() => {
            new lambda.Function(stack, 'Lambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_10_X,
                vpc,
                vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC }
            });
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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