import subprocess
from typing import Optional
import warnings

import torch
import torch.nn.functional as F

from bitsandbytes.functional import (
    QuantState,
    create_dynamic_map,
    get_4bit_type,
)

try:
    # to support Intel CPU/GPU (XPU) backend
    import intel_extension_for_pytorch as ipex

    ipex_cpu = ipex if ipex._C._has_cpu() else None
    ipex_xpu = ipex if ipex._C._has_xpu() else None
    ipex_cpu_only = ipex._C._has_cpu() and (not ipex._C._has_xpu())
except BaseException:
    ipex_cpu = None
    ipex_xpu = None
    ipex_cpu_only = None


gxx_available = False
try:
    subprocess.run(["g++", "--version"], capture_output=True)  # hide terminal output
    gxx_available = True
except BaseException:
    warnings.warn("g++ not found, torch.compile disabled for CPU/XPU.")


Tensor = torch.Tensor


def _torch_version_prereq(major, minor):
    ver_major = int(torch.__version__.split(".")[0])
    ver_minor = int(torch.__version__.split(".")[1])
    return ver_major * 32 + ver_minor >= major * 32 + minor


def _ipex_cpu_version_prereq(major, minor):
    if ipex_cpu is not None:
        ver_major = ipex_cpu.__version__.split(".")[0]
        ver_minor = ipex_cpu.__version__.split(".")[1]
        return int(ver_major) * 32 + int(ver_minor) >= major * 32 + minor
    return False


def _ipex_xpu_version_prereq(major, minor):
    if ipex_xpu is not None:
        ver_major = ipex_xpu.__version__.split(".")[0]
        ver_minor = ipex_xpu.__version__.split(".")[1]
        return int(ver_major) * 32 + int(ver_minor) >= major * 32 + minor
    return False


def _maybe_torch_compile(func):
    # torch.compile requires g++ and pytorch >= 2.0
    if gxx_available and _torch_version_prereq(2, 0) and not ipex_xpu:
        options = {}
        # fx_graph_cache requires pytorch >= 2.2
        if _torch_version_prereq(2, 2):
            options.update({"fx_graph_cache": True})
        return torch.compile(func, dynamic=True, options=options)
    return func


def reverse_4bit_compress_format(weight):
    out_1 = torch.empty(weight.size(0), dtype=torch.int32, device=weight.device)
    out_2 = torch.empty(weight.size(0), dtype=torch.int32, device=weight.device)
    out_1 = (weight & 0xF0) >> 4
    out_2 = (weight & 0xF) << 4
    out = out_1 | out_2
    return out


@_maybe_torch_compile
def double_quant_impl(A, col_stats=None, row_stats=None, out_col=None, out_row=None, threshold=0.0):
    """
    Find absolute max values of each row/column of a tensor, and symmetrically quantize it to int8.
    If threshold > 0.0, only values <= threshold are counted. All outliers are zeroed out in
    the original tensor and they are kept in COO format: (rows, cols, values)
    If threshold == 0.0, there are no outliers.
    Args:
        A The tensor to be analyzed and quantized.
        col_stats Absolute max values of each column of A. If it is not None, use the values directly.
            Otherwise, find the values.
        row_stats Absolute max values of each row of A. If it is not None, use the values directly.
            Otherwise, find the values.
        out_col Output buffer for the result quantized per column if it is not None
        out_row Output buffer for the result quantized per row if it is not None
        threshold The threshold for finding outliers if it is > 0.0. Otherwise it has no effect.
    Return:
        A tuple of output quantized per row, output quantized per column, absolute max values of
        each row of A, absolute max values of each column of A, outliers in COO format
    """

    cols = A.shape[-1]
    if len(A.shape) == 3:
        rows = A.shape[0] * A.shape[1]
    else:
        assert A.dim() == 2, f"double_quant: Input tensor should be 2d or 3d but got {A.dim()}d"
        rows = A.shape[0]
    A = A.reshape(rows, cols)

    def get_row_col_stats(A):
        row_stats = torch.max(torch.abs(A), 1).values  # absolute max of each row
        col_stats = torch.max(torch.abs(A), 0).values  # absolute max of each col
        return row_stats, col_stats

    def quant_to_int8(A, stats):
        return torch.clamp(torch.round(A * (127.0 / stats)), -128, 127).to(torch.int8)

    if threshold == 0.0:
        if row_stats is None or col_stats is None:
            row_stats, col_stats = get_row_col_stats(A)
        outlier_cols = None
    else:
        outlier_indices = torch.abs(A) >= threshold  # find outliers
        outlier_cols = torch.argwhere(outlier_indices.any(dim=0)).view(-1)
        outlier_values = A[outlier_indices].clone()

        # outlier_indices = torch.abs(A) >= threshold  # find outliers
        # outlier_coord = outlier_indices.nonzero()  # get outlier coordinates
        # outlier_rows = outlier_coord[:, 0]  # outlier row for COO sparse tensor
        # outlier_cols = outlier_coord[:, 1]  # outlier column for COO sparse tensor
        # outlier_values = A[outlier_indices]  # outlier values for COO sparse tensor
        # coo_tensor = COOSparseTensor(
        #     A.shape[0], A.shape[1], outlier_values.numel(), outlier_rows.int(), outlier_cols.int(), outlier_values
        # )
        if row_stats is None or col_stats is None:
            A[outlier_indices] = 0  # zero out outliers
            row_stats, col_stats = get_row_col_stats(A)

    quant_by_row = quant_to_int8(A, row_stats.unsqueeze(-1))
    quant_by_col = quant_to_int8(A, col_stats.unsqueeze(0))

    if outlier_cols is not None:
        A[outlier_indices] = outlier_values  # restore outliers for later use

        if rows > 1:
            # zero out outlier columns for all rows
            quant_by_row[:, outlier_cols] = 0

    if out_row is not None:
        out_row.copy_(quant_by_row)
    else:
        out_row = quant_by_row
    if out_col is not None:
        out_col.copy_(quant_by_col)
    else:
        out_col = quant_by_col
    # Return float stats to align with CUDA impl
    return out_row, out_col, row_stats.float(), col_stats.float(), outlier_cols


def int8_linear_matmul_impl(
    A: torch.Tensor,
    B: torch.Tensor,
    out: Optional[torch.Tensor] = None,
    dtype=torch.int32,
) -> torch.Tensor:
    """
    Do GEMMM computation. Data type: int8 * int8 -> int32.
    Args:
        A Activation of linear, data type is int8
        B Weight of linear, data type is int8
        out Specified output tensor if it is not None
        dtype Data type of output
    Return:
        A tuple of GEMM result in dtype and Sout
    """

    assert A.dtype == torch.int8
    assert B.dtype == torch.int8
    if out is not None:
        assert out.dtype == dtype

    dimsA = A.ndim
    dimsB = B.ndim
    shapeA = A.shape
    shapeB = B.shape
    assert dimsA in [2, 3], "Only two or three dimensional matrices are supported for argument A"
    assert dimsB == 2, "Only two dimensional matrices are supported for argument B"

    if dimsA == 2:
        m = shapeA[0]
    elif dimsA == 3:
        m = shapeA[0] * shapeA[1]
    n = shapeB[0]
    k = shapeA[-1]
    assert shapeA[-1] == shapeB[-1], f"Shapes of A and B do not match, got {shapeA} and {shapeB}"

    # if the tensor is empty, return a transformed empty tensor with the right dimensions
    if shapeA[0] == 0 and dimsA == 2:
        return torch.empty((0, n), device=A.device, dtype=A.dtype)
    elif shapeA[1] == 0 and dimsA == 3:
        return torch.empty(tuple(shapeA[:2] + [n]), device=A.device, dtype=A.dtype)

    A_reshaped = A.reshape(m, k)

    # torch._int_mm is available on CPU since torch 2.4, XPU since torch 2.6
    if (
        A.device.type == "cpu" and _torch_version_prereq(2, 4)
        # or (A.device.type == "xpu" and _torch_version_prereq(2, 6)
    ):
        C = torch._int_mm(A_reshaped, B.T).to(dtype)
    else:
        C = torch.matmul(A_reshaped.float(), B.t().float()).to(dtype)
    if C.ndim != dimsA:
        assert dimsA == 3
        shapeOut = (shapeA[0], m // shapeA[0], C.shape[-1])
        C = C.reshape(shapeOut)
    if out is not None:
        out.copy_(C)
    else:
        out = C

    return out


@_maybe_torch_compile
def int8_mm_dequant_impl(
    A: torch.Tensor,
    row_stats: torch.Tensor,
    col_stats: torch.Tensor,
    out: Optional[torch.Tensor] = None,
    bias: Optional[torch.Tensor] = None,
    compute_dtype=torch.float32,
    output_dtype=torch.float32,
) -> torch.Tensor:
    """
    Dequant and add bias
    out = A_int32 * (abs_max_A * abs_max_B) / 127 * 127 + bias
    Args:
        A The output of int8 gemm, whose dtype is int32
        row_stats Absolute max value of each row of input (A) of gemm
        col_stats Absolute max value of each row of weight (B) of gemm
        out Output buffer
        bias Bias of linear
        compute_dtype Data type for computation
        output_dtype Data type for output
    Return:
        The result
    """
    assert A.dtype == torch.int32
    out_shape = A.shape
    if len(out_shape) == 3:
        out_shape = (out_shape[0] * out_shape[1], out_shape[2])

    if compute_dtype not in [torch.float32, torch.bfloat16]:
        warnings.warn(
            f"mm_dequant_{A.device}: compute_dtype {compute_dtype} is not supported, will use bfloat16 instead"
        )
        compute_dtype = torch.bfloat16
    A_reshaped = A.reshape(out_shape).to(compute_dtype)
    row_stats = row_stats.reshape(-1).unsqueeze(-1).to(compute_dtype)
    col_stats = col_stats.reshape(-1).unsqueeze(0).to(compute_dtype)
    out = A_reshaped * row_stats * col_stats / (127 * 127)
    if bias is not None:
        out = out + bias.to(compute_dtype)
    out = out.to(output_dtype)
    return out


NF4_QUANT_TABLE = [
    -1.0 - 1e-2,  # 0b0000
    -0.8480964004993439,  # 0b0001
    -0.6106329262256622,  # 0b0010
    -0.4599952697753906,  # 0b0011
    -0.33967943489551544,  # 0b0100
    -0.23460740596055984,  # 0b0101
    -0.13791173323988914,  # 0b0110
    -0.045525018125772476,  # 0b0111
    0.03979014977812767,  # 0b1000
    0.1202552504837513,  # 0b1001
    0.2035212516784668,  # 0b1010
    0.2920137718319893,  # 0b1011
    0.3893125355243683,  # 0b1100
    0.5016634166240692,  # 0b1101
    0.6427869200706482,  # 0b1110
    0.8614784181118011,  # 0b1111
]


FP4_QUANT_TABLE = {
    0 - 1e-2: 0,  # 0b0000
    0.00260417: 1,  # 0b0001
    0.0859375: 6,  # 0b0110
    0.20833333: 7,  # 0b0111
    0.29166667: 4,  # 0b0100
    0.4166667: 5,  # 0b0101
    0.583333: 2,  # 0b0010
    0.8333333: 3,  # 0b0011
}

INT8_QUANT_TABLE = create_dynamic_map().tolist()


def quantize_4bit_impl(
    A: Tensor,
    absmax: Tensor = None,
    out: Tensor = None,
    blocksize=64,
    compress_statistics=False,
    quant_type="nf4",
    quant_storage=torch.uint8,
) -> Tensor:
    """
    Quantize tensor A in blocks of 4-bit values.

    Quantizes tensor A by dividing it into blocks which are independently quantized to FP4.

    Parameters
    ----------
    A : torch.Tensor
        The input tensor.
    absmax : torch.Tensor
        The absmax values.
    out : torch.Tensor
        The output tensor (8-bit).
    blocksize : int
        The blocksize used in quantization.
    quant_type : str
        The 4-bit quantization data type {fp4, nf4}, only nf4 is supported now
    quant_storage: torch.dtype
        We can use bytes to convert storage type.

    Returns
    -------
    torch.Tensor:
        The 8-bit tensor with packed 4-bit values.
    tuple(torch.Tensor, torch.Size, torch.dtype, int):
        The quantization state to undo the quantization.
    """
    if quant_type not in ["nf4", "fp4", "int8"]:
        raise NotImplementedError(f"4-bit quantization data type {quant_type} is not implemented for CPU/XPU.")
    if quant_type == "fp4":
        warnings.warn("fp4 quantization is currently slow on CPU/XPU. Please Use nf4 instead for better performance.")
    assert blocksize in [4096, 2048, 1024, 512, 256, 128, 64]
    n = A.numel()
    input_shape = A.shape
    blocks = n // blocksize
    blocks += 1 if n % blocksize > 0 else 0

    if absmax is None:
        absmax = torch.zeros((blocks,), device=A.device, dtype=A.dtype)

    if out is None:
        out = torch.zeros(((n + 1) // 2), dtype=torch.uint8, device=A.device)

    rem = n % blocksize
    has_rem = rem > 0

    # Scale tensor to [-1, 1]
    A_reshaped = A.reshape(n)
    A_com = A_reshaped[: n - rem]
    A_com_reshaped = A_com.reshape(n // blocksize, blocksize)
    absmax[: blocks - has_rem] = torch.abs(A_com_reshaped).max(dim=-1)[0]
    scaled_A = torch.clamp(A_com_reshaped * (1 / absmax[: blocks - has_rem].view(-1, 1)), -1, 1)
    scaled_A = scaled_A.reshape(-1)
    if has_rem:
        absmax[-1] = torch.abs(A_reshaped[n - rem :]).max()
        scaled_A_rem = torch.clamp(A_reshaped[n - rem :] * (1 / absmax[-1]), -1, 1)
        scaled_A = torch.cat([scaled_A, scaled_A_rem], dim=0)
    # map [-1, 1] to nf4/fp4
    out_uint8 = torch.empty(scaled_A.shape, dtype=torch.uint8, device=A.device)
    if quant_type == "nf4":
        for i in range(len(NF4_QUANT_TABLE)):
            out_uint8[scaled_A > NF4_QUANT_TABLE[i]] = i
    elif quant_type == "fp4":
        sign = scaled_A < 0
        abs_scaled_A = torch.abs(scaled_A)
        for key, val in FP4_QUANT_TABLE.items():
            out_uint8[abs_scaled_A > key] = val
        out_uint8 += sign.to(torch.uint8) * 8
    elif quant_type == "int8":
        map = torch.tensor(INT8_QUANT_TABLE, device=scaled_A.device)
        diff = torch.abs(scaled_A.unsqueeze(-1) - map)
        out_uint8 = torch.argmin(diff, dim=-1).to(torch.uint8).to(scaled_A.device)

    if quant_type == "int8":
        out = out_uint8
        code = torch.Tensor(INT8_QUANT_TABLE).to(A.device)
    else:
        if out_uint8.size(-1) % 2:
            out_uint8 = torch.nn.functional.pad(out_uint8, (0, 1), value=0)
        out[:] = out_uint8[::2].bitwise_left_shift(4).bitwise_or_(out_uint8[1::2])
        code = get_4bit_type(quant_type, device=A.device)

    if compress_statistics:
        offset = absmax.mean()
        absmax -= offset
        qabsmax, state2 = quantize_4bit_impl(absmax, blocksize=256, quant_type="int8")
        del absmax
        state = QuantState(
            absmax=qabsmax,
            shape=input_shape,
            dtype=A.dtype,
            blocksize=blocksize,
            code=code,
            quant_type=quant_type,
            offset=offset,
            state2=state2,
        )
    else:
        state = QuantState(
            absmax=absmax,
            shape=input_shape,
            dtype=A.dtype,
            blocksize=blocksize,
            code=code,
            quant_type=quant_type,
        )

    if quant_storage != torch.uint8:
        bytes_value = out.cpu().numpy().tobytes()
        out = torch.frombuffer(bytes_value, dtype=quant_storage).to(A.device)

    return out.reshape(-1, 1), state


def dequant_8bit(A, offset, quant_state):
    assert A.dtype == torch.uint8
    absmax = quant_state.code[A.reshape(-1).int()]
    blocks = absmax.shape[-1] // 256
    res = absmax.shape[-1] % 256
    if res != 0:
        absmax = F.pad(absmax, (0, 256 - res), mode="constant", value=0)
    absmax = (absmax.view(-1, 256) * quant_state.absmax.view(-1, 1)).to(quant_state.dtype).reshape(-1)
    absmax = absmax[: blocks * 256 + res]
    absmax = absmax.reshape(A.shape)
    absmax += offset
    return absmax


# Compile will fail in torch.frombuffer
# @_maybe_torch_compile
def dequantize_4bit_impl(
    A: Tensor,
    quant_state=None,
    absmax: Tensor = None,
    out: Tensor = None,
    blocksize: int = 64,
    quant_type="nf4",
) -> Tensor:
    """
    Dequantizes 4-bit blockwise quantized values.
    Dequantizes the tensor A with maximum absolute values absmax in blocks of size blocksize.

    Parameters
    ----------
    A : torch.Tensor
        The input 8-bit tensor (packed 4-bit values).
    quant_state : QuantState
        object with quantisation stats, incl. absmax values, original tensor shape and original dtype.
    absmax : torch.Tensor
        The absmax values.
    out : torch.Tensor
        Dequantized output tensor.
    blocksize : int
        The blocksize used in quantization.
    quant_type : str
        The 4-bit quantization data type {fp4, nf4}

    Returns
    -------
    torch.Tensor:
        Dequantized tensor.
    """
    transpose = True if A.shape[0] == 1 else False
    A = A.reshape(-1)
    device = A.device
    if A.dtype != torch.uint8:
        bytes_value = A.cpu().numpy().tobytes()
        A = torch.frombuffer(bytes_value, dtype=torch.uint8).to(device)

    if quant_state is None:
        assert absmax is not None and out is not None

        quant_state = QuantState(
            absmax=absmax,
            shape=out.shape,
            dtype=out.dtype,
            blocksize=blocksize,
            quant_type=quant_type,
        )

    else:
        absmax = quant_state.absmax

    if quant_type not in ["nf4", "fp4"]:
        raise NotImplementedError(
            f"4-bit quantization data type {quant_state.quant_type} is not implemented for CPU/XPU."
        )

    if quant_state.nested:
        absmax = dequant_8bit(absmax, quant_state.offset, quant_state.state2)

    if ipex_cpu_only and _ipex_cpu_version_prereq(2, 5) and getattr(quant_state, "ipex", False):
        ipex_weight = torch.ops.ipex_prepack.woq_linear_unpack_weight(A, "nf4", quant_state.shape, 2)
        A = reverse_4bit_compress_format(ipex_weight)
        quant_state.ipex = False

    # Map nf4 to [-1, 1]
    out_dq = torch.empty(A.size(0) * 2, dtype=torch.int32, device=A.device)
    n = out_dq.numel()
    out_dq[1::2] = A & 0xF
    out_dq[::2] = A >> 4
    # quant_state.code is fp32, cast to quant_state dtype to avoid the mismatch issue
    quant_state.code = quant_state.code.to(quant_state.dtype)
    out_dq = quant_state.code[out_dq]

    # Apply scales
    if out_dq.numel() != n:
        assert out_dq.numel() == n + 1
        out_dq = torch.narrow(out_dq, 0, 0, n)
    blocks = n // blocksize
    blocks += 1 if n % blocksize > 0 else 0
    rem = n % blocksize
    has_rem = rem > 0

    if has_rem:
        if out is None:
            out = torch.empty(quant_state.shape, dtype=quant_state.dtype, device=A.device)
        out_reshaped = out.reshape(-1)
        out_reshaped[: n - rem] = (
            out_dq[: n - rem].view(-1, blocksize) * absmax[: blocks - has_rem].view(-1, 1)
        ).reshape(-1)
        out_reshaped[n - rem :] = out_dq[n - rem :] * absmax[-1]
    else:
        out = (out_dq.view(-1, blocksize) * absmax.view(-1, 1)).reshape(quant_state.shape).to(quant_state.dtype)

    # take transpose here because weight is transposed (again) for computation
    if transpose:
        out = out.t()

    return out


# Do not need torch.compile here as we are calling torch/ipex kernel
def gemm_4bit_impl(
    A: torch.Tensor,
    B: torch.Tensor,
    out: Optional[torch.Tensor] = None,
    transposed_A=False,
    transposed_B=False,
    state: QuantState = None,
) -> torch.Tensor:
    """
    Matrix-matrix multiplication with 4-bit quantization.

    Parameters
    ----------
    A : torch.Tensor
        The first input tensor. Usually the activation tensor.
    B : torch.Tensor
        The second input tensor. Usually the weight tensor.
    out : torch.Tensor
        The output tensor.
    transposed_A : bool
        Whether A is transposed
    transposed_B : bool
        Whether B is transposed
    state : QuantState
        Contains quantization info, such as blocksize and dtype

    Returns
    -------
    torch.Tensor:
        GEMM output tensor.
    """
    if getattr(state, "ipex", False):
        # compute_dtype: 1 indicates fp16, 2 indicates bf16
        compute_dtype = 2 if A.dtype == torch.bfloat16 else 1
        output = torch.ops.torch_ipex.woq_linear(
            A,
            B,
            "nf4",
            state.shape,
            state.new_scales,
            state.new_zeros,
            None,
            None,
            state.blocksize,
            compute_dtype,
            1,
            state.compensation,
        )
    else:
        dqB = dequantize_4bit_impl(B, state, blocksize=state.blocksize)
        output = torch.matmul(A, dqB.to(A.dtype))
    if out is not None:
        out.copy_(output)
    else:
        out = output
    return out
