# -*- coding: utf-8 -*-
r"""
Database of simplicial hyperplane arrangements of rank 3 [CEL]_, [Gru]_.

This database contains three dictionaries: db_normals_CEL, db_invariants_CEL,
and db_wiring_CEL. All three dictionaries have keys following the
labeling in the article [CEL]_. Namely a key is a 3-tuple (A,B,C) where 
A is the number of hyperplanes, B is the number of regions, and C is an further
index for identification.

The dictionary db_normals_CEL has values which are tuples of 3-tuples. 
Each 3-tuple gives a normal to the hyperplane arrangement specified by the key.

The dictionary db_invariants_CEL has values which are 7-tuples of the invariants 
of the hyperplane arrangement specified by the key.
The entries are as follows, see Table 5 in [CEL]_ for more details:

    - The first entry is the labeling of Gruenbaum
    - The second entry is the f-vector
    - The third entry is the t-vector
    - The fourth entry is the r-vector
    - The fifth entry is the automorphism group of the CW-complex
    - The sixth entry is the automorphism group of the matroid
    - The seventh entry is the list of roots of the characteristic polynomial, when integers

The dictionary db_wiring_CEL has values which are tuples describing the wiring
diagram of the hyperplane arrangement specified by the key.
The first entry is the permutation and the second is the list of inversions 
with labels given by the order of the normals in the db_normals_CEL dictionary.
See Table 15 [CEL]_ for more details.

EXAMPLES:

Here is how to access an element in the database. To get the data of the 
normals, invariants, and wiring for the arrangement `A(7,32)_1`, 
in the notation of [CEL]_ we do as follows::

    sage: from cn_hyperarr import *
    sage: normals = db_normals_CEL[(7,32,1)]; normals
    ((0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (1, 0, 1), (1, 1, 0), (1, 1, 1))
    sage: db_invariants_CEL[(7,32,1)]
    ((7, 1), (9, 24, 16), (3, 6), (0, 4, 3), (48, 48), (24, 12), (1, 3, 3))
    sage: db_wiring_CEL[(7,32,1)]
    ((1, 6, 7, 2, 5, 3, 4), ((1, 3), (5, 6), (3, 5), (5, 7), (2, 3), (3, 5), (5, 6), (1, 3), (3, 5)))

To create the hyperplane arrangement, we do as follows::

    sage: vc = VectorConfiguration([vector(x) for x in normals]);vc
    Vector configuration of 7 vectors in dimension 3
    sage: ha = vectorconf_to_hyperplane_arrangement(vc); ha
    Arrangement of 7 hyperplanes of dimension 3 and rank 3
"""

##############################################################################
#     Copyright (C) 2020 Jean-Philippe Labbe < labbe at math.fu-berlin.de>
#                   2020 Sophia Elia         < sophiae56 at math.fu-berlin.de>
#                   2020 Michael Cuntz       < cuntz at math.uni-hannover.de>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#                  http://www.gnu.org/licenses/
##############################################################################

from sage.rings.rational_field import QQ
from sage.rings.number_field.number_field import NumberField, CyclotomicField
from sage.calculus.var import var
from sage.rings.qqbar import AA

# First create the necessary number fields elements

x = var('x')
A = NumberField(x**2-x-1, 'tau', embedding = 16/QQ(10)); tau = A.gens()[0]
A = NumberField(x**2-2, 'sqrt2', embedding = 14/QQ(10)); sqrt2 = A.gens()[0]
A = NumberField(x**2-3, 'sqrt3', embedding = 17/QQ(10)); sqrt3 = A.gens()[0]
A = NumberField(x**2-5, 'sqrt5', embedding = 223/QQ(100)); sqrt5 = A.gens()[0]

z = CyclotomicField(1).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r1', embedding = AA(z+z**-1)); r1 = A.gens()[0]
z = CyclotomicField(2).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r2', embedding = AA(z+z**-1)); r2 = A.gens()[0]
z = CyclotomicField(3).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r3', embedding = AA(z+z**-1)); r3 = A.gens()[0]
z = CyclotomicField(4).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r4', embedding = AA(z+z**-1)); r4 = A.gens()[0]
z = CyclotomicField(5).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r5', embedding = AA(z+z**-1)); r5 = A.gens()[0]
z = CyclotomicField(6).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r6', embedding = AA(z+z**-1)); r6 = A.gens()[0]
z = CyclotomicField(7).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r7', embedding = AA(z+z**-1)); r7 = A.gens()[0]
z = CyclotomicField(8).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r8', embedding = AA(z+z**-1)); r8 = A.gens()[0]
z = CyclotomicField(9).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r9', embedding = AA(z+z**-1)); r9 = A.gens()[0]
z = CyclotomicField(10).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r10', embedding = AA(z+z**-1)); r10 = A.gens()[0]
z = CyclotomicField(11).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r11', embedding = AA(z+z**-1)); r11 = A.gens()[0]
z = CyclotomicField(12).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r12', embedding = AA(z+z**-1)); r12 = A.gens()[0]
z = CyclotomicField(13).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r13', embedding = AA(z+z**-1)); r13 = A.gens()[0]
z = CyclotomicField(14).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r14', embedding = AA(z+z**-1)); r14 = A.gens()[0]
z = CyclotomicField(15).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r15', embedding = AA(z+z**-1)); r15 = A.gens()[0]
z = CyclotomicField(16).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r16', embedding = AA(z+z**-1)); r16 = A.gens()[0]
z = CyclotomicField(17).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r17', embedding = AA(z+z**-1)); r17 = A.gens()[0]
z = CyclotomicField(18).gens()[0];  A = NumberField((z+z**(-1)).minpoly(), 'r18', embedding = AA(z+z**-1)); r18 = A.gens()[0]

A = NumberField(x**3-3*x-25, 'rho', embedding = 32646329/QQ(10000000)); rho = A.gens()[0]
A = NumberField(x**4-3*x**3+3*x**2-3*x+1, 'gamma', embedding = 4643/QQ(10000)); gamma = A.gens()[0]

# This is the list of normals to the hyperplanes
# The key follows the labeling in the article [CEL]_

db_normals_CEL = {}
db_normals_CEL[(6, 24, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (1, 0, 1), (1, 1, 1))
db_normals_CEL[(7, 32, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (1, 0, 1), (1, 1, 0), (1, 1, 1))
db_normals_CEL[(8, 40, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (1, 0, 0), (1, 0, 1), (1, 1, 1), (1, 1, 2))
db_normals_CEL[(9, 48, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (1, 0, 0), (1, 1, 0), (1, 1, 1), (1, 1, 2), (1, 2, 2))
db_normals_CEL[(10, 60, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3))
db_normals_CEL[(10, 60, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (1, 0, 0), (1, 0, 1), (1, 1, 0), (1, 1, 1), (1, 1, 2), (1, 2, 2))
db_normals_CEL[(10, 60, 3)] = ((2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (1, 1, 1), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 1, 0), (1, 0, 0), (tau + 1, tau, tau))
db_normals_CEL[(11, 72, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 3))
db_normals_CEL[(12, 84, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (1, 0, 0), (1, 0, 1), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 3), (1, 2, 4))
db_normals_CEL[(12, 84, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 2), (1, 2, 3))
db_normals_CEL[(12, 84, 3)] = ((1, 1, 0), (1, 1, -1), (2, 2, -3), (1, 1, -2), (1, 1, -3), (2, 4, -3), (1, 2, -2), (1, 3, -1), (1, 3, -3), (0, 0, 1), (0, 2, -1), (0, 1, 0))
db_normals_CEL[(13, 96, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 3), (1, 2, 4))
db_normals_CEL[(13, 96, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 4))
db_normals_CEL[(13, 96, 3)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 2), (1, 2, 3), (1, 2, 4))
db_normals_CEL[(13, 104, 1)] = ((2*tau + 2, tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (2*tau + 2, tau + 2, 1), (3*tau + 1, 2*tau + 1, tau), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 1, 0), (tau + 1, 1, 0), (2*tau + 2, tau + 1, 1), (3*tau + 1, 2*tau, tau))
db_normals_CEL[(14, 112, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4))
db_normals_CEL[(14, 112, 2)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -r7**2 - r7, 0), (1, r7, 0), (1, -r7**2 + 1, 0), (1, -r7**2, 0), (1, 1, -1), (0, 1, -1), (1, -r7**2 - r7 + 1, r7**2 + r7 - 2), (1, -2*r7**2 + 1, r7**2 - 1), (1, -2*r7**2 - 2*r7 + 2, r7**2 + r7 - 2), (0, 1, 3/7*r7**2 + 2/7*r7 - 9/7), (1, r7 + 1, -1))
db_normals_CEL[(14, 112, 3)] = ((tau + 1, tau, 0), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (1, 1, 1), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (tau + 1, tau, tau))
db_normals_CEL[(14, 116, 1)] = ((2*tau + 2, tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (2*tau + 2, tau + 2, 1), (3*tau + 1, 2*tau + 1, tau), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 1, 0), (1, 0, 0), (tau + 1, 1, 0), (2*tau + 2, tau + 1, 1), (3*tau + 1, 2*tau, tau))
db_normals_CEL[(15, 120, 1)] = ((2*tau + 1, tau + 1, tau), (2*tau + 2, tau + 2, tau + 1), (3*tau + 1, 2*tau + 1, tau + 1), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (1, 1, 1), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 1, 0), (1, 0, 0), (tau + 1, 1, 1), (2*tau, tau, tau), (tau + 1, tau, tau))
db_normals_CEL[(15, 128, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4), (1, 2, 5))
db_normals_CEL[(15, 128, 2)] = ((2*tau + 1, tau + 1, 1), (2*tau + 2, tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (2*tau + 2, tau + 2, 1), (3*tau + 1, 2*tau + 1, tau), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 1, 0), (1, 0, 0), (tau + 1, 1, 0), (2*tau + 2, tau + 1, 1), (3*tau + 1, 2*tau, tau))
db_normals_CEL[(15, 132, 1)] = ((0, 0, 1), (0, 1, 1/9*rho**2 - 2/9*rho - 8/9), (1, -1, -1), (1, -1, 0), (1, 0, -1), (1, -1/9*rho**2 - 4/9*rho - 13/9, -1), (1, 1/9*rho**2 + 4/9*rho + 4/9, -1/9*rho**2 - 4/9*rho - 13/9), (1, 0, 1/9*rho**2 - 2/9*rho - 17/9), (1, -1/9*rho**2 - 4/9*rho - 13/9, 1/9*rho**2 + 1/9*rho - 2/9), (1, 1/9*rho**2 + 4/9*rho + 4/9, -1/3*rho - 5/3), (1, 1/9*rho**2 + 4/9*rho + 4/9, -1/3*rho - 2/3), (1, -1/9*rho**2 - 1/9*rho - 7/9, 1/9*rho**2 - 2/9*rho - 8/9), (0, 1, 1/9*rho**2 + 1/9*rho - 20/9), (1, 1/9*rho**2 + 4/9*rho + 4/9, -1/9*rho**2 - 7/9*rho - 19/9), (1, -1/9*rho**2 - 1/9*rho - 7/9, 1/9*rho**2 - 2/9*rho + 1/9))
db_normals_CEL[(15, 132, 2)] = ((1, 1, 3), (1, 2, 4), (1, 1, 2), (1, 0, 0), (1, 2, 3), (2, 3, 4), (1, 1, 1), (1, 3, 4), (1, 2, 2), (1, 1, 0), (1, 2, 1), (0, 1, 2), (0, 1, 3), (0, 1, 0), (0, 1, 1))
db_normals_CEL[(16, 140, 1)] = ((2*tau + 1, tau + 1, tau), (2*tau + 1, tau + 1, 1), (2*tau + 2, tau + 2, tau + 1), (3*tau + 1, 2*tau + 1, tau + 1), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (1, 1, 1), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 1, 0), (1, 0, 0), (tau + 1, 1, 1), (2*tau, tau, tau), (tau + 1, tau, tau))
db_normals_CEL[(16, 144, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4), (1, 2, 5))
db_normals_CEL[(16, 144, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (1, 0, 0), (1, 1, 0), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 1), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 3, 4), (2, 3, 4))
db_normals_CEL[(16, 144, 3)] = ((0, 1, 0), (0, 1, -1), (0, 1, -1/2*sqrt2 - 1), (0, 1, -sqrt2 - 1), (0, 1, -sqrt2 - 2), (0, 1, -2*sqrt2 - 3), (0, 0, 1), (0, 1, sqrt2 + 1), (1, -1, -1), (1, -sqrt2, -1), (1, -1, 0), (1, -2, 0), (1, -sqrt2, sqrt2), (1, -sqrt2 - 1, sqrt2), (1, -2, sqrt2 + 1), (1, -sqrt2 - 1, sqrt2 + 1))
db_normals_CEL[(16, 144, 4)] = ((1, -1, 0), (1, 1, 0), (1, 1, -1), (2, 2, -3), (1, 1, -2), (1, 1, -3), (3, 5, -6), (2, 4, -3), (1, 2, -2), (1, 2, -3), (1, 3, -3), (1, 3, -4), (0, 0, 1), (0, 2, -3), (0, 2, -1), (0, 1, 0))
db_normals_CEL[(16, 144, 5)] = ((2*tau + 1, tau + 1, tau), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (4*tau + 2, 3*tau + 1, tau + 1), (2*tau + 2, tau + 2, 1), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (2*tau, 2*tau, tau), (0, 0, 1), (0, 1, 0), (0, 1, 1), (2*tau, tau, tau), (2*tau + 2, tau + 1, tau), (2*tau + 2, tau + 1, 1))
db_normals_CEL[(16, 148, 1)] = ((1, -1, 1), (1, -1, 0), (1, 1, 0), (1, 1, -1), (2, 2, -3), (1, 1, -2), (1, 1, -3), (2, 4, -3), (1, 2, -2), (1, 3, -1), (1, 3, -3), (1, 3, -4), (0, 0, 1), (0, 2, -3), (0, 2, -1), (0, 1, 0))
db_normals_CEL[(17, 160, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 4), (1, 2, 5), (1, 2, 6))
db_normals_CEL[(17, 160, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4), (1, 2, 5), (2, 2, 5))
db_normals_CEL[(17, 160, 3)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (1, 0, 0), (1, 0, 1), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 3, 5), (2, 3, 5), (2, 3, 6))
db_normals_CEL[(17, 160, 4)] = ((0, 1, 0), (0, 1, -1), (0, 1, -1/2*sqrt2 - 1), (0, 1, -sqrt2 - 1), (0, 1, -sqrt2 - 2), (0, 1, -2*sqrt2 - 3), (0, 0, 1), (0, 1, sqrt2 + 1), (1, -1, -1), (1, -sqrt2, -1), (1, -1, 0), (1, -2, 0), (1, -1/2*sqrt2 - 1, 1/2*sqrt2), (1, -sqrt2, sqrt2), (1, -sqrt2 - 1, sqrt2), (1, -2, sqrt2 + 1), (1, -sqrt2 - 1, sqrt2 + 1))
db_normals_CEL[(17, 160, 5)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (4*tau + 2, 3*tau + 1, tau + 1), (2*tau + 2, tau + 2, 1), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (2*tau, 2*tau, tau), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (2*tau, tau, tau), (2*tau + 2, tau + 1, tau), (2*tau + 2, tau + 1, 1))
db_normals_CEL[(17, 164, 1)] = ((1, -1, 0), (1, 1, 0), (1, 1, -1), (2, 2, -3), (1, 1, -2), (1, 1, -3), (3, 5, -6), (2, 4, -3), (1, 2, -2), (1, 2, -3), (1, 3, -1), (1, 3, -3), (1, 3, -4), (0, 0, 1), (0, 2, -3), (0, 2, -1), (0, 1, 0))
db_normals_CEL[(17, 168, 1)] = ((1, -sqrt2 - 1, 2*sqrt2 + 2), (1, -sqrt2 - 2, 2*sqrt2 + 2), (0, 1, 0), (0, 1, -1), (0, 1, -1/2*sqrt2 - 1), (0, 1, -2*sqrt2 - 3), (1, 0, -sqrt2 - 2), (1, -1, -sqrt2 - 2), (1, -1, -1), (1, -1, 0), (1, -sqrt2 - 1, -1), (1, -2, 0), (1, -1/2*sqrt2 - 1, 1/2*sqrt2), (1, -sqrt2, sqrt2), (1, -1, sqrt2 + 1), (1, -sqrt2 - 1, sqrt2), (1, -sqrt2 - 1, sqrt2 + 1))
db_normals_CEL[(17, 168, 2)] = ((1, -1, 1), (1, -1, 0), (1, 1, 0), (1, 1, -1), (2, 2, -3), (1, 1, -2), (1, 1, -3), (2, 4, -3), (1, 2, -2), (1, 3, -1), (1, 3, -3), (1, 3, -4), (0, 0, 1), (1, 5, -4), (0, 2, -3), (0, 2, -1), (0, 1, 0))
db_normals_CEL[(18, 180, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 3, 5), (1, 3, 6))
db_normals_CEL[(18, 180, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (1, 0, 0), (1, 0, 1), (1, 1, 0), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 1), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 3, 3), (1, 3, 4), (2, 3, 4))
db_normals_CEL[(18, 180, 3)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -2/3*r9**2 - 1/3*r9 + 1/3, 0), (1, -r9**2 - r9, 0), (1, r9, 0), (1, -1/3*r9**2 + 1/3*r9 - 1/3, 0), (1, -r9**2 + 1, 0), (1, -r9**2, 0), (1, 1, -1), (0, 1, -1), (1, r9**2 - 4, -r9**2 + 3), (1, -2*r9**2 + 3, r9**2 - 2), (1, -r9**2 - 2/3*r9 + 4/3, 1/3*r9**2 + 1/3*r9 - 1), (1, -2*r9**2 + 2, r9**2 - 2), (1, -r9 - 3, -r9**2 + 3), (0, 1, 3/19*r9**2 + 2/19*r9 - 14/19), (1, r9 + 1, -1))
db_normals_CEL[(18, 180, 4)] = ((1, 2, 4), (2, 3, 6), (1, 1, 2), (1, 0, 0), (1, 2, 3), (1, 1, 1), (1, 2, 2), (1, 1, 0), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 1, 0), (0, 1, 1), (1, 2, 6), (1, 1, 4), (1, 2, 5), (1, 3, 6), (1, 1, 3))
db_normals_CEL[(18, 180, 5)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (4*tau + 2, 3*tau + 1, tau + 1), (2*tau + 2, tau + 2, 1), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (2*tau, 2*tau, tau), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (2*tau, tau, tau), (2*tau + 2, tau + 1, tau), (2*tau + 2, tau + 1, 1))
db_normals_CEL[(18, 180, 6)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (4*tau + 2, 3*tau + 1, tau + 1), (2*tau + 2, tau + 2, 1), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (1, 1, 1), (2*tau, 2*tau, tau), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (2*tau, tau, tau), (2*tau + 2, tau + 1, tau), (2*tau + 2, tau + 1, 1))
db_normals_CEL[(18, 184, 1)] = ((1, -1, 1), (1, -1, 0), (1, 1, 0), (1, 1, -1), (2, 2, -3), (1, 1, -2), (1, 1, -3), (3, 5, -6), (2, 4, -3), (1, 2, -2), (1, 2, -3), (1, 3, -1), (1, 3, -3), (1, 3, -4), (0, 0, 1), (0, 2, -3), (0, 2, -1), (0, 1, 0))
db_normals_CEL[(18, 184, 2)] = ((1, 2, 4), (1, 0, 1), (1, 2, 3), (1, 3, 4), (2, 3, 4), (1, 0, 0), (1, 1, 1), (1, 2, 2), (1, 3, 3), (1, 1, 0), (1, 2, 1), (0, 2, 3), (0, 1, 2), (0, 1, 3), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 1, 3))
db_normals_CEL[(19, 192, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (1, 0, 0), (1, 1, 0), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 3, 6), (2, 3, 6))
db_normals_CEL[(19, 192, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 1, 0), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 3, 3), (1, 3, 4), (1, 3, 5), (1, 3, 6), (1, 4, 6))
db_normals_CEL[(19, 200, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (1, 0, 0), (1, 0, 1), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 3, 6), (1, 3, 7), (2, 3, 7))
db_normals_CEL[(19, 200, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 1), (1, 0, 0), (1, 0, 1), (1, 1, 0), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 1), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 3, 3), (1, 3, 4), (2, 3, 4))
db_normals_CEL[(19, 200, 3)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 1, 0), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 2, 1), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 3, 3), (1, 3, 4), (2, 3, 4))
db_normals_CEL[(19, 200, 4)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (4*tau + 2, 3*tau + 1, tau + 1), (2*tau + 2, tau + 2, 1), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (1, 1, 1), (2*tau, 2*tau, tau), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (2*tau, tau, tau), (2*tau + 2, tau + 1, tau), (2*tau + 2, tau + 1, 1))
db_normals_CEL[(19, 204, 1)] = ((0, 2, 1), (0, 0, 1), (0, 1, 2), (1, 3, 3), (1, 2, 2), (1, 0, 0), (2, 3, 4), (1, 3, 4), (2, 2, 3), (1, 2, 3), (2, 3, 5), (1, 1, 2), (1, 0, 1), (2, 4, 7), (2, 3, 6), (1, 3, 5), (1, 1, 3), (1, 0, 2), (0, 1, 1))
db_normals_CEL[(20, 216, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 3, 6), (1, 3, 7))
db_normals_CEL[(20, 220, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 3, 6), (1, 3, 7), (2, 3, 7))
db_normals_CEL[(20, 220, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 3, 4), (1, 3, 5), (2, 3, 6))
db_normals_CEL[(20, 220, 3)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -tau, 0), (1, -tau - 1, 0), (1, -tau + 1, 0), (1, -1/2*tau - 1/2, 0), (1, -2, 0), (1, -2*tau - 1, 0), (1, tau, 0), (1, 1, -1), (0, 1, -1), (1, -tau, tau - 1), (1, -tau, 1/2*tau - 1/2), (1, -3*tau + 3, 2*tau - 3), (1, -2*tau + 1, 2*tau - 3), (1, -3/2*tau - 1/2, 1/2*tau - 1/2), (1, -3*tau, tau - 1), (0, 1, 1/2*tau - 1), (1, tau + 1, -1))
db_normals_CEL[(20, 220, 4)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 1, tau + 1, 1), (2*tau + 2, tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (4*tau + 2, 3*tau + 1, tau + 1), (2*tau + 2, tau + 2, 1), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (1, 1, 1), (2*tau, 2*tau, tau), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (2*tau, tau, tau), (2*tau + 2, tau + 1, tau), (2*tau + 2, tau + 1, 1))
db_normals_CEL[(21, 240, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 3, 6), (1, 3, 7), (2, 3, 7))
db_normals_CEL[(21, 240, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 3, 4), (1, 3, 5), (1, 3, 6), (2, 3, 6))
db_normals_CEL[(21, 240, 3)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 3, 5), (1, 3, 6), (2, 2, 5), (2, 3, 6), (2, 3, 7))
db_normals_CEL[(21, 240, 4)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -tau, 0), (1, -tau - 1, 0), (1, -tau + 1, 0), (1, -1/2*tau - 1/2, 0), (1, -2, 0), (1, -2*tau - 1, 0), (1, tau, 0), (1, 1, -1), (0, 1, -1), (1, -tau, tau - 1), (1, -tau, 1/2*tau - 1/2), (1, -3*tau + 3, 2*tau - 3), (1, -2*tau + 1, 2*tau - 3), (1, -3/2*tau - 1/2, 1/2*tau - 1/2), (1, -3*tau, tau - 1), (0, 1, 1/2*tau - 1), (1, tau + 1, -1), (1, -2*tau, tau - 1))
db_normals_CEL[(21, 240, 5)] = ((2*tau, tau, 0), (2*tau + 1, tau + 1, tau), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 1, tau + 1, 1), (2*tau + 2, tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (4*tau + 2, 3*tau + 1, tau + 1), (2*tau + 2, tau + 2, 1), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (1, 1, 1), (2*tau, 2*tau, tau), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (2*tau, tau, tau), (2*tau + 2, tau + 1, tau), (2*tau + 2, tau + 1, 1))
db_normals_CEL[(21, 248, 1)] = ((0, 0, 1), (0, 1, 0), (0, 2, -1), (1, -1, 0), (1, -1, -2), (1, 0, -1), (1, 3, -4), (1, 3, -1), (1, 1, -1), (1, 1, -2), (1, 1, 0), (1, 2, -2), (0, 2, -3), (1, -1, 1), (1, 2, -3), (1, 1, -3), (1, 3, -3), (3, 5, -6), (1, 5, -4), (2, 4, -3), (2, 2, -3))
db_normals_CEL[(21, 252, 1)] = ((0, 0, 1), (0, 1, -1), (1, -1, -1), (1, -1, 1/3*rho - 1/3), (1, 0, -1), (1, 1/9*rho**2 + 1/9*rho - 11/9, -1), (0, 1, -1/3*rho - 2/3), (1, -1, 1/9*rho**2 + 1/9*rho - 11/9), (1, 1/3*rho - 1/3, -1), (0, 1, -1/3*rho + 1/3), (1, 1/3*rho - 4/3, -1), (1, 1/3*rho - 1/3, -1/3*rho - 2/3), (1, 1/3*rho - 1/3, -1/9*rho**2 + 2/9*rho - 10/9), (1, 1/9*rho**2 + 4/9*rho - 5/9, -1/9*rho**2 - 1/9*rho - 16/9), (1, 0, 1/9*rho**2 - 2/9*rho - 8/9), (1, 1/9*rho**2 + 4/9*rho + 4/9, -1/9*rho**2 - 4/9*rho - 22/9), (1, -1/3*rho - 5/3, 1/9*rho**2 + 4/9*rho - 5/9), (1, 1/9*rho**2 + 1/9*rho - 2/9, -2), (1, 1/9*rho**2 + 4/9*rho + 4/9, -1/9*rho**2 - 4/9*rho - 13/9), (1, 1/9*rho**2 + 4/9*rho - 5/9, -1/9*rho**2 - 1/9*rho - 7/9), (1, 1/3*rho - 4/3, -1/9*rho**2 + 2/9*rho - 1/9))
db_normals_CEL[(22, 264, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 3, 4), (1, 3, 5), (2, 2, 5), (2, 3, 5), (2, 3, 6))
db_normals_CEL[(22, 264, 2)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -r11**3 - r11**2 + 3*r11 + 1, 0), (1, -r11**4 + 3*r11**2 - r11 - 1, 0), (1, -r11**2 - r11, 0), (1, r11, 0), (1, r11**4 - 4*r11**2 + r11 + 1, 0), (1, r11**3 - 3*r11 - 1, 0), (1, -r11**2 + 1, 0), (1, -r11**2, 0), (1, 1, -1), (0, 1, -1), (1, r11**4 + r11**3 - 4*r11**2 - 3*r11 + 2, -r11**4 - r11**3 + 4*r11**2 + 3*r11 - 3), (1, 2*r11**3 - 6*r11 - 1, -r11**3 + 3*r11), (1, -r11**4 - r11**3 + 2*r11**2 + 2*r11 + 2, r11**4 - 3*r11**2 + r11 - 1), (1, -3*r11**4 - 2*r11**3 + 11*r11**2 + 5*r11 - 8, 3*r11**4 + 2*r11**3 - 12*r11**2 - 5*r11 + 9), (1, -2*r11**4 + 6*r11**2 - 2*r11, r11**4 - 3*r11**2 + r11 - 1), (1, r11**3 - r11**2 - 3*r11, -r11**3 + 3*r11), (1, r11**4 + r11**3 - 5*r11**2 - 4*r11 + 3, -r11**4 - r11**3 + 4*r11**2 + 3*r11 - 3), (0, 1, -6/43*r11**4 - 7/43*r11**3 + 30/43*r11**2 + 23/43*r11 - 50/43), (1, r11 + 1, -1))
db_normals_CEL[(22, 264, 3)] = ((2*tau + 1, tau + 1, tau), (2*tau + 1, tau + 1, 1), (2*tau + 2, tau + 2, tau + 1), (3*tau + 3, 2*tau + 2, tau + 1), (tau + 1, tau, 0), (2*tau + 2, tau + 2, 1), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (1, 1, 1), (tau + 1, tau + 1, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (tau + 1, 1, 1), (tau + 1, 1, 0), (2*tau + 2, tau + 1, tau), (tau + 1, tau, tau), (2*tau + 2, tau + 1, 1))
db_normals_CEL[(22, 276, 1)] = ((0, 2, 1), (0, 0, 1), (0, 1, 3), (0, 1, 2), (1, 2, 1), (1, 3, 3), (1, 2, 2), (1, 0, 0), (2, 3, 4), (1, 3, 4), (2, 2, 3), (1, 2, 3), (2, 3, 5), (1, 1, 2), (1, 0, 1), (2, 4, 7), (2, 3, 6), (1, 3, 5), (1, 1, 3), (1, 0, 2), (1, 1, 4), (0, 1, 1))
db_normals_CEL[(22, 288, 1)] = ((1, 3/2*tau - 1/2, -tau), (1, 3/2*tau - 1/2, -1/2*tau - 1), (0, 0, 1), (0, 1, 1/5*tau - 3/5), (0, 1, 0), (1, -1, -1), (1, -1, -1/2*tau - 1/2), (1, -1/2*tau + 1/2, -1), (1, -1/2*tau + 1/2, -1/2*tau - 1/2), (1, 1/2*tau - 1/2, -1/2*tau - 1/2), (1, -tau + 2, -1/2*tau - 1/2), (1, 1/2*tau - 1/2, -2/5*tau - 4/5), (1, -tau + 2, tau - 3), (1, 3/4*tau - 3/4, -1/2*tau - 1/2), (1, 3/4*tau - 3/4, -1/4*tau - 1), (1, 5/2*tau - 7/2, tau - 3), (1, 5/2*tau - 7/2, -3/2*tau + 1), (1, 1/2*tau, -1/4*tau - 1), (1, 1/2*tau, -3/2*tau + 1), (1, 1/2*tau, -2/5*tau - 4/5), (1, 1/2*tau, -tau), (1, 1/2*tau, -1/2*tau - 1))
db_normals_CEL[(23, 296, 1)] = ((0, 2, 1), (0, 0, 1), (0, 1, 3), (0, 1, 2), (1, 2, 1), (1, 3, 3), (1, 2, 2), (1, 0, 0), (2, 3, 4), (1, 3, 4), (2, 2, 3), (1, 2, 3), (2, 3, 5), (1, 1, 2), (1, 0, 1), (2, 4, 7), (2, 3, 6), (1, 3, 5), (1, 2, 4), (1, 1, 3), (1, 0, 2), (1, 1, 4), (0, 1, 1))
db_normals_CEL[(23, 304, 1)] = ((1, 3/2*tau - 1/2, -tau), (1, 3/2*tau - 1/2, -1/2*tau - 1), (0, 0, 1), (0, 1, 1/5*tau - 3/5), (0, 1, 0), (1, -1, -1), (1, -1, -1/2*tau - 1/2), (1, -1/2*tau + 1/2, -1), (1, -1/2*tau + 1/2, -1/2*tau - 1/2), (1, 1/2*tau - 1/2, -1/2*tau - 1/2), (1, -tau + 2, -1/2*tau - 1/2), (1, 1/2*tau - 1/2, -2/5*tau - 4/5), (1, -tau + 2, tau - 3), (1, 3/4*tau - 3/4, -1/2*tau - 1/2), (1, 3/4*tau - 3/4, -1/4*tau - 1), (1, 5/2*tau - 7/2, tau - 3), (1, 5/2*tau - 7/2, -3/2*tau + 1), (1, 1/2*tau, -1/4*tau - 1), (1, 1/2*tau, -3/2*tau + 1), (1, 1/2*tau, -2/5*tau - 4/5), (1, 1/2*tau, -3/2), (1, 1/2*tau, -tau), (1, 1/2*tau, -1/2*tau - 1))
db_normals_CEL[(24, 304, 1)] = ((1, -sqrt2 - 2, sqrt2 + 1), (1, -sqrt2 - 1, 2*sqrt2 + 2), (1, -sqrt2 - 2, 2*sqrt2 + 2), (0, 1, 0), (0, 1, -1), (0, 1, -1/2*sqrt2 - 1), (0, 1, -sqrt2 - 1), (0, 1, -sqrt2 - 2), (0, 1, -2*sqrt2 - 3), (0, 0, 1), (0, 1, sqrt2 + 1), (1, 0, -sqrt2 - 2), (1, -1, -sqrt2 - 2), (1, 0, -1), (1, -1, -1), (1, -sqrt2, -1), (1, -1, 0), (1, -sqrt2 - 1, -1), (1, -2, 0), (1, -sqrt2, sqrt2), (1, -1, sqrt2 + 1), (1, -sqrt2 - 1, sqrt2), (1, -2, sqrt2 + 1), (1, -sqrt2 - 1, sqrt2 + 1))
db_normals_CEL[(24, 312, 1)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -3/2, 0), (1, -2, 0), (1, -3, 0), (1, 1/2*sqrt3 - 3/2, 0), (1, sqrt3 - 3, 0), (1, -sqrt3, 0), (1, -1/2*sqrt3 - 3/2, 0), (1, -sqrt3 - 3, 0), (1, sqrt3, 0), (1, 1, -1), (0, 1, -1), (1, -1/3*sqrt3 - 1, 1/3*sqrt3), (1, 2*sqrt3 - 5, -sqrt3 + 2), (1, -1/4*sqrt3 - 5/4, 1/4*sqrt3 - 1/4), (1, -5/3*sqrt3 + 1, 2/3*sqrt3 - 1), (1, -2/3*sqrt3 - 1, 2/3*sqrt3 - 1), (1, -3/4*sqrt3 - 5/4, 1/4*sqrt3 - 1/4), (1, sqrt3 - 5, -sqrt3 + 2), (1, -4/3*sqrt3 - 3, 1/3*sqrt3), (0, 1, 1/13*sqrt3 - 4/13), (1, sqrt3 + 1, -1))
db_normals_CEL[(24, 316, 1)] = ((0, 2, 1), (0, 0, 1), (0, 1, 3), (0, 1, 2), (1, 2, 1), (1, 3, 3), (1, 2, 2), (1, 1, 1), (1, 0, 0), (2, 3, 4), (1, 3, 4), (2, 2, 3), (1, 2, 3), (2, 3, 5), (1, 1, 2), (1, 0, 1), (2, 4, 7), (2, 3, 6), (1, 3, 5), (1, 2, 4), (1, 1, 3), (1, 0, 2), (1, 1, 4), (0, 1, 1))
db_normals_CEL[(24, 320, 1)] = ((1, 3/2*tau - 1/2, -tau), (1, 3/2*tau - 1/2, -1/2*tau - 1), (0, 0, 1), (0, 1, 1/5*tau - 3/5), (0, 1, 0), (1, -1, -1), (1, -1, -1/2*tau - 1/2), (1, -1/2*tau + 1/2, -1), (1, -1/2*tau + 1/2, -1/2*tau - 1/2), (1, 1/2*tau - 1/2, -1/2*tau - 1/2), (1, -tau + 2, -1/2*tau - 1/2), (1, 1/2*tau - 1/2, -2/5*tau - 4/5), (1, -tau + 2, tau - 3), (1, 3/4*tau - 3/4, -1/2*tau - 1/2), (1, 3/4*tau - 3/4, -1/4*tau - 1), (1, 5/2*tau - 7/2, tau - 3), (1, 5/2*tau - 7/2, -3/2*tau + 1), (1, 1/2*tau, -1/2*tau - 1/2), (1, 1/2*tau, -1/4*tau - 1), (1, 1/2*tau, -3/2*tau + 1), (1, 1/2*tau, -2/5*tau - 4/5), (1, 1/2*tau, -3/2), (1, 1/2*tau, -tau), (1, 1/2*tau, -1/2*tau - 1))
db_normals_CEL[(25, 320, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, -1), (0, 1, -1/2*sqrt2 - 1), (1, -1, -1), (1, -1, 0), (1, 0, -1), (1, -sqrt2, -1), (1, -sqrt2 - 2, sqrt2 + 1), (0, 1, sqrt2 + 1), (1, -1, sqrt2 + 1), (1, -sqrt2 - 1, sqrt2), (1, 0, -sqrt2 - 2), (1, -2, 0), (1, -sqrt2, sqrt2), (1, -sqrt2 - 2, 2*sqrt2 + 2), (1, -sqrt2 - 1, sqrt2 + 1), (1, -2, sqrt2 + 1), (0, 1, -sqrt2 - 1), (0, 1, -2*sqrt2 - 3), (0, 1, -sqrt2 - 2), (1, -1, -sqrt2 - 2), (1, -sqrt2 - 1, -1), (1, -1/2*sqrt2 - 1, 1/2*sqrt2), (1, -sqrt2 - 1, 2*sqrt2 + 2))
db_normals_CEL[(25, 336, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 3, 8))
db_normals_CEL[(25, 336, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 3, 7), (2, 3, 8), (2, 3, 9))
db_normals_CEL[(25, 336, 3)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 1), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 1), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 3, 3), (1, 3, 4), (1, 3, 5), (2, 2, 3), (2, 3, 4), (2, 3, 5), (2, 3, 6), (2, 4, 7))
db_normals_CEL[(25, 336, 4)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 2), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 3, 4), (1, 3, 5), (1, 3, 6), (2, 2, 5), (2, 3, 5), (2, 3, 6), (2, 3, 7), (2, 4, 7))
db_normals_CEL[(25, 336, 5)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -3/2, 0), (1, -2, 0), (1, -3, 0), (1, 1/2*sqrt3 - 3/2, 0), (1, sqrt3 - 3, 0), (1, -sqrt3, 0), (1, -1/2*sqrt3 - 3/2, 0), (1, -sqrt3 - 3, 0), (1, sqrt3, 0), (1, 1, -1), (0, 1, -1), (1, -1/3*sqrt3 - 1, 1/3*sqrt3), (1, 2*sqrt3 - 5, -sqrt3 + 2), (1, -1/4*sqrt3 - 5/4, 1/4*sqrt3 - 1/4), (1, -5/3*sqrt3 + 1, 2/3*sqrt3 - 1), (1, -2/3*sqrt3 - 1, 2/3*sqrt3 - 1), (1, -3/4*sqrt3 - 5/4, 1/4*sqrt3 - 1/4), (1, sqrt3 - 5, -sqrt3 + 2), (1, -4/3*sqrt3 - 3, 1/3*sqrt3), (0, 1, 1/13*sqrt3 - 4/13), (1, sqrt3 + 1, -1), (1, -sqrt3 - 1, 1/2*sqrt3 - 1/2))
db_normals_CEL[(25, 336, 6)] = ((0, 0, 1), (0, 1, 0), (1, -1, -1), (1, -1, -1/2*tau - 1/2), (1, -tau + 2, tau - 3), (0, 1, 1/5*tau - 3/5), (1, 1/2*tau - 1/2, -1/2*tau - 1/2), (1, -tau + 2, -1/2*tau - 1/2), (1, tau - 3/2, -1/2*tau - 1/2), (1, 1/2*tau, -1/2*tau - 1), (1, 1/2*tau, -1/4*tau - 1), (1, 1/2*tau, -3/2*tau + 1), (1, -1/2*tau + 1/2, -1), (1, 1/2*tau, -1/2*tau - 1/2), (1, -1/2*tau + 1/2, -1/2*tau - 1/2), (1, 3/4*tau - 3/4, -1/2*tau - 1/2), (1, 5/2*tau - 7/2, tau - 3), (1, 1/2*tau, -3/2), (1, 3/2*tau - 1/2, -1/2*tau - 1), (1, 3/4*tau - 3/4, -1/4*tau - 1), (1, 5/2*tau - 7/2, -3/2*tau + 1), (1, 1/2*tau - 1/2, -2/5*tau - 4/5), (1, 1/2*tau, -2/5*tau - 4/5), (1, 3/2*tau - 1/2, -tau), (1, 1/2*tau, -tau))
db_normals_CEL[(25, 360, 1)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (3*tau + 3, 2*tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (3*tau + 1, 2*tau + 1, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (tau + 1, tau, 0), (2*tau + 2, tau + 2, 1), (3*tau + 1, 2*tau + 1, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (0, 0, 1), (0, 1, 0), (0, 1, 1), (tau + 1, 1, 1), (2*tau, tau, tau), (tau + 1, 1, 0), (2*tau + 2, tau + 1, tau), (tau + 1, tau, tau), (2*tau + 2, tau + 1, 1), (3*tau + 1, 2*tau, tau))
db_normals_CEL[(26, 364, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 3, 7), (2, 3, 8))
db_normals_CEL[(26, 364, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 3, 7), (2, 3, 8), (2, 3, 9))
db_normals_CEL[(26, 364, 3)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, r13**4 + r13**3 - 4*r13**2 - 3*r13 + 1, 0), (1, -r13**5 + 4*r13**3 - r13**2 - 2*r13, 0), (1, r13**5 - 5*r13**3 + 5*r13 - 1, 0), (1, -r13**2 - r13, 0), (1, r13, 0), (1, -r13**5 + 5*r13**3 - r13**2 - 5*r13 + 1, 0), (1, r13**5 - 4*r13**3 + 2*r13, 0), (1, -r13**4 - r13**3 + 3*r13**2 + 3*r13 - 1, 0), (1, -r13**2 + 1, 0), (1, -r13**2, 0), (1, 1, -1), (0, 1, -1), (1, -r13**5 - r13**4 + 5*r13**3 + 4*r13**2 - 6*r13 - 4, r13**5 + r13**4 - 5*r13**3 - 4*r13**2 + 6*r13 + 3), (1, 2*r13**5 - 8*r13**3 + 4*r13 + 1, -r13**5 + 4*r13**3 - 2*r13 - 1), (1, -r13**5 + r13**4 + 7*r13**3 - 5*r13**2 - 12*r13 + 4, r13**5 - 6*r13**3 + r13**2 + 9*r13 - 3), (1, -r13**4 - r13**3 + 2*r13**2 + 3*r13 + 2, r13**2 - 3), (1, -4*r13**5 + 17*r13**3 - 2*r13**2 - 11*r13 - 1, 3*r13**5 - 13*r13**3 + r13**2 + 9*r13 + 1), (1, -2*r13**2 + 4, r13**2 - 3), (1, r13**3 - r13**2 - 4*r13 + 2, r13**5 - 6*r13**3 + r13**2 + 9*r13 - 3), (1, r13**5 - 4*r13**3 - r13**2 + 2*r13 + 1, -r13**5 + 4*r13**3 - 2*r13 - 1), (1, -r13**5 - r13**4 + 5*r13**3 + 3*r13**2 - 7*r13 - 3, r13**5 + r13**4 - 5*r13**3 - 4*r13**2 + 6*r13 + 3), (0, 1, 9/103*r13**5 - 2/103*r13**4 - 54/103*r13**3 + 30/103*r13**2 + 86/103*r13 - 101/103), (1, r13 + 1, -1))
db_normals_CEL[(26, 380, 1)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (3*tau + 3, 2*tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (3*tau + 1, 2*tau + 1, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (tau + 1, tau, 0), (2*tau + 2, tau + 2, 1), (3*tau + 1, 2*tau + 1, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (tau + 1, 1, 1), (2*tau, tau, tau), (tau + 1, 1, 0), (2*tau + 2, tau + 1, tau), (tau + 1, tau, tau), (2*tau + 2, tau + 1, 1), (3*tau + 1, 2*tau, tau))
db_normals_CEL[(27, 392, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 3, 7), (2, 3, 8), (2, 3, 9))
db_normals_CEL[(27, 392, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 2, 5), (2, 3, 7), (2, 3, 8), (2, 3, 9))
db_normals_CEL[(27, 392, 3)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 2, 7), (2, 3, 7), (2, 3, 8), (2, 3, 9))
db_normals_CEL[(27, 400, 1)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (3*tau + 3, 2*tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (3*tau + 1, 2*tau + 1, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (tau + 1, tau, 0), (2*tau + 2, tau + 2, 1), (3*tau + 1, 2*tau + 1, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (tau + 1, 1, 1), (2*tau, tau, tau), (tau + 1, 1, 0), (2*tau + 2, tau + 1, tau), (tau + 1, tau, tau), (2*tau + 2, tau + 1, 1), (3*tau + 1, 2*tau, tau))
db_normals_CEL[(28, 420, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 2, 5), (2, 3, 7), (2, 3, 8), (2, 3, 9))
db_normals_CEL[(28, 420, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 2, 5), (2, 3, 7), (2, 3, 8), (2, 3, 9), (2, 4, 9))
db_normals_CEL[(28, 420, 3)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 2, 5), (2, 3, 7), (2, 3, 8), (2, 3, 9), (2, 4, 11))
db_normals_CEL[(28, 420, 4)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -r14**2 + r14, 0), (1, -r14, 0), (1, -r14**2 + 1, 0), (1, -r14**2, 0), (1, -r14**2 + 2*r14 - 1, 0), (1, -r14**2 + 2, 0), (1, -1/2*r14**2, 0), (1, -2, 0), (1, -2*r14 + 1, 0), (1, -r14**2 - r14, 0), (1, r14, 0), (1, 1, -1), (0, 1, -1), (1, r14**2 - r14 - 3, -r14**2 + r14 + 2), (1, -2*r14**2 + 5, r14**2 - 3), (1, -2*r14 + 2, -r14**2 + 3*r14 - 2), (1, -r14**2 + 3/2, 1/2*r14**2 - 3/2), (1, 2*r14**2 - 3*r14 - 3, -2*r14**2 + 2*r14 + 3), (1, 2*r14**2 - 2*r14 - 5, -2*r14**2 + 2*r14 + 3), (1, -3/2*r14**2 + 5/2, 1/2*r14**2 - 3/2), (1, r14**2 - 5*r14 + 3, -r14**2 + 3*r14 - 2), (1, -2*r14**2 + 3, r14**2 - 3), (1, -2*r14 - 2, -r14**2 + r14 + 2), (0, 1, 1/13*r14**2 + 2/13*r14 - 9/13), (1, r14 + 1, -1))
db_normals_CEL[(28, 420, 5)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (3*tau + 3, 2*tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (3*tau + 1, 2*tau + 1, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (tau + 1, tau, 0), (2*tau + 2, tau + 2, 1), (3*tau + 1, 2*tau + 1, tau), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (tau + 1, 1, 1), (2*tau, tau, tau), (tau + 1, 1, 0), (2*tau + 2, tau + 1, tau), (tau + 1, tau, tau), (2*tau + 2, tau + 1, 1), (3*tau + 1, 2*tau, tau))
db_normals_CEL[(28, 420, 6)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (3*tau + 3, 2*tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (3*tau + 1, 2*tau + 1, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (tau + 1, tau, 0), (2*tau + 2, tau + 2, 1), (3*tau + 1, 2*tau + 1, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (1, 1, 1), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (tau + 1, 1, 1), (2*tau, tau, tau), (tau + 1, 1, 0), (2*tau + 2, tau + 1, tau), (tau + 1, tau, tau), (2*tau + 2, tau + 1, 1), (3*tau + 1, 2*tau, tau))
db_normals_CEL[(29, 440, 1)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 2, tau + 2, tau + 1), (3*tau + 3, 2*tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (3*tau + 1, 2*tau + 1, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (tau + 1, tau, 0), (2*tau + 2, tau + 2, 1), (3*tau + 1, 2*tau + 1, tau), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (1, 1, 1), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (tau + 1, 1, 1), (2*tau, tau, tau), (tau + 1, 1, 0), (2*tau + 2, tau + 1, tau), (tau + 1, tau, tau), (2*tau + 2, tau + 1, 1), (3*tau + 1, 2*tau, tau))
db_normals_CEL[(29, 448, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 2, 5), (2, 2, 7), (2, 3, 7), (2, 3, 8), (2, 3, 9))
db_normals_CEL[(29, 448, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 2, 5), (2, 3, 7), (2, 3, 8), (2, 3, 9), (2, 4, 9))
db_normals_CEL[(29, 448, 3)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 2, 5), (2, 3, 7), (2, 3, 8), (2, 3, 9), (2, 4, 11))
db_normals_CEL[(29, 448, 4)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -r14**2 + r14, 0), (1, -r14, 0), (1, -r14**2 + 1, 0), (1, -r14**2, 0), (1, -r14**2 + 2*r14 - 1, 0), (1, -r14**2 + 2, 0), (1, -1/2*r14**2, 0), (1, -2, 0), (1, -2*r14 + 1, 0), (1, -r14**2 - r14, 0), (1, r14, 0), (1, 1, -1), (0, 1, -1), (1, r14**2 - r14 - 3, -r14**2 + r14 + 2), (1, -2*r14**2 + 5, r14**2 - 3), (1, -2*r14 + 2, -r14**2 + 3*r14 - 2), (1, -r14**2 + 3/2, 1/2*r14**2 - 3/2), (1, 2*r14**2 - 3*r14 - 3, -2*r14**2 + 2*r14 + 3), (1, 2*r14**2 - 2*r14 - 5, -2*r14**2 + 2*r14 + 3), (1, -3/2*r14**2 + 5/2, 1/2*r14**2 - 3/2), (1, r14**2 - 5*r14 + 3, -r14**2 + 3*r14 - 2), (1, -2*r14**2 + 3, r14**2 - 3), (1, -2*r14 - 2, -r14**2 + r14 + 2), (0, 1, 1/13*r14**2 + 2/13*r14 - 9/13), (1, r14 + 1, -1), (1, -2*r14**2 + 4, r14**2 - 3))
db_normals_CEL[(30, 460, 1)] = ((2*tau, tau, 0), (4*tau + 2, 2*tau + 2, tau + 1), (2*tau + 1, tau + 1, 1), (2*tau + 2, tau + 2, tau + 1), (3*tau + 3, 2*tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau), (3*tau + 1, 2*tau + 1, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (tau + 1, tau, 0), (2*tau + 2, tau + 2, 1), (3*tau + 1, 2*tau + 1, tau), (2*tau + 1, 2*tau, tau), (2*tau + 2, 2*tau + 1, tau + 1), (2*tau + 2, 2*tau + 1, tau), (1, 1, 1), (tau + 1, tau + 1, tau), (2*tau, 2*tau, tau), (tau + 1, tau + 1, 1), (1, 1, 0), (0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (tau + 1, 1, 1), (2*tau, tau, tau), (tau + 1, 1, 0), (2*tau + 2, tau + 1, tau), (tau + 1, tau, tau), (2*tau + 2, tau + 1, 1), (3*tau + 1, 2*tau, tau))
db_normals_CEL[(30, 476, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 2, 5), (2, 2, 7), (2, 3, 7), (2, 3, 8), (2, 3, 9), (2, 4, 9))
db_normals_CEL[(30, 480, 1)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, 3/5*r15**3 - 1/5*r15**2 - 11/5*r15 - 2/5, 0), (1, -2*r15**3 - r15**2 + 7*r15 + 1, 0), (1, -1/3*r15**3 - 2/3*r15**2 + r15 + 1/3, 0), (1, -1/5*r15**3 - 3/5*r15**2 + 2/5*r15 - 1/5, 0), (1, -r15**2 - r15, 0), (1, r15, 0), (1, 1/5*r15**3 - 2/5*r15**2 - 2/5*r15 + 1/5, 0), (1, 1/3*r15**3 - 1/3*r15**2 - r15 - 1/3, 0), (1, 2*r15**3 - 7*r15 - 1, 0), (1, -3/5*r15**3 - 4/5*r15**2 + 11/5*r15 + 2/5, 0), (1, -r15**2 + 1, 0), (1, -r15**2, 0), (1, 1, -1), (0, 1, -1), (1, r15**3 - r15**2 - 4*r15 + 3, -r15**3 + r15**2 + 4*r15 - 4), (1, 2/3*r15**3 - 2/3*r15**2 - 2*r15 + 1/3, -1/3*r15**3 + 1/3*r15**2 + r15 - 2/3), (1, r15**3 - 2/5*r15**2 - 19/5*r15 + 3/5, -2/5*r15**3 + 1/5*r15**2 + 8/5*r15 - 1), (1, r15**3 + 2*r15**2 - 3*r15 - 9, r15**3 - 2*r15**2 - 4*r15 + 8), (1, -7/3*r15**3 - 5/3*r15**2 + 8*r15 + 10/3, 1/3*r15**3 + 2/3*r15**2 - r15 - 7/3), (1, -4/5*r15**3 - 6/5*r15**2 + 3*r15 + 7/5, 1/5*r15**3 + 2/5*r15**2 - 4/5*r15 - 1), (1, -2/3*r15**3 - 4/3*r15**2 + 2*r15 + 8/3, 1/3*r15**3 + 2/3*r15**2 - r15 - 7/3), (1, -r15**3 + r15**2 + 4*r15 - 7, r15**3 - 2*r15**2 - 4*r15 + 8), (1, 1/5*r15**3 - 4/5*r15**2 - 6/5*r15 + 4/5, -2/5*r15**3 + 1/5*r15**2 + 8/5*r15 - 1), (1, 1/3*r15**3 - 4/3*r15**2 - r15 + 2/3, -1/3*r15**3 + 1/3*r15**2 + r15 - 2/3), (1, r15**3 - 2*r15**2 - 5*r15 + 4, -r15**3 + r15**2 + 4*r15 - 4), (0, 1, -9/61*r15**3 + 13/61*r15**2 + 37/61*r15 - 66/61), (1, r15 + 1, -1))
db_normals_CEL[(31, 480, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (1, 0, 0), (1, 1, 0), (1, 1, 1), (tau + 1, 1, 0), (tau + 1, 1, 1), (tau + 1, tau, 0), (tau + 1, tau, tau), (tau + 1, tau + 1, 1), (tau + 1, tau + 1, tau), (2*tau, tau, 0), (2*tau, tau, tau), (2*tau, 2*tau, tau), (2*tau + 1, tau + 1, 1), (2*tau + 1, tau + 1, tau), (2*tau + 1, 2*tau, tau), (2*tau + 2, tau + 1, 1), (2*tau + 2, tau + 1, tau), (2*tau + 2, tau + 2, 1), (2*tau + 2, tau + 2, tau + 1), (2*tau + 2, 2*tau + 1, tau), (2*tau + 2, 2*tau + 1, tau + 1), (3*tau + 1, 2*tau, tau), (3*tau + 1, 2*tau + 1, tau), (3*tau + 1, 2*tau + 1, tau + 1), (3*tau + 3, 2*tau + 2, tau + 1), (4*tau + 2, 2*tau + 2, tau + 1), (4*tau + 2, 3*tau + 1, tau + 1), (4*tau + 2, 3*tau + 1, 2*tau))
db_normals_CEL[(31, 504, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 1, 5), (0, 1, 6), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 1, 6), (1, 1, 7), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 2, 8), (1, 2, 9), (1, 2, 10), (1, 3, 9), (1, 3, 10), (1, 3, 11), (2, 3, 10), (2, 3, 11), (2, 3, 12))
db_normals_CEL[(31, 504, 2)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 6), (1, 3, 7), (1, 3, 8), (2, 2, 5), (2, 2, 7), (2, 3, 7), (2, 3, 8), (2, 3, 9), (2, 4, 9), (2, 4, 11))
db_normals_CEL[(32, 544, 1)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -r16**2 + 2, 0), (1, -1/2*r16**2, 0), (1, -2, 0), (1, -r16**2 + 1, 0), (1, -r16**2, 0), (1, r16**3 - r16**2 - 3*r16 + 2, 0), (1, r16**3 - 3*r16 - 2, 0), (1, -r16**2 + r16, 0), (1, -r16, 0), (1, -r16**3 - r16**2 + 3*r16 + 2, 0), (1, -r16**3 + 3*r16 - 2, 0), (1, -r16**2 - r16, 0), (1, r16, 0), (1, 1, -1), (0, 1, -1), (1, 1/2*r16**3 - 2*r16 - 1, -1/2*r16**3 + 2*r16), (1, 2*r16**3 - 6*r16 - 3, -r16**3 + 3*r16 + 1), (1, 2*r16**3 - 3*r16**2 - 7*r16 + 9, -2*r16**3 + 2*r16**2 + 7*r16 - 7), (1, 1/2*r16**3 - 2*r16**2 + 2, -1/2*r16**3 + r16**2 + r16 - 2), (1, -1/2*r16**3 - r16**2 + 3/2*r16 + 2, 1/2*r16**3 + 1/2*r16**2 - 3/2*r16 - 2), (1, r16**3 - 5*r16 + 1, -r16**3 + 4*r16 - 1), (1, r16**3 - 4*r16 - 1, -r16**3 + 4*r16 - 1), (1, -3/2*r16**3 - 3/2*r16**2 + 9/2*r16 + 4, 1/2*r16**3 + 1/2*r16**2 - 3/2*r16 - 2), (1, 1/2*r16**3 - 2*r16**2 - r16 + 3, -1/2*r16**3 + r16**2 + r16 - 2), (1, r16**3 - 2*r16**2 - 4*r16 + 5, -2*r16**3 + 2*r16**2 + 7*r16 - 7), (1, r16**3 - r16**2 - 3*r16 - 1, -r16**3 + 3*r16 + 1), (1, 1/2*r16**3 - r16**2 - 3*r16, -1/2*r16**3 + 2*r16), (0, 1, -5/31*r16**3 + 6/31*r16**2 + 19/31*r16 - 29/31), (1, r16 + 1, -1))
db_normals_CEL[(33, 576, 1)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -r16**2 + 2, 0), (1, -1/2*r16**2, 0), (1, -2, 0), (1, -r16**2 + 1, 0), (1, -r16**2, 0), (1, r16**3 - r16**2 - 3*r16 + 2, 0), (1, r16**3 - 3*r16 - 2, 0), (1, -r16**2 + r16, 0), (1, -r16, 0), (1, -r16**3 - r16**2 + 3*r16 + 2, 0), (1, -r16**3 + 3*r16 - 2, 0), (1, -r16**2 - r16, 0), (1, r16, 0), (1, 1, -1), (0, 1, -1), (1, 1/2*r16**3 - 2*r16 - 1, -1/2*r16**3 + 2*r16), (1, 2*r16**3 - 6*r16 - 3, -r16**3 + 3*r16 + 1), (1, 2*r16**3 - 3*r16**2 - 7*r16 + 9, -2*r16**3 + 2*r16**2 + 7*r16 - 7), (1, 1/2*r16**3 - 2*r16**2 + 2, -1/2*r16**3 + r16**2 + r16 - 2), (1, -1/2*r16**3 - r16**2 + 3/2*r16 + 2, 1/2*r16**3 + 1/2*r16**2 - 3/2*r16 - 2), (1, r16**3 - 5*r16 + 1, -r16**3 + 4*r16 - 1), (1, r16**3 - 4*r16 - 1, -r16**3 + 4*r16 - 1), (1, -3/2*r16**3 - 3/2*r16**2 + 9/2*r16 + 4, 1/2*r16**3 + 1/2*r16**2 - 3/2*r16 - 2), (1, 1/2*r16**3 - 2*r16**2 - r16 + 3, -1/2*r16**3 + r16**2 + r16 - 2), (1, r16**3 - 2*r16**2 - 4*r16 + 5, -2*r16**3 + 2*r16**2 + 7*r16 - 7), (1, r16**3 - r16**2 - 3*r16 - 1, -r16**3 + 3*r16 + 1), (1, 1/2*r16**3 - r16**2 - 3*r16, -1/2*r16**3 + 2*r16), (0, 1, -5/31*r16**3 + 6/31*r16**2 + 19/31*r16 - 29/31), (1, r16 + 1, -1), (1, -2*r16**3 - 2*r16**2 + 6*r16 + 6, r16**3 + r16**2 - 3*r16 - 4))
db_normals_CEL[(34, 612, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 2, 3), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 3, 5), (1, 3, 6), (2, 1, 3), (2, 1, 4), (2, 2, 5), (2, 3, 5), (2, 3, 6), (2, 3, 7), (2, 3, 8), (2, 4, 7), (2, 4, 9), (3, 3, 7), (3, 3, 8), (3, 4, 8), (3, 4, 9), (3, 4, 10), (3, 5, 10), (3, 5, 11))
db_normals_CEL[(34, 612, 2)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, r17**7 + r17**6 - 6*r17**5 - 5*r17**4 + 9*r17**3 + 5*r17**2 - r17, 0), (1, r17**6 + r17**5 - 5*r17**4 - 4*r17**3 + 5*r17**2 + 2*r17 - 1, 0), (1, -r17**5 + 5*r17**3 - r17**2 - 5*r17 + 1, 0), (1, r17**7 - 6*r17**5 + 9*r17**3 - 2*r17, 0), (1, -r17**7 + 7*r17**5 - r17**4 - 14*r17**3 + 3*r17**2 + 6*r17 - 1, 0), (1, -r17**2 - r17, 0), (1, r17, 0), (1, r17**7 - 7*r17**5 + r17**4 + 14*r17**3 - 4*r17**2 - 6*r17 + 1, 0), (1, -r17**7 + 6*r17**5 - 9*r17**3 - r17**2 + 2*r17, 0), (1, r17**5 - 5*r17**3 + 5*r17 - 1, 0), (1, -r17**6 - r17**5 + 5*r17**4 + 4*r17**3 - 6*r17**2 - 2*r17 + 1, 0), (1, -r17**7 - r17**6 + 6*r17**5 + 5*r17**4 - 9*r17**3 - 6*r17**2 + r17, 0), (1, -r17**2 + 1, 0), (1, -r17**2, 0), (1, 1, -1), (0, 1, -1), (1, r17**7 + r17**6 - 7*r17**5 - 6*r17**4 + 15*r17**3 + 10*r17**2 - 10*r17 - 5, -r17**7 - r17**6 + 7*r17**5 + 6*r17**4 - 15*r17**3 - 10*r17**2 + 10*r17 + 4), (1, -2*r17**7 + 12*r17**5 - 18*r17**3 - 2*r17**2 + 4*r17 + 1, r17**7 - 6*r17**5 + 9*r17**3 + r17**2 - 2*r17 - 1), (1, 2*r17**6 + r17**5 - 12*r17**4 - 4*r17**3 + 17*r17**2 + 2*r17, r17**7 - r17**6 - 7*r17**5 + 7*r17**4 + 13*r17**3 - 12*r17**2 - 3*r17), (1, r17**7 - 6*r17**5 + r17**4 + 10*r17**3 - 5*r17**2 - 5*r17 + 5, -r17**7 + 7*r17**5 - r17**4 - 15*r17**3 + 5*r17**2 + 10*r17 - 6), (1, r17**7 + 2*r17**6 - 5*r17**5 - 11*r17**4 + 6*r17**3 + 15*r17**2 - 3*r17 - 4, -r17**7 - r17**6 + 6*r17**5 + 6*r17**4 - 10*r17**3 - 10*r17**2 + 5*r17 + 3), (1, -r17**7 - 2*r17**6 + 5*r17**5 + 10*r17**4 - 5*r17**3 - 12*r17**2 - r17 + 3, r17**7 + r17**6 - 6*r17**5 - 5*r17**4 + 9*r17**3 + 6*r17**2 - r17 - 2), (1, -5*r17**7 - 2*r17**6 + 34*r17**5 + 9*r17**4 - 69*r17**3 - 7*r17**2 + 41*r17 - 6, 5*r17**7 + 2*r17**6 - 35*r17**5 - 9*r17**4 + 74*r17**3 + 6*r17**2 - 46*r17 + 7), (1, -2*r17**7 - 2*r17**6 + 12*r17**5 + 10*r17**4 - 18*r17**3 - 12*r17**2 + 2*r17 + 2, r17**7 + r17**6 - 6*r17**5 - 5*r17**4 + 9*r17**3 + 6*r17**2 - r17 - 2), (1, 2*r17**7 + r17**6 - 12*r17**5 - 6*r17**4 + 19*r17**3 + 10*r17**2 - 7*r17 - 3, -r17**7 - r17**6 + 6*r17**5 + 6*r17**4 - 10*r17**3 - 10*r17**2 + 5*r17 + 3), (1, r17**7 - 7*r17**5 + r17**4 + 15*r17**3 - 6*r17**2 - 10*r17 + 7, -r17**7 + 7*r17**5 - r17**4 - 15*r17**3 + 5*r17**2 + 10*r17 - 6), (1, -2*r17**7 + r17**6 + 14*r17**5 - 8*r17**4 - 27*r17**3 + 15*r17**2 + 9*r17 - 1, r17**7 - r17**6 - 7*r17**5 + 7*r17**4 + 13*r17**3 - 12*r17**2 - 3*r17), (1, -r17**7 + 6*r17**5 - 9*r17**3 - 2*r17**2 + 2*r17 + 1, r17**7 - 6*r17**5 + 9*r17**3 + r17**2 - 2*r17 - 1), (1, r17**7 + r17**6 - 7*r17**5 - 6*r17**4 + 15*r17**3 + 9*r17**2 - 11*r17 - 4, -r17**7 - r17**6 + 7*r17**5 + 6*r17**4 - 15*r17**3 - 10*r17**2 + 10*r17 + 4), (0, 1, -27/577*r17**7 - 8/577*r17**6 + 216/577*r17**5 + 10/577*r17**4 - 583/577*r17**3 + 183/577*r17**2 + 590/577*r17 - 585/577), (1, r17 + 1, -1))
db_normals_CEL[(35, 680, 1)] = ((1, 0, 0), (0, 1, 0), (0, 0, 1), (1, 1, 1), (1, 0, gamma), (1, gamma**3 - 2*gamma**2 + gamma - 1, 0), (0, 1, -gamma + 1), (1, 1, -gamma + 1), (1, -gamma + 1, -gamma + 1), (1, gamma**2 - gamma + 1, gamma), (1, -gamma + 1, 0), (1, -gamma + 1, gamma**2 - 2*gamma + 1), (1, -gamma**3 + 3*gamma**2 - 3*gamma + 2, -gamma**3 + 3*gamma**2 - 2*gamma + 1), (1, 1, -gamma**2 + 2*gamma), (1, -gamma**3 + 2*gamma**2 - 2*gamma + 2, 1), (1, -gamma**3 + 2*gamma**2 - 2*gamma + 2, -gamma**3 + 2*gamma**2 - gamma + 1), (1, gamma**2 - gamma + 1, -gamma**3 + 2*gamma**2 - gamma + 1), (1, gamma**2 - gamma + 1, -gamma**3 + 3*gamma**2 - 2*gamma + 1), (1, -1/3*gamma**3 + 2/3*gamma**2 - 2/3*gamma + 1, -1/3*gamma**2 + 2/3*gamma + 1/3), (1, -gamma**3 + 2*gamma**2 - gamma + 1, -gamma**2 + 2*gamma), (0, 1, gamma**3 - 2*gamma**2), (1, -gamma**3 + 3*gamma**2 - 2*gamma + 1, -gamma**3 + gamma**2 + 3*gamma - 1), (1, 0, -gamma**2 + 2*gamma - 1), (1, -gamma**3 + 2*gamma**2 - 2*gamma + 2, gamma), (1, -gamma**3 + gamma**2 - gamma + 1, gamma), (1, -gamma**3 + 2*gamma**2 - 2*gamma + 2, 0), (1, -gamma**3 + 2*gamma**2 - gamma + 1, -gamma**2 + 3*gamma - 1), (1, 0, -gamma**2 + 3*gamma - 1), (0, 1, gamma**3 - 2*gamma**2 - gamma + 1), (1, -gamma + 1, -gamma**3 + 3*gamma**2 - 2*gamma + 1), (1, -gamma**3 + 2*gamma**2 - 3*gamma + 2, -gamma**3 + 3*gamma**2 - 2*gamma + 1), (1, gamma**2 + 1, -gamma**2 + 2*gamma), (1, -gamma**3 + 3*gamma**2 - 3*gamma + 2, -2*gamma**3 + 5*gamma**2 - 2*gamma + 1), (1, -gamma**3 + 2*gamma**2 - gamma + 1, -gamma**3 + gamma**2 + 3*gamma - 1), (1, -gamma**3 + gamma**2 - gamma + 1, -gamma**2 + 2*gamma))
db_normals_CEL[(36, 684, 1)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -2/3*r18**2 + 1/3*r18 + 1/3, 0), (1, -r18**2 + r18, 0), (1, -r18, 0), (1, -1/3*r18**2 - 1/3*r18 - 1/3, 0), (1, -r18**2 + 1, 0), (1, -r18**2, 0), (1, -r18**2 + r18 + 1, 0), (1, r18**2 - 2*r18 - 1, 0), (1, -r18**2 + 2, 0), (1, -1/2*r18**2, 0), (1, -2, 0), (1, -2*r18**2 + 2*r18 + 1, 0), (1, -r18 - 1, 0), (1, -r18**2 - r18, 0), (1, r18, 0), (1, 1, -1), (0, 1, -1), (1, -r18**2 + 2, r18**2 - 3), (1, 2*r18**2 - 4*r18 - 1, -r18**2 + 2*r18), (1, -r18**2 + 2, 1/3*r18**2 + 1/3*r18 - 5/3), (1, 4*r18**2 - 2*r18 - 12, -5*r18**2 + 2*r18 + 14), (1, -4*r18**2 + 5*r18 + 3, 3*r18**2 - 4*r18 - 3), (1, -2/3*r18**2 - 1/6*r18 + 5/6, 1/6*r18**2 + 1/6*r18 - 5/6), (1, -2*r18**2 - r18 + 7, 2*r18**2 - 7), (1, -2*r18**2 + 5, 2*r18**2 - 7), (1, -1/2*r18**2 - 1/2*r18 + 1/2, 1/6*r18**2 + 1/6*r18 - 5/6), (1, -5*r18**2 + 6*r18 + 4, 3*r18**2 - 4*r18 - 3), (1, 4*r18**2 - 2*r18 - 13, -5*r18**2 + 2*r18 + 14), (1, -1/3*r18**2 - 4/3*r18 + 2/3, 1/3*r18**2 + 1/3*r18 - 5/3), (1, -2*r18, -r18**2 + 2*r18), (1, -2*r18**2 - r18 + 3, r18**2 - 3), (0, 1, 1/3*r18**2 - 4/3), (1, r18 + 1, -1))
db_normals_CEL[(37, 720, 1)] = ((0, 0, 1), (0, 1, 0), (0, 1, 1), (0, 1, 2), (0, 1, 3), (0, 1, 4), (0, 2, 3), (0, 2, 5), (1, 0, 0), (1, 0, 1), (1, 0, 2), (1, 1, 1), (1, 1, 2), (1, 1, 3), (1, 1, 4), (1, 1, 5), (1, 2, 3), (1, 2, 4), (1, 2, 5), (1, 2, 6), (1, 2, 7), (1, 3, 5), (1, 3, 6), (1, 3, 7), (1, 3, 8), (1, 3, 9), (1, 4, 8), (1, 4, 9), (1, 4, 10), (2, 3, 7), (2, 3, 8), (2, 3, 9), (2, 4, 9), (2, 4, 11), (2, 5, 11), (2, 5, 12), (2, 5, 13))
db_normals_CEL[(37, 720, 2)] = ((0, 1, 0), (1, 0, 0), (1, -1, 0), (1, -2/3*r18**2 + 1/3*r18 + 1/3, 0), (1, -r18**2 + r18, 0), (1, -r18, 0), (1, -1/3*r18**2 - 1/3*r18 - 1/3, 0), (1, -r18**2 + 1, 0), (1, -r18**2, 0), (1, -r18**2 + r18 + 1, 0), (1, r18**2 - 2*r18 - 1, 0), (1, -r18**2 + 2, 0), (1, -1/2*r18**2, 0), (1, -2, 0), (1, -2*r18**2 + 2*r18 + 1, 0), (1, -r18 - 1, 0), (1, -r18**2 - r18, 0), (1, r18, 0), (1, 1, -1), (0, 1, -1), (1, -r18**2 + 2, r18**2 - 3), (1, 2*r18**2 - 4*r18 - 1, -r18**2 + 2*r18), (1, -r18**2 + 2, 1/3*r18**2 + 1/3*r18 - 5/3), (1, 4*r18**2 - 2*r18 - 12, -5*r18**2 + 2*r18 + 14), (1, -4*r18**2 + 5*r18 + 3, 3*r18**2 - 4*r18 - 3), (1, -2/3*r18**2 - 1/6*r18 + 5/6, 1/6*r18**2 + 1/6*r18 - 5/6), (1, -2*r18**2 - r18 + 7, 2*r18**2 - 7), (1, -2*r18**2 + 5, 2*r18**2 - 7), (1, -1/2*r18**2 - 1/2*r18 + 1/2, 1/6*r18**2 + 1/6*r18 - 5/6), (1, -5*r18**2 + 6*r18 + 4, 3*r18**2 - 4*r18 - 3), (1, 4*r18**2 - 2*r18 - 13, -5*r18**2 + 2*r18 + 14), (1, -1/3*r18**2 - 4/3*r18 + 2/3, 1/3*r18**2 + 1/3*r18 - 5/3), (1, -2*r18, -r18**2 + 2*r18), (1, -2*r18**2 - r18 + 3, r18**2 - 3), (0, 1, 1/3*r18**2 - 4/3), (1, r18 + 1, -1), (1, -2/3*r18**2 - 2/3*r18 + 4/3, 1/3*r18**2 + 1/3*r18 - 5/3))
db_normals_CEL[(37, 720, 3)] = ((1, 1/2*sqrt3 + 1/2, -1/2*sqrt3 - 1/2), (1, 1/2*sqrt3 + 1, -1/2), (0, 1, sqrt3 + 2), (0, 0, 1), (0, 1, -sqrt3 - 2), (0, 1, -sqrt3), (0, 1, -1), (0, 1, -1/3*sqrt3), (0, 1, sqrt3 - 2), (0, 1, 0), (0, 1, -sqrt3 + 2), (0, 1, 1/3*sqrt3), (0, 1, 1), (1, -1/2*sqrt3 - 1/2, 1/2*sqrt3 + 1/2), (1, -1/2*sqrt3 - 1, 1/2), (1, -1/2, 1/2*sqrt3 + 1), (1, -1/2*sqrt3 - 1, -1/2), (1, -1/2*sqrt3, 1/2), (1, -1/2, 1/2*sqrt3), (1, -1, 0), (1, 1/2, 1/2*sqrt3 + 1), (1, 0, 1), (1, -1/2*sqrt3, -1/2), (1, -1/2*sqrt3 - 1/2, -1/2*sqrt3 - 1/2), (1, 1/2, 1/2*sqrt3), (1, 0, 0), (1, -1/2, -1/2*sqrt3), (1, 1/2*sqrt3 + 1/2, 1/2*sqrt3 + 1/2), (1, 1/2*sqrt3, 1/2), (1, 0, -1), (1, -1/2, -1/2*sqrt3 - 1), (1, 1, 0), (1, 1/2, -1/2*sqrt3), (1, 1/2*sqrt3, -1/2), (0, 1, sqrt3), (1, 1/2, -1/2*sqrt3 - 1), (1, 1/2*sqrt3 + 1, 1/2))

# This is the list of invariants of the hyperplane arrangements listed in [CEL]_ Table 5.
# The first entry is the labeling of Gruenbaum
# The second entry is the f-vector
# The third entry is the t-vector
# The fourth entry is the r-vector
# The fifth entry is the automorphism group of the CW-complex
# The sixth entry is the automorphism group of the matroid
# The seventh entry is the list of roots of the characteristic polynomial, when integers

db_invariants_CEL = {}
db_invariants_CEL[(6, 24, 1)] = ((6, 1), (7, 18, 12), (3, 4), (0, 6), (48, 48), (24, 12), (1, 2, 3))
db_invariants_CEL[(7, 32, 1)] = ((7, 1), (9, 24, 16), (3, 6), (0, 4, 3), (48, 48), (24, 12), (1, 3, 3))
db_invariants_CEL[(8, 40, 1)] = ((8, 1), (11, 30, 20), (4, 6, 1), (0, 2, 6), (16, 11), (8, 3), (1, 3, 4))
db_invariants_CEL[(9, 48, 1)] = ((9, 1), (13, 36, 24), (6, 4, 3), (0, 0, 9), (48, 48), (24, 12), (1, 3, 5))
db_invariants_CEL[(10, 60, 1)] = ((10, 3), (16, 45, 30), (6, 7, 3), (0, 1, 3, 6), (24, 14), (12, 4), (1, 4, 5))
db_invariants_CEL[(10, 60, 2)] = ((10, 2), (16, 45, 30), (6, 7, 3), (0, 0, 6, 3, 1), (12, 4), (6, 1), (1, 4, 5))
db_invariants_CEL[(10, 60, 3)] = ((10, 1), (16, 45, 30), (5, 10, 0, 1), (0, 0, 5, 5), (20, 4), (20, 3), (1, 4, 5))
db_invariants_CEL[(11, 72, 1)] = ((11, 1), (19, 54, 36), (7, 8, 4), (0, 0, 4, 4, 3), (8, 5), (4, 2), (1, 5, 5))
db_invariants_CEL[(12, 84, 1)] = ((12, 2), (22, 63, 42), (8, 10, 3, 1), (0, 0, 3, 3, 6), (8, 5), (4, 2), (1, 5, 6))
db_invariants_CEL[(12, 84, 2)] = ((12, 3), (22, 63, 42), (9, 7, 6), (0, 0, 3, 3, 6), (12, 4), (6, 1), (1, 5, 6))
db_invariants_CEL[(12, 84, 3)] = ((12, 1), (22, 63, 42), (6, 15, 0, 0, 1), (0, 0, 3, 3, 6), (24, 14), (12, 4), (1, 5, 6))
db_invariants_CEL[(13, 96, 1)] = ((13, 1), (25, 72, 48), (9, 12, 3, 0, 1), (0, 0, 3, 0, 10), (24, 14), (12, 4), (1, 5, 7))
db_invariants_CEL[(13, 96, 2)] = ((13, 3), (25, 72, 48), (10, 10, 3, 2), (0, 0, 1, 4, 8), (8, 5), (4, 2), (1, 5, 7))
db_invariants_CEL[(13, 96, 3)] = ((13, 2), (25, 72, 48), (12, 4, 9), (0, 0, 3, 0, 10), (48, 48), (24, 12), (1, 5, 7))
db_invariants_CEL[(13, 104, 1)] = ((13, 4), (27, 78, 52), (6, 18, 3), (0, 0, 0, 0, 13), (24, 13), (24, 12), ())
db_invariants_CEL[(14, 112, 1)] = ((14, 2), (29, 84, 56), (11, 12, 4, 2), (0, 0, 1, 4, 4, 4, 1), (4, 2), (2, 1), (1, 6, 7))
db_invariants_CEL[(14, 112, 2)] = ((14, 1), (29, 84, 56), (7, 21, 0, 0, 0, 1), (0, 0, 0, 7, 0, 7), (28, 3), (42, 1), (1, 6, 7))
db_invariants_CEL[(14, 112, 3)] = ((14, 4), (29, 84, 56), (10, 14, 4, 0, 1), (0, 0, 0, 4, 6, 4), (8, 5), (8, 3), (1, 6, 7))
db_invariants_CEL[(14, 116, 1)] = ((14, 3), (30, 87, 58), (9, 16, 4, 1), (0, 0, 0, 0, 11, 3), (4, 2), (4, 2), ())
db_invariants_CEL[(15, 120, 1)] = ((15, 1), (31, 90, 60), (15, 10, 0, 6), (0, 0, 0, 0, 15), (120, 35), (120, 34), (1, 5, 9))
db_invariants_CEL[(15, 128, 1)] = ((15, 2), (33, 96, 64), (13, 12, 6, 2), (0, 0, 1, 4, 2, 4, 4), (16, 11), (8, 3), (1, 7, 7))
db_invariants_CEL[(15, 128, 2)] = ((15, 4), (33, 96, 64), (12, 14, 6, 0, 1), (0, 0, 0, 0, 10, 4, 1), (8, 5), (8, 3), (1, 7, 7))
db_invariants_CEL[(15, 132, 1)] = ((15, 5), (34, 99, 66), (9, 22, 0, 3), (0, 0, 0, 0, 9, 3, 3), (12, 4), (6, 1), ())
db_invariants_CEL[(15, 132, 2)] = ((15, 3), (34, 99, 66), (12, 13, 9), (0, 0, 0, 0, 9, 3, 3), (12, 4), (6, 1), ())
db_invariants_CEL[(16, 140, 1)] = ((16, 4), (36, 105, 70), (15, 15, 0, 6), (0, 0, 0, 0, 10, 5, 0, 0, 1), (20, 4), (20, 3), (1, 6, 9))
db_invariants_CEL[(16, 144, 1)] = ((16, 2), (37, 108, 72), (14, 15, 6, 1, 1), (0, 0, 1, 2, 4, 2, 7), (8, 5), (4, 2), (1, 7, 8))
db_invariants_CEL[(16, 144, 2)] = ((16, 3), (37, 108, 72), (15, 13, 6, 3), (0, 0, 0, 0, 10, 0, 6), (12, 4), (6, 1), (1, 7, 8))
db_invariants_CEL[(16, 144, 3)] = ((16, 1), (37, 108, 72), (8, 28, 0, 0, 0, 0, 1), (0, 0, 0, 4, 4, 0, 8), (32, 39), (32, 43), (1, 7, 8))
db_invariants_CEL[(16, 144, 4)] = ((16, 6), (37, 108, 72), (15, 12, 9, 0, 1), (0, 0, 0, 0, 7, 6, 3), (12, 4), (6, 1), (1, 7, 8))
db_invariants_CEL[(16, 144, 5)] = ((16, 5), (37, 108, 72), (14, 16, 3, 4), (0, 0, 0, 2, 4, 8, 0, 2), (8, 5), (8, 3), (1, 7, 8))
db_invariants_CEL[(16, 148, 1)] = ((16, 7), (38, 111, 74), (12, 19, 6, 0, 1), (0, 0, 0, 3, 3, 2, 8), (8, 5), (4, 2), ())
db_invariants_CEL[(17, 160, 1)] = ((17, 2), (41, 120, 80), (16, 16, 7, 0, 2), (0, 0, 1, 0, 6, 0, 10), (16, 11), (8, 3), (1, 7, 9))
db_invariants_CEL[(17, 160, 2)] = ((17, 4), (41, 120, 80), (16, 16, 7, 0, 2), (0, 0, 1, 0, 6, 0, 10), (16, 11), (8, 3), (1, 7, 9))
db_invariants_CEL[(17, 160, 3)] = ((17, 3), (41, 120, 80), (18, 12, 7, 4), (0, 0, 0, 0, 8, 0, 9), (16, 11), (8, 3), (1, 7, 9))
db_invariants_CEL[(17, 160, 4)] = ((17, 1), (41, 120, 80), (12, 24, 4, 0, 0, 0, 1), (0, 0, 0, 0, 8, 0, 9), (32, 39), (32, 43), (1, 7, 9))
db_invariants_CEL[(17, 160, 5)] = ((17, 5), (41, 120, 80), (16, 18, 1, 6), (0, 0, 0, 0, 6, 8, 1, 0, 2), (8, 5), (8, 3), (1, 7, 9))
db_invariants_CEL[(17, 164, 1)] = ((17, 6), (42, 123, 82), (16, 15, 10, 0, 1), (0, 0, 0, 0, 6, 3, 7, 0, 1), (4, 2), (2, 1), ())
db_invariants_CEL[(17, 168, 1)] = ((17, 8), (43, 126, 84), (14, 20, 7, 2), (0, 0, 0, 0, 1, 8, 8), (8, 5), (8, 3), ())
db_invariants_CEL[(17, 168, 2)] = ((17, 7), (43, 126, 84), (13, 22, 7, 0, 1), (0, 0, 0, 0, 6, 0, 10, 0, 1), (8, 5), (4, 2), ())
db_invariants_CEL[(18, 180, 1)] = ((18, 7), (46, 135, 90), (18, 18, 6, 3, 1), (0, 0, 0, 3, 3, 0, 6, 6), (12, 4), (6, 1), (1, 8, 9))
db_invariants_CEL[(18, 180, 2)] = ((18, 3), (46, 135, 90), (19, 16, 6, 5), (0, 0, 0, 0, 6, 2, 6, 3, 1), (4, 2), (2, 1), (1, 8, 9))
db_invariants_CEL[(18, 180, 3)] = ((18, 1), (46, 135, 90), (9, 36, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 9, 0, 0, 9), (36, 4), (54, 6), (1, 8, 9))
db_invariants_CEL[(18, 180, 4)] = ((18, 2), (46, 135, 90), (18, 18, 6, 3, 1), (0, 0, 0, 0, 3, 3, 12), (24, 14), (12, 4), (1, 8, 9))
db_invariants_CEL[(18, 180, 5)] = ((18, 5), (46, 135, 90), (18, 19, 3, 6), (0, 0, 0, 0, 3, 9, 3, 0, 3), (12, 4), (6, 1), (1, 8, 9))
db_invariants_CEL[(18, 180, 6)] = ((18, 4), (46, 135, 90), (18, 19, 3, 6), (0, 0, 0, 0, 3, 9, 3, 0, 3), (12, 4), (6, 1), (1, 8, 9))
db_invariants_CEL[(18, 184, 1)] = ((18, 6), (47, 138, 92), (18, 16, 12, 0, 1), (0, 0, 0, 0, 5, 2, 7, 2, 2), (4, 2), (2, 1), ())
db_invariants_CEL[(18, 184, 2)] = ((18, 8), (47, 138, 92), (16, 22, 6, 2, 1), (0, 0, 0, 0, 6, 0, 7, 4, 1), (4, 2), (2, 1), ())
db_invariants_CEL[(19, 192, 1)] = ((19, 1), (49, 144, 96), (21, 18, 6, 0, 4), (0, 0, 0, 0, 4, 0, 15), (24, 14), (12, 4), (1, 7, 11))
db_invariants_CEL[(19, 192, 2)] = ((19, 3), (49, 144, 96), (24, 12, 6, 6, 1), (0, 0, 0, 0, 4, 0, 15), (24, 14), (12, 4), (1, 7, 11))
db_invariants_CEL[(19, 200, 1)] = ((19, 4), (51, 150, 100), (20, 20, 6, 4, 1), (0, 0, 0, 0, 4, 4, 4, 4, 3), (8, 5), (4, 2), (1, 9, 9))
db_invariants_CEL[(19, 200, 2)] = ((19, 5), (51, 150, 100), (20, 20, 6, 4, 1), (0, 0, 0, 0, 4, 4, 4, 4, 3), (8, 2), (4, 1), (1, 9, 9))
db_invariants_CEL[(19, 200, 3)] = ((19, 6), (51, 150, 100), (20, 20, 6, 4, 1), (0, 0, 0, 0, 6, 0, 6, 4, 3), (8, 5), (4, 2), (1, 9, 9))
db_invariants_CEL[(19, 200, 4)] = ((19, 2), (51, 150, 100), (21, 18, 6, 6), (0, 0, 0, 0, 1, 8, 6, 0, 4), (8, 5), (8, 3), (1, 9, 9))
db_invariants_CEL[(19, 204, 1)] = ((19, 7), (52, 153, 102), (21, 15, 15, 0, 1), (0, 0, 0, 0, 4, 3, 3, 6, 3), (12, 4), (6, 1), ())
db_invariants_CEL[(20, 216, 1)] = ((20, 5), (55, 162, 108), (20, 26, 4, 4, 0, 0, 1), (0, 0, 0, 2, 2, 0, 4, 12), (16, 11), (8, 3), (1, 8, 11))
db_invariants_CEL[(20, 220, 1)] = ((20, 3), (56, 165, 110), (21, 24, 6, 4, 0, 1), (0, 0, 0, 0, 4, 2, 4, 6, 3, 1), (4, 2), (2, 1), (1, 9, 10))
db_invariants_CEL[(20, 220, 2)] = ((20, 4), (56, 165, 110), (23, 20, 7, 5, 1), (0, 0, 0, 0, 5, 1, 4, 4, 6), (4, 2), (2, 1), (1, 9, 10))
db_invariants_CEL[(20, 220, 3)] = ((20, 1), (56, 165, 110), (10, 45, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 5, 5, 0, 0, 10), (40, 13), (40, 12), (1, 9, 10))
db_invariants_CEL[(20, 220, 4)] = ((20, 2), (56, 165, 110), (25, 15, 10, 6), (0, 0, 0, 0, 0, 5, 10, 0, 5), (20, 4), (20, 3), (1, 9, 10))
db_invariants_CEL[(21, 240, 1)] = ((21, 4), (61, 180, 120), (22, 28, 6, 4, 0, 0, 1), (0, 0, 0, 0, 4, 0, 4, 8, 4, 0, 1), (16, 11), (8, 3), (1, 9, 11))
db_invariants_CEL[(21, 240, 2)] = ((21, 5), (61, 180, 120), (26, 20, 9, 4, 2), (0, 0, 0, 0, 5, 0, 3, 4, 9), (8, 5), (4, 2), (1, 9, 11))
db_invariants_CEL[(21, 240, 3)] = ((21, 3), (61, 180, 120), (24, 24, 9, 0, 4), (0, 0, 0, 0, 6, 0, 3, 0, 12), (48, 48), (24, 12), (1, 9, 11))
db_invariants_CEL[(21, 240, 4)] = ((21, 1), (61, 180, 120), (15, 40, 5, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 5, 0, 5, 0, 11), (40, 13), (40, 12), (1, 9, 11))
db_invariants_CEL[(21, 240, 5)] = ((21, 2), (61, 180, 120), (30, 10, 15, 6), (0, 0, 0, 0, 0, 0, 15, 0, 6), (120, 35), (120, 34), (1, 9, 11))
db_invariants_CEL[(21, 248, 1)] = ((21, 6), (63, 186, 124), (25, 20, 15, 2, 1), (0, 0, 0, 0, 1, 0, 11, 0, 8, 0, 1), (8, 5), (4, 2), ())
db_invariants_CEL[(21, 252, 1)] = ((21, 7), (64, 189, 126), (24, 22, 15, 3), (0, 0, 0, 0, 0, 0, 12, 0, 6, 3), (12, 4), (6, 1), ())
db_invariants_CEL[(22, 264, 1)] = ((22, 4), (67, 198, 132), (27, 25, 9, 3, 3), (0, 0, 0, 0, 4, 0, 6, 0, 6, 6), (12, 4), (6, 1), (1, 10, 11))
db_invariants_CEL[(22, 264, 2)] = ((22, 1), (67, 198, 132), (11, 55, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 11, 0, 0, 0, 11), (44, 3), (110, 1), (1, 10, 11))
db_invariants_CEL[(22, 264, 3)] = ((22, 3), (67, 198, 132), (27, 28, 0, 12), (0, 0, 0, 0, 0, 0, 12, 0, 9, 0, 1), (12, 4), (12, 4), (1, 10, 11))
db_invariants_CEL[(22, 276, 1)] = ((22, 2), (70, 207, 138), (24, 30, 12, 3, 1), (0, 0, 0, 0, 1, 0, 6, 3, 9, 0, 3), (12, 4), (6, 1), ())
db_invariants_CEL[(22, 288, 1)] = ((22, 5), (73, 216, 144), (12, 58, 0, 0, 3), (0, 0, 0, 0, 0, 0, 0, 12, 6, 0, 4), (48, 48), (48, 48), ())
db_invariants_CEL[(23, 296, 1)] = ((23, 1), (75, 222, 148), (27, 32, 10, 4, 2), (0, 0, 0, 0, 1, 0, 6, 2, 7, 4, 3), (4, 2), (2, 1), ())
db_invariants_CEL[(23, 304, 1)] = ((23, 2), (77, 228, 152), (16, 56, 2, 0, 1, 2), (0, 0, 0, 0, 0, 0, 1, 8, 10, 0, 4), (16, 11), (16, 11), ())
db_invariants_CEL[(24, 304, 1)] = ((24, 2), (77, 228, 152), (32, 32, 0, 12, 0, 0, 1), (0, 0, 0, 0, 0, 4, 0, 0, 20), (32, 39), (32, 43), ())
db_invariants_CEL[(24, 312, 1)] = ((24, 1), (79, 234, 156), (12, 66, 0, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 12), (48, 36), (48, 38), (1, 11, 12))
db_invariants_CEL[(24, 316, 1)] = ((24, 3), (80, 237, 158), (31, 32, 9, 5, 3), (0, 0, 0, 0, 1, 0, 6, 1, 6, 6, 4), (4, 2), (2, 1), ())
db_invariants_CEL[(24, 320, 1)] = ((24, 4), (81, 240, 160), (20, 54, 4, 0, 0, 2, 1), (0, 0, 0, 0, 0, 0, 2, 4, 14, 0, 4), (16, 11), (16, 11), ())
db_invariants_CEL[(25, 320, 1)] = ((25, 5), (81, 240, 160), (36, 32, 0, 8, 4, 0, 1), (0, 0, 0, 0, 0, 0, 5, 0, 20), (32, 39), (32, 43), (1, 9, 15))
db_invariants_CEL[(25, 336, 1)] = ((25, 7), (85, 252, 168), (33, 34, 12, 2, 3, 0, 1), (0, 0, 0, 0, 2, 0, 4, 4, 4, 0, 11), (8, 5), (4, 2), (1, 11, 13))
db_invariants_CEL[(25, 336, 2)] = ((25, 4), (85, 252, 168), (36, 30, 9, 6, 4), (0, 0, 0, 0, 1, 0, 9, 0, 3, 0, 12), (24, 14), (12, 4), (1, 11, 13))
db_invariants_CEL[(25, 336, 3)] = ((25, 6), (85, 252, 168), (36, 30, 9, 6, 4), (0, 0, 0, 0, 1, 0, 6, 0, 6, 6, 6), (12, 4), (6, 1), (1, 11, 13))
db_invariants_CEL[(25, 336, 4)] = ((25, 2), (85, 252, 168), (36, 28, 15, 0, 6), (0, 0, 0, 0, 4, 0, 3, 0, 6, 0, 12), (48, 48), (24, 12), (1, 11, 13))
db_invariants_CEL[(25, 336, 5)] = ((25, 1), (85, 252, 168), (18, 60, 6, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 12, 0, 0, 0, 13), (48, 36), (48, 38), (1, 11, 13))
db_invariants_CEL[(25, 336, 6)] = ((25, 8), (85, 252, 168), (24, 52, 6, 0, 0, 0, 3), (0, 0, 0, 0, 0, 0, 3, 0, 18, 0, 4), (48, 48), (48, 48), (1, 11, 13))
db_invariants_CEL[(25, 360, 1)] = ((25, 3), (91, 270, 180), (30, 40, 15, 6), (0, 0, 0, 0, 0, 0, 0, 0, 15, 0, 10), (120, 35), (120, 34), ())
db_invariants_CEL[(26, 364, 1)] = ((26, 4), (92, 273, 182), (35, 39, 10, 4, 3, 0, 1), (0, 0, 0, 0, 1, 1, 4, 4, 2, 2, 7, 4, 1), (4, 2), (2, 1), (1, 12, 13))
db_invariants_CEL[(26, 364, 2)] = ((26, 3), (92, 273, 182), (37, 36, 9, 6, 3, 1), (0, 0, 0, 0, 1, 0, 7, 2, 2, 1, 8, 4, 1), (4, 2), (2, 1), (1, 12, 13))
db_invariants_CEL[(26, 364, 3)] = ((26, 1), (92, 273, 182), (13, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 13, 0, 0, 0, 0, 13), (52, 4), (156, 7), (1, 12, 13))
db_invariants_CEL[(26, 380, 1)] = ((26, 2), (96, 285, 190), (35, 40, 10, 11), (0, 0, 0, 0, 0, 0, 0, 0, 11, 5, 10), (20, 4), (20, 3), ())
db_invariants_CEL[(27, 392, 1)] = ((27, 4), (99, 294, 196), (38, 42, 9, 6, 3, 0, 1), (0, 0, 0, 0, 1, 0, 5, 4, 2, 0, 7, 4, 4), (8, 5), (4, 2), (1, 13, 13))
db_invariants_CEL[(27, 392, 2)] = ((27, 3), (99, 294, 196), (39, 40, 10, 6, 2, 2), (0, 0, 0, 0, 1, 0, 6, 2, 2, 2, 5, 6, 3), (4, 2), (2, 1), (1, 13, 13))
db_invariants_CEL[(27, 392, 3)] = ((27, 2), (99, 294, 196), (39, 40, 10, 6, 2, 2), (0, 0, 0, 0, 1, 0, 5, 4, 1, 2, 4, 8, 2), (4, 2), (2, 1), (1, 13, 13))
db_invariants_CEL[(27, 400, 1)] = ((27, 1), (101, 300, 200), (40, 40, 6, 14, 1), (0, 0, 0, 0, 0, 0, 0, 0, 8, 8, 11), (8, 5), (8, 3), ())
db_invariants_CEL[(28, 420, 1)] = ((28, 4), (106, 315, 210), (41, 44, 11, 6, 2, 1, 1), (0, 0, 0, 0, 1, 0, 4, 4, 2, 1, 4, 6, 6), (2, 1), (1, 1), (1, 13, 14))
db_invariants_CEL[(28, 420, 2)] = ((28, 5), (106, 315, 210), (42, 42, 12, 6, 1, 3), (0, 0, 0, 0, 1, 0, 4, 4, 1, 3, 1, 10, 4), (4, 2), (2, 1), (1, 13, 14))
db_invariants_CEL[(28, 420, 3)] = ((28, 6), (106, 315, 210), (42, 42, 12, 6, 1, 3), (0, 0, 0, 0, 1, 0, 6, 0, 3, 3, 3, 6, 6), (12, 4), (6, 1), (1, 13, 14))
db_invariants_CEL[(28, 420, 4)] = ((28, 1), (106, 315, 210), (14, 91, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 7, 7, 0, 0, 0, 0, 14), (56, 12), (84, 7), (1, 13, 14))
db_invariants_CEL[(28, 420, 5)] = ((28, 3), (106, 315, 210), (45, 40, 3, 15, 3), (0, 0, 0, 0, 0, 0, 0, 0, 6, 9, 13), (12, 4), (6, 1), (1, 13, 14))
db_invariants_CEL[(28, 420, 6)] = ((28, 2), (106, 315, 210), (45, 40, 3, 15, 3), (0, 0, 0, 0, 0, 0, 0, 0, 6, 9, 13), (12, 4), (6, 1), (1, 13, 14))
db_invariants_CEL[(29, 440, 1)] = ((29, 2), (111, 330, 220), (50, 40, 1, 14, 6), (0, 0, 0, 0, 0, 0, 0, 0, 5, 8, 16), (8, 5), (8, 3), ())
db_invariants_CEL[(29, 448, 1)] = ((29, 3), (113, 336, 224), (44, 46, 13, 6, 2, 0, 2), (0, 0, 0, 0, 1, 0, 3, 4, 3, 0, 4, 4, 10), (8, 5), (4, 2), (1, 13, 15))
db_invariants_CEL[(29, 448, 2)] = ((29, 4), (113, 336, 224), (45, 44, 14, 6, 1, 2, 1), (0, 0, 0, 0, 1, 0, 3, 4, 2, 2, 1, 8, 8), (4, 2), (2, 1), (1, 13, 15))
db_invariants_CEL[(29, 448, 3)] = ((29, 5), (113, 336, 224), (45, 44, 14, 6, 1, 2, 1), (0, 0, 0, 0, 1, 0, 4, 2, 3, 2, 2, 6, 9), (4, 2), (2, 1), (1, 13, 15))
db_invariants_CEL[(29, 448, 4)] = ((29, 1), (113, 336, 224), (21, 84, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 7, 0, 7, 0, 0, 0, 15), (56, 12), (84, 7), (1, 13, 15))
db_invariants_CEL[(30, 460, 1)] = ((30, 2), (116, 345, 230), (55, 40, 0, 11, 10), (0, 0, 0, 0, 0, 0, 0, 0, 5, 5, 20), (20, 4), (20, 3), ())
db_invariants_CEL[(30, 476, 1)] = ((30, 3), (120, 357, 238), (49, 44, 17, 6, 1, 1, 2), (0, 0, 0, 0, 1, 0, 3, 2, 4, 1, 2, 4, 13), (4, 2), (2, 1), (1, 13, 16))
db_invariants_CEL[(30, 480, 1)] = ((30, 1), (121, 360, 240), (15, 105, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 0, 15, 0, 0, 0, 0, 0, 15), (60, 12), (120, 36), (1, 14, 15))
db_invariants_CEL[(31, 480, 1)] = ((31, 1), (121, 360, 240), (60, 40, 0, 6, 15), (0, 0, 0, 0, 0, 0, 0, 0, 6, 0, 25), (120, 35), (120, 34), (1, 11, 19))
db_invariants_CEL[(31, 504, 1)] = ((31, 2), (127, 378, 252), (54, 42, 21, 6, 1, 0, 3), (0, 0, 0, 0, 1, 0, 0, 0, 9, 0, 6, 0, 15), (24, 14), (12, 4), (1, 13, 17))
db_invariants_CEL[(31, 504, 2)] = ((31, 3), (127, 378, 252), (54, 42, 21, 6, 1, 0, 3), (0, 0, 0, 0, 1, 0, 3, 0, 6, 0, 3, 0, 18), (24, 14), (12, 4), (1, 13, 17))
db_invariants_CEL[(32, 544, 1)] = ((32, 1), (137, 408, 272), (16, 120, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 0, 0, 16), (64, 186), (128, 913), (1, 15, 16))
db_invariants_CEL[(33, 576, 1)] = ((33, 1), (145, 432, 288), (24, 112, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 0, 0, 17), (64, 186), (128, 913), (1, 15, 17))
db_invariants_CEL[(34, 612, 1)] = ((34, 2), (154, 459, 306), (60, 63, 18, 6, 4, 0, 3), (0, 0, 0, 0, 0, 0, 3, 3, 3, 0, 4, 0, 6, 0, 9, 6), (12, 4), (6, 1), (1, 16, 17))
db_invariants_CEL[(34, 612, 2)] = ((34, 1), (154, 459, 306), (17, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 0, 0, 17, 0, 0, 0, 0, 0, 0, 17), (68, 4), (272, 50), (1, 16, 17))
db_invariants_CEL[(35, 680, 1)] = ((35, 1), (171, 510, 340), (70, 55, 25, 21), (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 25, 0, 10), (20, 4), (20, 4), ())
db_invariants_CEL[(36, 684, 1)] = ((36, 1), (172, 513, 342), (18, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 0, 0, 9, 9, 0, 0, 0, 0, 0, 0, 18), (72, 17), (108, 26), (1, 17, 18))
db_invariants_CEL[(37, 720, 1)] = ((37, 3), (181, 540, 360), (72, 72, 24, 0, 10, 0, 3), (0, 0, 0, 0, 0, 0, 3, 0, 6, 0, 4, 0, 0, 0, 12, 0, 12), (48, 48), (24, 12), (1, 17, 19))
db_invariants_CEL[(37, 720, 2)] = ((37, 1), (181, 540, 360), (27, 144, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 0, 0, 9, 0, 9, 0, 0, 0, 0, 0, 19), (72, 17), (108, 26), (1, 17, 19))
db_invariants_CEL[(37, 720, 3)] = ((37, 2), (181, 540, 360), (72, 72, 12, 24, 0, 0, 0, 0, 0, 0, 1), (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 13, 0, 0, 0, 24), (48, 36), (48, 17), (1, 17, 19))

# This is the list of wiring diagrams of the hyperplane arrangements listed in [CEL]_ Table 15.
# The first entry is the permutation and the second is the list of inversions with labels given by the order of the normals in the above dictionary.

db_wiring_CEL = {}
db_wiring_CEL[(6, 24, 1)] = ((1, 6, 2, 5, 3, 4), ((1, 2), (4, 5), (2, 4), (4, 6), (3, 4), (1, 3), (3, 5)))
db_wiring_CEL[(7, 32, 1)] = ((1, 6, 7, 2, 5, 3, 4), ((1, 3), (5, 6), (3, 5), (5, 7), (2, 3), (3, 5), (5, 6), (1, 3), (3, 5)))
db_wiring_CEL[(8, 40, 1)] = ((4, 7, 1, 2, 8, 6, 3, 5), ((5, 7), (1, 2), (2, 5), (5, 6), (6, 8), (4, 6), (3, 4), (1, 3), (3, 5), (5, 7), (4, 5)))
db_wiring_CEL[(9, 48, 1)] = ((7, 4, 6, 1, 8, 2, 9, 3, 5), ((5, 7), (4, 5), (2, 4), (7, 8), (4, 7), (3, 4), (7, 9), (6, 7), (4, 6), (1, 4), (4, 5), (5, 8), (8, 9)))
db_wiring_CEL[(10, 60, 1)] = ((10, 1, 8, 9, 2, 7, 3, 6, 4, 5), ((1, 4), (4, 5), (5, 7), (7, 9), (9, 10), (6, 7), (3, 6), (6, 9), (2, 3), (5, 6), (3, 5), (5, 7), (7, 8), (1, 3), (3, 5), (5, 7)))
db_wiring_CEL[(10, 60, 2)] = ((4, 8, 7, 1, 9, 2, 10, 6, 3, 5), ((1, 2), (5, 7), (4, 5), (7, 9), (2, 4), (4, 7), (7, 8), (8, 10), (3, 4), (6, 8), (4, 6), (6, 7), (1, 4), (4, 6), (6, 9), (3, 4)))
db_wiring_CEL[(10, 60, 3)] = ((3, 7, 6, 10, 2, 8, 5, 9, 1, 4), ((5, 7), (4, 5), (9, 10), (7, 9), (5, 7), (1, 5), (5, 6), (6, 8), (4, 6), (3, 4), (8, 10), (6, 8), (4, 6), (2, 4), (8, 9), (6, 8)))
db_wiring_CEL[(11, 72, 1)] = ((9, 8, 1, 10, 2, 7, 11, 3, 6, 4, 5), ((6, 7), (4, 6), (6, 8), (3, 4), (8, 10), (7, 8), (4, 7), (2, 4), (10, 11), (7, 10), (6, 7), (4, 6), (3, 4), (6, 8), (8, 9), (1, 3), (3, 6), (6, 8), (8, 11)))
db_wiring_CEL[(12, 84, 1)] = ((4, 12, 5, 8, 1, 10, 2, 11, 3, 7, 9, 6), ((6, 8), (2, 4), (4, 6), (6, 7), (7, 9), (9, 10), (3, 4), (1, 3), (3, 7), (7, 9), (9, 12), (6, 7), (7, 9), (5, 7), (4, 5), (2, 4), (1, 2), (4, 6), (9, 10), (6, 9), (5, 6), (2, 5)))
db_wiring_CEL[(12, 84, 2)] = ((4, 8, 1, 10, 11, 2, 7, 12, 3, 6, 9, 5), ((4, 5), (7, 8), (5, 7), (3, 5), (7, 9), (1, 3), (3, 4), (4, 7), (7, 8), (8, 10), (10, 12), (9, 10), (6, 9), (5, 6), (2, 5), (1, 2), (5, 7), (7, 8), (4, 5), (8, 11), (5, 8), (2, 5)))
db_wiring_CEL[(12, 84, 3)] = ((2, 11, 3, 12, 4, 8, 6, 5, 10, 7, 1, 9), ((7, 8), (8, 10), (6, 8), (10, 12), (4, 6), (2, 4), (8, 10), (6, 8), (4, 6), (6, 7), (3, 4), (7, 9), (9, 11), (8, 9), (1, 3), (3, 8), (8, 10), (10, 12), (7, 8), (2, 3), (8, 10), (6, 8)))
db_wiring_CEL[(13, 96, 1)] = ((4, 9, 5, 13, 1, 12, 2, 11, 3, 8, 6, 10, 7), ((4, 6), (6, 7), (7, 9), (9, 11), (3, 4), (8, 9), (1, 3), (3, 8), (8, 10), (10, 13), (7, 8), (2, 3), (8, 10), (6, 8), (5, 6), (3, 5), (1, 3), (5, 7), (10, 11), (7, 10), (6, 7), (3, 6), (2, 3), (6, 8), (10, 12)))
db_wiring_CEL[(13, 96, 2)] = ((5, 10, 9, 1, 12, 11, 2, 8, 3, 13, 7, 4, 6), ((6, 7), (7, 10), (5, 7), (10, 12), (7, 8), (4, 5), (8, 10), (1, 2), (2, 4), (4, 8), (8, 9), (9, 11), (11, 13), (3, 4), (10, 11), (7, 10), (6, 7), (4, 6), (6, 8), (1, 4), (8, 9), (4, 6), (6, 8), (8, 12), (3, 4)))
db_wiring_CEL[(13, 96, 3)] = ((4, 13, 8, 1, 10, 11, 2, 7, 12, 3, 6, 9, 5), ((5, 6), (8, 9), (6, 8), (2, 3), (3, 6), (6, 7), (7, 10), (10, 11), (1, 3), (3, 4), (4, 7), (7, 8), (8, 10), (10, 13), (9, 10), (6, 9), (5, 6), (2, 5), (1, 2), (5, 7), (7, 8), (4, 5), (8, 11), (5, 8), (2, 5)))
db_wiring_CEL[(13, 104, 1)] = ((8, 13, 6, 10, 2, 9, 4, 1, 3, 5, 11, 7, 12), ((7, 8), (8, 10), (3, 5), (10, 12), (5, 8), (1, 3), (8, 10), (10, 11), (3, 5), (5, 6), (6, 8), (8, 10), (4, 6), (2, 4), (6, 8), (10, 13), (8, 10), (7, 8), (4, 7), (3, 4), (1, 3), (7, 9), (3, 5), (9, 11), (5, 7), (7, 9), (11, 12)))
db_wiring_CEL[(14, 112, 1)] = ((5, 10, 9, 1, 12, 11, 2, 13, 8, 3, 14, 7, 4, 6), ((6, 8), (1, 2), (5, 6), (8, 11), (6, 8), (4, 6), (8, 9), (11, 13), (9, 11), (2, 4), (4, 5), (5, 9), (9, 10), (10, 12), (12, 14), (11, 12), (3, 5), (8, 11), (7, 8), (5, 7), (4, 5), (7, 9), (9, 10), (1, 4), (4, 7), (7, 9), (9, 13), (6, 7), (3, 4)))
db_wiring_CEL[(14, 112, 2)] = ((5, 1, 4, 7, 3, 13, 12, 14, 11, 8, 10, 9, 6, 2), ((5, 6), (6, 8), (8, 10), (10, 12), (4, 6), (6, 8), (8, 10), (10, 11), (3, 4), (4, 6), (6, 8), (8, 10), (2, 4), (4, 6), (6, 8), (8, 9), (1, 2), (2, 4), (4, 6), (6, 8), (8, 14), (7, 8), (5, 7), (3, 5), (1, 3), (7, 9), (5, 7), (3, 5), (2, 3)))
db_wiring_CEL[(14, 112, 3)] = ((1, 4, 5, 10, 9, 8, 14, 3, 11, 7, 13, 2, 12, 6), ((8, 10), (7, 8), (12, 14), (10, 12), (8, 10), (3, 8), (1, 3), (12, 13), (8, 9), (9, 12), (3, 4), (7, 9), (6, 7), (9, 10), (4, 6), (2, 4), (6, 9), (9, 11), (4, 6), (11, 14), (6, 7), (10, 11), (7, 10), (5, 7), (10, 12), (3, 5), (9, 10), (12, 13), (7, 9)))
db_wiring_CEL[(14, 116, 1)] = ((5, 3, 1, 4, 9, 2, 10, 14, 6, 8, 11, 13, 7, 12), ((8, 9), (6, 8), (3, 4), (8, 10), (10, 11), (11, 13), (1, 3), (3, 6), (6, 8), (8, 11), (11, 12), (12, 14), (5, 6), (6, 8), (4, 6), (8, 9), (9, 12), (6, 9), (2, 4), (4, 6), (9, 10), (6, 7), (3, 4), (7, 9), (9, 11), (1, 3), (3, 7), (11, 13), (7, 9), (9, 11)))
db_wiring_CEL[(15, 120, 1)] = ((6, 4, 10, 15, 2, 9, 14, 11, 8, 13, 3, 1, 12, 7, 5), ((11, 15), (5, 6), (10, 11), (6, 10), (4, 6), (10, 12), (3, 4), (6, 7), (9, 10), (12, 13), (7, 9), (1, 3), (3, 7), (7, 8), (8, 12), (12, 14), (14, 15), (6, 8), (11, 12), (8, 9), (5, 6), (2, 3), (9, 11), (3, 5), (5, 9), (9, 10), (4, 5), (10, 14), (8, 10), (7, 8), (5, 7)))
db_wiring_CEL[(15, 128, 1)] = ((15, 5, 10, 9, 1, 12, 13, 2, 11, 8, 3, 14, 7, 4, 6), ((1, 3), (6, 7), (9, 12), (7, 9), (3, 4), (4, 7), (7, 8), (8, 10), (10, 11), (11, 14), (14, 15), (9, 11), (2, 4), (4, 5), (5, 9), (9, 10), (10, 12), (12, 14), (11, 12), (3, 5), (8, 11), (7, 8), (5, 7), (4, 5), (7, 9), (9, 10), (1, 4), (4, 7), (7, 9), (9, 13), (6, 7), (3, 4), (7, 9)))
db_wiring_CEL[(15, 128, 2)] = ((6, 4, 2, 3, 10, 5, 11, 15, 7, 9, 12, 14, 1, 8, 13), ((4, 6), (8, 9), (13, 14), (6, 8), (3, 4), (8, 10), (10, 11), (11, 13), (1, 3), (3, 6), (6, 8), (8, 11), (11, 12), (12, 15), (5, 6), (6, 8), (4, 6), (8, 9), (9, 12), (2, 4), (6, 9), (12, 13), (4, 6), (9, 10), (6, 7), (3, 4), (7, 9), (9, 12), (1, 3), (8, 9), (3, 8), (12, 14), (8, 10)))
db_wiring_CEL[(15, 132, 1)] = ((1, 4, 14, 15, 7, 9, 13, 3, 12, 2, 10, 6, 11, 8, 5), ((7, 8), (5, 7), (7, 9), (11, 12), (3, 5), (5, 7), (9, 11), (7, 9), (9, 10), (1, 3), (3, 5), (5, 7), (7, 9), (9, 13), (4, 5), (2, 4), (8, 9), (13, 15), (4, 8), (8, 10), (7, 8), (12, 13), (10, 12), (8, 10), (6, 8), (5, 6), (3, 5), (1, 3), (3, 4), (12, 14), (10, 12), (8, 10), (4, 8), (2, 4)))
db_wiring_CEL[(15, 132, 2)] = ((6, 7, 12, 5, 13, 10, 2, 11, 9, 1, 14, 3, 15, 8, 4), ((4, 6), (6, 9), (9, 11), (11, 14), (5, 6), (10, 11), (8, 10), (6, 8), (3, 6), (10, 12), (6, 7), (2, 3), (7, 10), (14, 15), (10, 11), (5, 7), (11, 14), (9, 11), (7, 9), (6, 7), (3, 6), (11, 12), (6, 8), (8, 11), (11, 13), (7, 8), (1, 3), (3, 4), (4, 7), (7, 9), (9, 11), (11, 12), (12, 15), (6, 7)))
db_wiring_CEL[(16, 140, 1)] = ((7, 5, 11, 16, 3, 4, 15, 10, 12, 14, 9, 1, 13, 8, 6, 2), ((6, 8), (8, 9), (9, 11), (11, 15), (5, 6), (10, 11), (6, 10), (4, 6), (10, 12), (3, 4), (6, 7), (9, 10), (12, 13), (7, 9), (1, 3), (3, 7), (7, 8), (8, 12), (12, 14), (6, 8), (14, 16), (13, 14), (11, 13), (10, 11), (8, 10), (7, 8), (5, 7), (4, 5), (2, 4), (1, 2), (10, 12), (4, 6), (6, 10), (10, 11), (5, 6), (11, 15)))
db_wiring_CEL[(16, 144, 1)] = ((11, 5, 16, 10, 1, 14, 13, 2, 12, 9, 3, 15, 6, 8, 4, 7), ((9, 12), (3, 4), (7, 9), (4, 7), (12, 13), (7, 8), (8, 10), (10, 12), (2, 4), (9, 10), (12, 15), (4, 5), (10, 12), (5, 10), (15, 16), (10, 11), (11, 13), (3, 5), (13, 15), (12, 13), (5, 6), (9, 12), (8, 9), (6, 8), (4, 6), (8, 10), (10, 11), (1, 4), (4, 5), (5, 8), (8, 10), (10, 14), (14, 16), (7, 8), (8, 10), (3, 5), (10, 11)))
db_wiring_CEL[(16, 144, 2)] = ((16, 8, 4, 13, 5, 7, 14, 1, 11, 12, 10, 2, 9, 3, 15, 6), ((4, 6), (6, 10), (10, 12), (12, 15), (11, 12), (5, 6), (9, 11), (15, 16), (8, 9), (6, 8), (11, 13), (3, 6), (6, 7), (7, 11), (11, 12), (12, 15), (5, 7), (10, 12), (2, 3), (9, 10), (7, 9), (6, 7), (12, 13), (3, 6), (6, 8), (8, 12), (12, 14), (7, 8), (1, 3), (3, 4), (4, 7), (7, 9), (9, 10), (10, 12), (12, 13), (13, 16), (6, 7)))
db_wiring_CEL[(16, 144, 3)] = ((9, 8, 7, 6, 11, 13, 5, 15, 4, 16, 3, 14, 2, 12, 1, 10), ((1, 2), (4, 5), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (13, 14), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (1, 3), (11, 12), (9, 11), (7, 9), (9, 10), (5, 7), (7, 9), (3, 5), (5, 7), (7, 8), (8, 15), (6, 8), (4, 6), (8, 9), (2, 4), (6, 8), (8, 10), (4, 6), (10, 11), (6, 8), (8, 10), (10, 12)))
db_wiring_CEL[(16, 144, 4)] = ((11, 1, 4, 15, 7, 5, 16, 9, 6, 8, 13, 10, 2, 12, 14, 3), ((4, 5), (8, 9), (1, 4), (9, 12), (12, 14), (4, 6), (11, 12), (6, 9), (14, 15), (9, 11), (5, 6), (8, 9), (6, 8), (11, 14), (3, 6), (8, 11), (6, 8), (2, 3), (5, 6), (8, 9), (11, 12), (14, 16), (9, 11), (3, 5), (11, 14), (10, 11), (5, 10), (4, 5), (10, 12), (1, 4), (14, 15), (9, 10), (12, 14), (4, 6), (6, 7), (7, 9), (9, 12)))
db_wiring_CEL[(16, 144, 5)] = ((10, 15, 4, 1, 12, 7, 14, 9, 3, 11, 6, 5, 8, 13, 16, 2), ((1, 3), (6, 7), (7, 9), (12, 13), (9, 10), (10, 12), (8, 10), (3, 4), (12, 14), (4, 8), (14, 15), (8, 9), (9, 12), (12, 14), (7, 9), (2, 4), (6, 7), (14, 16), (11, 12), (9, 11), (4, 6), (6, 9), (3, 4), (9, 10), (10, 14), (4, 6), (8, 10), (6, 8), (5, 6), (8, 9), (14, 15), (9, 11), (11, 12), (1, 5), (5, 9), (9, 11), (11, 14)))
db_wiring_CEL[(16, 148, 1)] = ((4, 2, 15, 5, 16, 6, 10, 8, 7, 13, 3, 12, 14, 9, 11, 1), ((8, 9), (13, 14), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (2, 3), (11, 12), (12, 15), (9, 12), (7, 9), (5, 7), (3, 5), (9, 10), (7, 9), (12, 13), (9, 12), (5, 7), (7, 9), (15, 16), (12, 15), (11, 12), (9, 11), (8, 9), (6, 8), (4, 6), (6, 7), (11, 13), (1, 4), (13, 14), (4, 6), (6, 11), (11, 13), (13, 16), (10, 11), (11, 13), (9, 11)))
db_wiring_CEL[(17, 160, 1)] = ((5, 17, 6, 11, 10, 1, 14, 13, 2, 15, 3, 9, 16, 4, 8, 12, 7), ((8, 10), (12, 13), (7, 8), (2, 4), (10, 12), (8, 10), (4, 5), (5, 8), (8, 9), (9, 11), (11, 14), (3, 5), (14, 15), (10, 11), (5, 6), (1, 3), (3, 5), (5, 10), (10, 12), (12, 14), (14, 17), (9, 10), (10, 12), (4, 5), (12, 14), (11, 12), (8, 11), (7, 8), (5, 7), (2, 5), (7, 9), (9, 10), (1, 2), (5, 7), (14, 15), (7, 9), (4, 5), (9, 14), (8, 9), (5, 8), (2, 5)))
db_wiring_CEL[(17, 160, 2)] = ((11, 5, 16, 10, 1, 13, 14, 2, 12, 9, 3, 15, 6, 8, 17, 4, 7), ((6, 7), (9, 12), (3, 4), (7, 9), (15, 16), (12, 13), (4, 7), (7, 8), (8, 10), (10, 12), (12, 15), (9, 10), (2, 4), (10, 12), (4, 5), (15, 17), (5, 10), (10, 11), (11, 13), (13, 15), (3, 5), (12, 13), (15, 16), (9, 12), (5, 6), (8, 9), (6, 8), (4, 6), (8, 10), (10, 11), (1, 4), (4, 5), (5, 8), (8, 10), (10, 15), (15, 17), (9, 10), (7, 9), (6, 7), (9, 11), (3, 6)))
db_wiring_CEL[(17, 160, 3)] = ((4, 13, 5, 8, 16, 14, 1, 11, 12, 2, 10, 17, 3, 7, 15, 9, 6), ((2, 5), (12, 13), (13, 15), (5, 9), (9, 10), (10, 13), (13, 14), (8, 10), (4, 5), (7, 8), (10, 11), (5, 7), (1, 2), (2, 5), (5, 6), (6, 10), (10, 13), (13, 17), (9, 10), (12, 13), (10, 12), (8, 10), (7, 8), (4, 7), (3, 4), (7, 9), (12, 14), (1, 3), (6, 7), (9, 12), (7, 9), (14, 15), (3, 7), (12, 14), (7, 8), (8, 10), (10, 12), (2, 3), (9, 10), (12, 13), (6, 9)))
db_wiring_CEL[(17, 160, 4)] = ((10, 8, 9, 7, 11, 6, 14, 5, 16, 4, 17, 3, 13, 15, 2, 12, 1), ((12, 13), (3, 5), (5, 7), (7, 9), (9, 12), (12, 14), (1, 3), (8, 9), (14, 16), (3, 5), (5, 8), (8, 10), (10, 12), (12, 14), (14, 15), (11, 12), (4, 5), (9, 11), (7, 9), (15, 17), (9, 10), (5, 7), (7, 9), (2, 5), (5, 7), (7, 8), (8, 15), (6, 8), (4, 6), (3, 4), (8, 9), (6, 8), (1, 3), (8, 10), (3, 6), (10, 11), (15, 16), (6, 8), (8, 10), (10, 12), (12, 13)))
db_wiring_CEL[(17, 160, 5)] = ((13, 7, 1, 4, 3, 11, 6, 5, 10, 15, 8, 12, 16, 9, 14, 2, 17), ((4, 8), (8, 9), (9, 11), (11, 12), (12, 14), (7, 9), (14, 15), (2, 4), (4, 5), (5, 7), (7, 8), (8, 12), (12, 14), (6, 8), (11, 12), (1, 2), (2, 6), (6, 7), (7, 9), (9, 11), (11, 13), (13, 17), (8, 9), (5, 8), (12, 13), (4, 5), (8, 12), (7, 8), (5, 7), (3, 5), (1, 3), (7, 9), (12, 14), (3, 4), (11, 12), (9, 11), (8, 9), (4, 8), (14, 15), (8, 10), (2, 4)))
db_wiring_CEL[(17, 164, 1)] = ((12, 1, 4, 16, 9, 17, 5, 7, 11, 6, 8, 14, 10, 2, 13, 15, 3), ((5, 8), (8, 10), (10, 13), (4, 5), (13, 15), (15, 16), (9, 10), (1, 4), (12, 13), (4, 6), (6, 9), (9, 12), (12, 15), (5, 6), (8, 9), (6, 8), (11, 12), (3, 6), (8, 11), (6, 8), (2, 3), (5, 6), (8, 9), (11, 13), (9, 11), (3, 5), (15, 17), (11, 12), (12, 15), (10, 12), (5, 10), (4, 5), (1, 4), (10, 11), (11, 13), (4, 6), (9, 11), (6, 7), (15, 16), (7, 9), (13, 15), (9, 10)))
db_wiring_CEL[(17, 168, 1)] = ((10, 4, 16, 5, 13, 17, 2, 7, 9, 1, 6, 8, 14, 15, 11, 12, 3), ((5, 9), (9, 12), (3, 5), (5, 6), (8, 9), (12, 16), (11, 12), (9, 11), (6, 9), (4, 6), (2, 4), (9, 10), (10, 13), (8, 10), (6, 8), (4, 6), (13, 14), (6, 7), (16, 17), (14, 16), (12, 14), (10, 12), (7, 10), (5, 7), (3, 5), (10, 11), (11, 13), (9, 11), (1, 3), (13, 15), (7, 9), (11, 13), (3, 4), (4, 7), (7, 8), (8, 11), (11, 12), (12, 14), (14, 17), (6, 8), (8, 9), (5, 6), (13, 14)))
db_wiring_CEL[(17, 168, 2)] = ((4, 14, 2, 16, 5, 17, 6, 10, 7, 8, 13, 3, 12, 9, 15, 11, 1), ((12, 14), (10, 12), (8, 10), (6, 8), (4, 6), (2, 4), (12, 13), (4, 5), (13, 16), (10, 13), (8, 10), (5, 8), (3, 5), (8, 9), (9, 11), (11, 12), (7, 9), (12, 14), (14, 15), (9, 12), (5, 7), (7, 9), (15, 17), (12, 15), (11, 12), (9, 11), (8, 9), (6, 8), (4, 6), (11, 13), (6, 7), (1, 4), (13, 14), (4, 6), (6, 11), (11, 13), (13, 16), (16, 17), (10, 11), (11, 13), (9, 11), (8, 9), (3, 4)))
db_wiring_CEL[(18, 180, 1)] = ((11, 5, 10, 16, 1, 14, 13, 2, 15, 17, 3, 9, 12, 6, 8, 18, 4, 7), ((7, 9), (4, 7), (9, 10), (15, 16), (2, 4), (7, 9), (13, 15), (12, 13), (9, 12), (8, 9), (6, 8), (12, 14), (4, 6), (11, 12), (3, 4), (6, 7), (14, 17), (12, 14), (7, 12), (4, 7), (12, 13), (13, 15), (17, 18), (7, 8), (15, 17), (14, 15), (11, 14), (10, 11), (8, 10), (6, 8), (5, 6), (10, 12), (12, 13), (1, 5), (5, 7), (7, 10), (10, 12), (12, 16), (16, 18), (9, 10), (6, 7), (10, 12), (15, 16), (12, 13), (4, 6), (6, 10)))
db_wiring_CEL[(18, 180, 2)] = ((18, 9, 4, 14, 5, 8, 15, 1, 12, 13, 11, 2, 16, 10, 7, 3, 17, 6), ((4, 6), (6, 10), (10, 13), (13, 17), (9, 10), (12, 13), (5, 6), (10, 12), (17, 18), (8, 10), (6, 8), (12, 14), (8, 9), (3, 6), (6, 8), (14, 15), (8, 12), (5, 6), (12, 14), (14, 17), (11, 12), (6, 8), (12, 14), (2, 3), (10, 12), (8, 10), (7, 8), (14, 15), (10, 11), (3, 7), (7, 10), (10, 14), (14, 16), (6, 7), (9, 10), (1, 3), (3, 4), (4, 6), (6, 9), (9, 11), (11, 12), (12, 14), (14, 15), (15, 18), (8, 9), (5, 6)))
db_wiring_CEL[(18, 180, 3)] = ((1, 3, 5, 7, 9, 13, 17, 18, 12, 2, 16, 14, 11, 10, 15, 4, 6, 8), ((10, 12), (12, 14), (14, 15), (5, 6), (6, 8), (8, 10), (10, 12), (12, 14), (4, 6), (6, 8), (8, 10), (10, 12), (12, 13), (3, 4), (4, 6), (6, 8), (8, 10), (10, 12), (2, 4), (4, 6), (6, 8), (8, 10), (10, 11), (1, 2), (2, 4), (4, 6), (6, 8), (8, 10), (10, 18), (9, 10), (7, 9), (5, 7), (3, 5), (1, 3), (9, 11), (7, 9), (5, 7), (3, 5), (2, 3), (11, 12), (9, 11), (7, 9), (5, 7), (3, 5), (11, 13), (9, 11)))
db_wiring_CEL[(18, 180, 4)] = ((3, 2, 10, 6, 11, 8, 16, 14, 7, 5, 15, 12, 18, 13, 1, 17, 9, 4), ((13, 15), (11, 13), (7, 11), (5, 7), (3, 5), (11, 12), (7, 8), (15, 16), (12, 15), (10, 12), (8, 10), (5, 8), (4, 5), (12, 13), (8, 9), (15, 17), (13, 15), (9, 13), (7, 9), (5, 7), (13, 14), (9, 10), (17, 18), (14, 17), (12, 14), (10, 12), (7, 10), (6, 7), (14, 15), (10, 11), (1, 6), (6, 8), (8, 10), (10, 14), (14, 16), (16, 18), (9, 10), (13, 14), (5, 6), (6, 9), (9, 11), (11, 13), (13, 16), (16, 17), (8, 9), (12, 13)))
db_wiring_CEL[(18, 180, 5)] = ((14, 8, 1, 5, 6, 12, 3, 4, 11, 16, 9, 13, 17, 7, 10, 15, 2, 18), ((8, 9), (13, 14), (9, 11), (11, 13), (10, 11), (13, 15), (7, 10), (2, 4), (15, 16), (6, 7), (4, 6), (6, 8), (8, 9), (9, 13), (3, 4), (13, 15), (7, 9), (12, 13), (1, 3), (3, 7), (7, 8), (8, 10), (10, 12), (12, 14), (14, 18), (9, 10), (6, 9), (13, 14), (5, 6), (9, 13), (8, 9), (6, 8), (4, 6), (2, 4), (8, 10), (13, 15), (1, 2), (4, 5), (12, 13), (10, 12), (9, 10), (5, 9), (15, 16), (9, 11), (2, 5), (11, 15)))
db_wiring_CEL[(18, 180, 6)] = ((7, 1, 9, 5, 6, 12, 3, 4, 11, 16, 8, 13, 17, 10, 15, 2, 18, 14), ((8, 9), (9, 11), (11, 12), (12, 14), (3, 4), (14, 15), (7, 9), (1, 3), (3, 5), (5, 7), (7, 8), (8, 12), (12, 14), (4, 5), (5, 8), (11, 12), (8, 9), (14, 18), (12, 14), (9, 12), (7, 9), (6, 7), (2, 6), (1, 2), (9, 10), (12, 13), (6, 9), (5, 6), (13, 15), (9, 13), (8, 9), (6, 8), (4, 6), (2, 4), (8, 10), (15, 16), (4, 5), (13, 15), (12, 13), (10, 12), (9, 10), (5, 9), (3, 5), (9, 11), (15, 17), (11, 15)))
db_wiring_CEL[(18, 184, 1)] = ((1, 13, 2, 10, 17, 5, 18, 6, 12, 9, 7, 8, 15, 11, 3, 14, 16, 4), ((10, 12), (4, 6), (9, 10), (6, 9), (9, 11), (11, 14), (5, 6), (14, 16), (16, 17), (10, 11), (2, 5), (13, 14), (5, 7), (7, 10), (10, 13), (13, 16), (6, 7), (9, 10), (7, 9), (12, 13), (4, 7), (9, 12), (7, 9), (1, 2), (2, 4), (4, 5), (5, 7), (7, 8), (8, 10), (10, 11), (11, 14), (14, 15), (9, 11), (3, 5), (15, 18), (11, 12), (12, 15), (10, 12), (5, 10), (4, 5), (1, 4), (10, 11), (11, 13), (4, 6), (9, 11), (6, 7), (15, 16)))
db_wiring_CEL[(18, 184, 2)] = ((12, 5, 7, 13, 10, 14, 3, 1, 15, 11, 8, 18, 16, 9, 2, 17, 4, 6), ((7, 11), (11, 14), (14, 17), (10, 11), (6, 7), (13, 14), (11, 13), (9, 11), (7, 9), (4, 7), (1, 2), (9, 10), (13, 15), (7, 9), (2, 4), (4, 5), (5, 7), (7, 8), (8, 13), (13, 14), (14, 16), (16, 18), (12, 14), (6, 8), (11, 12), (14, 16), (12, 14), (10, 12), (8, 10), (7, 8), (3, 7), (10, 11), (7, 10), (10, 13), (13, 15), (6, 7), (1, 3), (3, 4), (4, 6), (6, 8), (8, 10), (10, 11), (11, 13), (13, 14), (14, 17), (7, 8), (12, 14)))
db_wiring_CEL[(19, 192, 1)] = ((10, 19, 5, 9, 6, 8, 16, 17, 1, 13, 14, 12, 2, 11, 3, 15, 18, 4, 7), ((14, 16), (12, 14), (7, 12), (5, 7), (3, 5), (12, 13), (7, 8), (16, 17), (13, 16), (11, 13), (10, 11), (8, 10), (5, 8), (4, 5), (13, 14), (8, 9), (16, 18), (14, 16), (9, 14), (7, 9), (5, 7), (14, 15), (9, 10), (18, 19), (15, 18), (13, 15), (12, 13), (10, 12), (7, 10), (6, 7), (15, 16), (10, 11), (1, 6), (6, 8), (8, 10), (10, 15), (15, 17), (17, 19), (9, 10), (14, 15), (5, 6), (6, 9), (9, 11), (11, 12), (12, 14), (14, 17), (17, 18), (8, 9), (13, 14)))
db_wiring_CEL[(19, 192, 2)] = ((13, 9, 4, 17, 8, 5, 12, 18, 1, 15, 14, 2, 16, 11, 3, 10, 19, 6, 7), ((11, 13), (4, 5), (5, 7), (16, 17), (7, 11), (11, 12), (6, 7), (12, 16), (10, 12), (9, 10), (7, 9), (3, 7), (12, 13), (16, 18), (7, 8), (15, 16), (13, 15), (2, 3), (8, 13), (18, 19), (6, 8), (5, 6), (13, 14), (3, 5), (8, 9), (14, 18), (12, 14), (11, 12), (9, 11), (5, 9), (14, 15), (9, 10), (4, 5), (10, 14), (14, 16), (8, 10), (16, 17), (7, 8), (13, 14), (10, 11), (1, 4), (4, 7), (7, 10), (10, 13), (13, 16), (16, 19), (9, 10), (6, 7), (12, 13)))
db_wiring_CEL[(19, 200, 1)] = ((18, 10, 5, 9, 6, 15, 16, 1, 13, 14, 2, 12, 3, 17, 11, 8, 4, 19, 7), ((6, 10), (10, 11), (11, 14), (5, 6), (9, 11), (1, 2), (2, 5), (5, 7), (7, 9), (9, 10), (10, 12), (12, 13), (13, 17), (17, 19), (6, 7), (4, 6), (11, 13), (3, 4), (6, 11), (13, 14), (4, 6), (11, 13), (16, 17), (10, 11), (13, 16), (11, 13), (9, 11), (8, 9), (6, 8), (5, 6), (1, 5), (8, 10), (13, 14), (5, 8), (8, 9), (9, 13), (13, 15), (4, 5), (7, 9), (15, 18), (14, 15), (12, 14), (11, 12), (9, 11), (8, 9), (5, 8), (3, 5), (2, 3), (8, 10), (14, 16), (5, 6)))
db_wiring_CEL[(19, 200, 2)] = ((19, 10, 4, 9, 5, 15, 16, 1, 13, 14, 12, 2, 17, 6, 11, 8, 3, 18, 7), ((6, 10), (10, 13), (13, 14), (5, 6), (9, 10), (14, 18), (12, 14), (10, 12), (8, 10), (6, 8), (3, 6), (2, 3), (12, 13), (8, 9), (6, 8), (13, 15), (5, 6), (8, 13), (15, 16), (6, 8), (13, 15), (12, 13), (8, 9), (18, 19), (15, 18), (13, 15), (11, 13), (9, 11), (7, 9), (3, 7), (11, 12), (15, 16), (7, 8), (8, 11), (11, 15), (15, 17), (6, 8), (10, 11), (1, 3), (3, 4), (4, 6), (6, 7), (7, 10), (10, 12), (12, 13), (13, 15), (15, 16), (16, 19), (9, 10), (5, 7), (14, 16)))
db_wiring_CEL[(19, 200, 3)] = ((6, 19, 10, 4, 9, 5, 15, 16, 1, 13, 14, 12, 2, 17, 11, 8, 3, 18, 7), ((7, 11), (11, 14), (14, 18), (10, 11), (6, 7), (13, 14), (11, 13), (9, 11), (7, 9), (4, 7), (9, 10), (13, 15), (7, 9), (1, 2), (2, 4), (4, 5), (5, 7), (7, 8), (8, 13), (13, 14), (14, 16), (16, 17), (17, 19), (12, 14), (6, 8), (11, 12), (14, 17), (12, 14), (10, 12), (8, 10), (7, 8), (3, 7), (10, 11), (14, 15), (7, 10), (1, 3), (10, 14), (6, 7), (14, 16), (9, 10), (3, 4), (4, 6), (6, 9), (9, 11), (11, 12), (12, 14), (14, 15), (15, 18), (8, 9), (5, 6), (13, 15)))
db_wiring_CEL[(19, 200, 4)] = ((15, 8, 1, 10, 5, 6, 13, 3, 4, 12, 17, 9, 14, 7, 18, 11, 16, 2, 19), ((9, 10), (10, 12), (12, 14), (4, 5), (11, 12), (14, 16), (8, 11), (16, 17), (7, 8), (2, 4), (4, 7), (7, 9), (9, 10), (10, 14), (3, 4), (6, 7), (14, 16), (7, 10), (13, 14), (10, 11), (1, 3), (3, 7), (7, 8), (8, 10), (10, 13), (13, 15), (15, 19), (9, 10), (12, 13), (6, 9), (13, 15), (5, 6), (9, 13), (8, 9), (6, 8), (4, 6), (2, 4), (15, 16), (8, 10), (1, 2), (13, 15), (4, 5), (12, 13), (10, 12), (9, 10), (5, 9), (15, 17), (9, 11), (2, 5), (11, 15), (5, 6)))
db_wiring_CEL[(19, 204, 1)] = ((16, 9, 3, 14, 7, 2, 5, 15, 17, 4, 18, 1, 10, 8, 19, 13, 12, 11, 6), ((11, 13), (4, 5), (8, 11), (11, 12), (5, 8), (12, 14), (8, 9), (9, 12), (12, 13), (13, 15), (15, 16), (1, 2), (2, 5), (5, 6), (6, 9), (9, 10), (10, 13), (13, 15), (15, 17), (17, 19), (4, 6), (6, 7), (7, 10), (10, 11), (16, 17), (11, 13), (13, 16), (12, 13), (9, 12), (8, 9), (5, 8), (3, 5), (1, 3), (3, 4), (4, 6), (6, 7), (7, 10), (10, 11), (11, 14), (14, 15), (5, 7), (7, 8), (15, 18), (8, 11), (6, 8), (11, 12), (12, 15), (10, 12), (8, 10), (7, 8), (2, 7), (12, 13)))
db_wiring_CEL[(20, 216, 1)] = ((12, 8, 20, 5, 11, 6, 16, 15, 1, 18, 2, 17, 14, 3, 19, 7, 13, 10, 4, 9), ((6, 7), (12, 15), (15, 16), (10, 12), (3, 6), (6, 8), (8, 10), (10, 11), (11, 13), (13, 15), (15, 19), (7, 8), (12, 13), (19, 20), (5, 7), (13, 15), (2, 3), (3, 5), (5, 6), (6, 13), (13, 14), (14, 16), (16, 17), (17, 19), (12, 14), (4, 6), (11, 12), (14, 17), (12, 14), (6, 7), (10, 12), (9, 10), (7, 9), (14, 15), (5, 7), (9, 11), (1, 5), (5, 6), (6, 9), (9, 10), (10, 14), (14, 16), (16, 18), (8, 10), (13, 14), (4, 6), (10, 11), (6, 8), (14, 16), (18, 20), (8, 10), (3, 4), (10, 14), (4, 6), (9, 10)))
db_wiring_CEL[(20, 220, 1)] = ((19, 11, 5, 10, 6, 16, 17, 1, 14, 15, 2, 13, 3, 18, 7, 12, 9, 4, 20, 8), ((6, 10), (1, 2), (10, 11), (5, 6), (11, 14), (14, 15), (9, 11), (2, 5), (5, 7), (7, 9), (9, 10), (10, 12), (12, 14), (14, 18), (18, 20), (11, 12), (6, 7), (4, 6), (12, 14), (3, 4), (6, 12), (14, 15), (17, 18), (12, 14), (4, 6), (11, 12), (14, 17), (12, 14), (6, 7), (10, 12), (9, 10), (7, 9), (14, 15), (5, 7), (9, 11), (1, 5), (5, 6), (6, 9), (9, 10), (10, 14), (14, 16), (16, 19), (15, 16), (8, 10), (13, 15), (4, 6), (12, 13), (10, 12), (9, 10), (6, 9), (15, 17), (5, 6), (3, 5), (9, 11), (2, 3), (5, 7)))
db_wiring_CEL[(20, 220, 2)] = ((4, 16, 5, 10, 17, 1, 14, 15, 13, 2, 18, 12, 9, 3, 19, 8, 6, 20, 11, 7), ((2, 4), (4, 8), (8, 11), (11, 15), (15, 17), (7, 8), (10, 11), (8, 10), (3, 4), (6, 8), (10, 12), (14, 15), (4, 6), (12, 14), (6, 7), (11, 12), (1, 4), (4, 6), (6, 11), (11, 13), (13, 16), (16, 20), (10, 11), (5, 6), (11, 13), (3, 5), (15, 16), (13, 15), (12, 13), (9, 12), (8, 9), (5, 8), (4, 5), (2, 4), (1, 2), (8, 10), (10, 11), (7, 8), (15, 17), (11, 15), (8, 11), (4, 8), (2, 4), (11, 12), (17, 18), (8, 9), (15, 17), (14, 15), (12, 14), (9, 12), (7, 9), (6, 7), (4, 6), (12, 13), (9, 10), (3, 4)))
db_wiring_CEL[(20, 220, 3)] = ((10, 1, 9, 5, 8, 4, 18, 19, 17, 20, 16, 7, 15, 11, 14, 12, 13, 3, 6, 2), ((12, 14), (14, 16), (16, 17), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (5, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 15), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (3, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 13), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (1, 2), (2, 4), (4, 6), (6, 8), (8, 10), (10, 11), (11, 20), (9, 11), (7, 9), (5, 7), (3, 5), (1, 3), (11, 12), (9, 11), (7, 9), (5, 7), (3, 5), (2, 3), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (13, 14), (11, 13), (9, 11)))
db_wiring_CEL[(20, 220, 4)] = ((16, 9, 3, 1, 11, 6, 7, 14, 4, 5, 13, 18, 10, 15, 19, 8, 12, 17, 2, 20), ((15, 16), (10, 11), (11, 13), (13, 15), (5, 6), (12, 13), (2, 3), (15, 17), (9, 12), (17, 18), (3, 5), (8, 9), (5, 8), (8, 10), (10, 11), (11, 15), (4, 5), (7, 8), (15, 17), (8, 11), (14, 15), (11, 12), (1, 4), (4, 8), (8, 9), (9, 11), (11, 14), (14, 16), (16, 20), (10, 11), (13, 14), (7, 10), (14, 16), (6, 7), (3, 4), (10, 14), (9, 10), (7, 9), (4, 7), (2, 4), (7, 8), (16, 17), (8, 11), (1, 2), (14, 16), (11, 12), (4, 5), (12, 14), (10, 12), (9, 10), (14, 15), (5, 9), (15, 18), (9, 11), (2, 5), (11, 15)))
db_wiring_CEL[(21, 240, 1)] = ((20, 8, 12, 5, 11, 6, 16, 15, 1, 18, 17, 2, 14, 3, 19, 7, 13, 10, 4, 21, 9), ((11, 12), (1, 3), (6, 7), (12, 15), (15, 16), (10, 12), (3, 6), (6, 8), (8, 10), (10, 11), (11, 13), (13, 15), (15, 19), (19, 21), (12, 13), (7, 8), (5, 7), (13, 15), (2, 3), (3, 5), (5, 6), (6, 13), (13, 14), (14, 16), (16, 17), (17, 19), (19, 20), (12, 14), (4, 6), (11, 12), (14, 17), (12, 14), (6, 7), (10, 12), (9, 10), (7, 9), (14, 15), (5, 7), (9, 11), (1, 5), (5, 6), (6, 9), (9, 10), (10, 14), (14, 16), (16, 19), (15, 16), (8, 10), (13, 15), (4, 6), (12, 13), (10, 12), (9, 10), (6, 9), (15, 17), (5, 6), (3, 5), (9, 11), (2, 3), (5, 7), (11, 15)))
db_wiring_CEL[(21, 240, 2)] = ((4, 20, 16, 5, 10, 17, 1, 14, 15, 13, 2, 18, 12, 9, 3, 19, 8, 6, 21, 11, 7), ((3, 5), (5, 9), (9, 12), (12, 16), (16, 18), (8, 9), (11, 12), (9, 11), (2, 3), (3, 5), (5, 6), (6, 9), (9, 10), (10, 13), (13, 14), (14, 16), (16, 17), (4, 6), (12, 14), (6, 7), (11, 12), (1, 4), (4, 6), (6, 11), (11, 13), (13, 16), (16, 21), (10, 11), (5, 6), (11, 13), (3, 5), (15, 16), (13, 15), (12, 13), (9, 12), (8, 9), (5, 8), (4, 5), (2, 4), (1, 2), (8, 10), (10, 11), (7, 8), (15, 17), (11, 15), (8, 11), (4, 8), (2, 4), (11, 12), (17, 18), (8, 9), (15, 17), (14, 15), (12, 14), (9, 12), (7, 9), (6, 7), (4, 6), (12, 13), (3, 4), (9, 10)))
db_wiring_CEL[(21, 240, 3)] = ((11, 5, 20, 10, 16, 1, 14, 13, 2, 21, 12, 9, 3, 17, 15, 6, 8, 19, 18, 4, 7), ((10, 15), (2, 3), (18, 19), (8, 10), (15, 16), (5, 8), (16, 18), (10, 11), (14, 16), (8, 10), (13, 14), (3, 5), (10, 13), (18, 20), (9, 10), (13, 15), (7, 9), (12, 13), (5, 7), (15, 18), (7, 8), (4, 5), (13, 15), (8, 13), (20, 21), (13, 14), (5, 8), (14, 16), (18, 20), (16, 18), (8, 9), (15, 16), (12, 15), (11, 12), (9, 11), (18, 19), (7, 9), (6, 7), (11, 13), (13, 14), (1, 6), (6, 8), (8, 11), (11, 13), (13, 18), (18, 21), (12, 13), (10, 12), (5, 6), (6, 8), (8, 10), (17, 18), (10, 11), (11, 14), (7, 8), (14, 15), (15, 17), (4, 7), (13, 15), (17, 19), (12, 13)))
db_wiring_CEL[(21, 240, 4)] = ((10, 1, 9, 5, 8, 4, 18, 19, 17, 20, 16, 21, 11, 15, 12, 14, 7, 13, 3, 6, 2), ((17, 18), (6, 8), (8, 10), (10, 13), (13, 15), (15, 17), (5, 6), (6, 8), (8, 10), (10, 11), (11, 13), (13, 15), (15, 16), (4, 6), (6, 8), (8, 11), (11, 13), (13, 15), (3, 4), (4, 6), (6, 8), (8, 9), (9, 11), (11, 13), (13, 14), (2, 4), (4, 6), (6, 9), (9, 11), (11, 13), (1, 2), (2, 4), (4, 6), (6, 7), (7, 9), (9, 11), (11, 12), (12, 21), (10, 12), (8, 10), (5, 8), (3, 5), (1, 3), (12, 13), (10, 12), (8, 10), (7, 8), (5, 7), (3, 5), (2, 3), (12, 14), (10, 12), (7, 10), (5, 7), (3, 5), (14, 15), (12, 14), (10, 12), (9, 10), (7, 9), (5, 7)))
db_wiring_CEL[(21, 240, 5)] = ((10, 4, 1, 12, 6, 5, 15, 8, 7, 14, 19, 11, 2, 16, 20, 9, 13, 18, 3, 21, 17), ((5, 9), (15, 16), (9, 10), (10, 12), (12, 15), (15, 17), (11, 12), (4, 5), (17, 18), (8, 11), (1, 2), (14, 15), (7, 8), (2, 4), (4, 7), (7, 9), (9, 10), (10, 14), (14, 17), (6, 7), (3, 4), (7, 10), (13, 14), (10, 11), (17, 21), (16, 17), (14, 16), (11, 14), (9, 11), (8, 9), (4, 8), (1, 4), (11, 12), (14, 15), (8, 11), (4, 5), (7, 8), (15, 18), (11, 15), (10, 11), (8, 10), (5, 8), (3, 5), (8, 9), (15, 16), (2, 3), (9, 12), (18, 19), (5, 6), (12, 13), (16, 18), (13, 16), (11, 13), (10, 11), (16, 17), (6, 10), (13, 14), (3, 6), (10, 13), (17, 20), (9, 10)))
db_wiring_CEL[(21, 248, 1)] = ((14, 17, 4, 19, 21, 6, 3, 12, 2, 5, 10, 8, 20, 16, 1, 18, 15, 11, 7, 13, 9), ((15, 16), (4, 5), (13, 15), (12, 13), (8, 12), (7, 8), (5, 7), (12, 14), (2, 5), (14, 17), (5, 6), (13, 14), (17, 19), (6, 9), (9, 10), (16, 17), (10, 13), (19, 20), (8, 10), (13, 16), (10, 11), (7, 8), (11, 13), (8, 11), (16, 19), (13, 14), (4, 8), (1, 2), (14, 16), (11, 14), (10, 11), (16, 17), (8, 10), (7, 8), (2, 4), (4, 5), (5, 7), (7, 9), (9, 12), (12, 13), (13, 16), (16, 18), (8, 9), (18, 21), (11, 13), (3, 5), (13, 14), (9, 11), (17, 18), (14, 17), (11, 14), (10, 11), (5, 10), (14, 15), (4, 5), (10, 12), (12, 14), (1, 4), (11, 12), (17, 19), (9, 11), (14, 17), (4, 6)))
db_wiring_CEL[(21, 252, 1)] = ((7, 15, 6, 16, 4, 14, 8, 19, 17, 21, 2, 5, 10, 18, 13, 20, 11, 3, 9, 1, 12), ((15, 17), (8, 10), (17, 18), (14, 15), (18, 21), (15, 18), (13, 15), (10, 13), (9, 10), (6, 9), (4, 6), (1, 4), (9, 11), (13, 14), (6, 7), (11, 13), (7, 9), (13, 16), (9, 11), (18, 19), (11, 13), (13, 14), (4, 7), (16, 18), (7, 9), (14, 16), (9, 11), (16, 17), (11, 14), (6, 7), (10, 11), (14, 16), (7, 10), (16, 20), (5, 7), (15, 16), (3, 5), (10, 12), (5, 6), (6, 8), (12, 15), (11, 12), (8, 11), (11, 13), (2, 3), (13, 14), (7, 8), (10, 11), (14, 17), (3, 7), (20, 21), (17, 18), (7, 10), (10, 14), (14, 15), (18, 20), (9, 10), (15, 18), (6, 7), (13, 15), (12, 13), (10, 12), (18, 19), (7, 10)))
db_wiring_CEL[(22, 264, 1)] = ((4, 16, 5, 10, 21, 17, 1, 14, 13, 15, 2, 12, 18, 9, 3, 20, 19, 8, 22, 6, 11, 7), ((19, 20), (9, 10), (12, 13), (16, 17), (2, 5), (5, 9), (9, 12), (12, 16), (16, 19), (8, 9), (11, 12), (9, 11), (4, 5), (7, 9), (11, 13), (15, 16), (5, 7), (13, 15), (7, 8), (12, 13), (1, 2), (2, 5), (5, 7), (7, 12), (12, 14), (14, 17), (17, 18), (18, 22), (11, 12), (6, 7), (12, 14), (4, 6), (16, 18), (14, 16), (13, 14), (10, 13), (9, 10), (6, 9), (5, 6), (3, 5), (1, 3), (16, 17), (9, 11), (3, 4), (11, 12), (8, 9), (17, 19), (12, 17), (9, 12), (4, 9), (2, 4), (12, 13), (9, 10), (10, 12), (12, 14), (8, 10), (19, 20), (14, 15), (17, 19), (7, 8), (15, 17), (13, 15), (10, 13), (8, 10), (6, 8), (17, 18), (4, 6)))
db_wiring_CEL[(22, 264, 2)] = ((4, 1, 9, 6, 3, 11, 8, 21, 16, 15, 22, 20, 17, 14, 12, 19, 18, 13, 5, 2, 10, 7), ((7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 18), (6, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 16), (4, 5), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (3, 5), (5, 7), (7, 9), (9, 11), (11, 13), (13, 14), (2, 3), (3, 5), (5, 7), (7, 9), (9, 11), (11, 13), (1, 3), (3, 5), (5, 7), (7, 9), (9, 11), (11, 12), (12, 22), (10, 12), (8, 10), (6, 8), (4, 6), (2, 4), (1, 2), (12, 13), (10, 12), (8, 10), (6, 8), (4, 6), (2, 4), (12, 14), (10, 12), (8, 10), (6, 8), (4, 6), (3, 4), (14, 15), (12, 14), (10, 12), (8, 10), (6, 8), (4, 6)))
db_wiring_CEL[(22, 264, 3)] = ((11, 22, 16, 4, 19, 8, 2, 10, 5, 6, 14, 3, 21, 13, 9, 1, 18, 15, 12, 20, 7, 17), ((20, 21), (3, 4), (16, 20), (6, 7), (12, 16), (11, 12), (7, 11), (16, 17), (11, 13), (13, 14), (14, 16), (16, 18), (10, 11), (18, 19), (1, 3), (3, 7), (7, 8), (8, 10), (10, 14), (14, 16), (16, 18), (18, 22), (13, 14), (6, 8), (17, 18), (8, 10), (10, 11), (11, 13), (13, 17), (5, 6), (17, 19), (6, 8), (12, 13), (19, 20), (8, 12), (16, 17), (12, 14), (7, 8), (2, 3), (14, 16), (3, 7), (7, 9), (9, 10), (10, 12), (12, 14), (14, 15), (15, 19), (11, 12), (6, 7), (19, 21), (7, 11), (11, 15), (5, 7), (15, 16), (10, 11), (1, 3), (3, 5), (5, 6), (6, 8), (8, 10), (10, 12), (12, 13), (13, 15), (15, 17), (17, 19), (7, 8), (11, 13)))
db_wiring_CEL[(22, 276, 1)] = ((18, 11, 4, 16, 3, 9, 2, 5, 7, 21, 17, 19, 6, 12, 1, 20, 10, 22, 15, 14, 13, 8), ((11, 14), (4, 6), (10, 11), (14, 15), (6, 10), (15, 17), (10, 12), (12, 15), (15, 16), (5, 6), (16, 18), (11, 12), (18, 19), (1, 2), (2, 5), (5, 7), (7, 8), (8, 11), (11, 13), (13, 16), (16, 18), (18, 20), (20, 22), (6, 8), (12, 13), (8, 9), (4, 6), (6, 8), (8, 12), (12, 14), (14, 16), (11, 12), (7, 8), (12, 14), (19, 20), (16, 19), (14, 16), (13, 14), (10, 13), (8, 10), (5, 8), (3, 5), (1, 3), (10, 11), (3, 4), (16, 17), (4, 6), (6, 7), (7, 10), (10, 12), (12, 16), (5, 7), (7, 8), (16, 18), (11, 12), (8, 11), (6, 8), (11, 13), (13, 14), (18, 21), (17, 18), (14, 17), (12, 14), (10, 12), (8, 10), (7, 8), (2, 7), (14, 15), (12, 14), (7, 9)))
db_wiring_CEL[(22, 288, 1)] = ((22, 21, 14, 8, 20, 2, 11, 4, 19, 6, 10, 1, 18, 9, 7, 5, 13, 12, 15, 16, 17, 3), ((5, 7), (9, 11), (18, 20), (7, 9), (16, 18), (15, 16), (13, 15), (11, 13), (9, 11), (8, 9), (6, 8), (15, 17), (4, 6), (13, 15), (17, 19), (15, 17), (11, 13), (2, 4), (13, 15), (12, 13), (19, 21), (17, 19), (15, 17), (13, 15), (19, 20), (8, 13), (17, 19), (6, 8), (4, 6), (15, 17), (8, 9), (6, 8), (1, 2), (8, 10), (13, 15), (12, 13), (10, 12), (12, 14), (2, 4), (14, 16), (16, 18), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 17), (17, 22), (15, 17), (13, 15), (11, 13), (9, 11), (11, 12), (7, 9), (5, 7), (12, 14), (3, 5), (7, 8), (14, 16), (5, 7), (16, 18), (7, 12), (1, 3), (12, 14), (18, 19), (14, 16), (16, 18), (11, 12), (18, 20), (12, 14), (10, 12), (14, 16)))
db_wiring_CEL[(23, 296, 1)] = ((13, 11, 4, 16, 3, 9, 19, 2, 5, 7, 22, 17, 20, 6, 12, 1, 21, 10, 23, 15, 14, 18, 8), ((12, 15), (11, 12), (4, 6), (15, 16), (16, 18), (6, 11), (11, 13), (13, 16), (16, 17), (17, 19), (10, 11), (19, 20), (5, 6), (11, 13), (13, 14), (20, 22), (18, 20), (14, 18), (12, 14), (9, 12), (8, 9), (6, 8), (3, 6), (8, 10), (14, 15), (18, 19), (10, 11), (6, 8), (8, 10), (2, 3), (10, 14), (22, 23), (14, 16), (16, 18), (13, 14), (9, 10), (14, 16), (18, 22), (16, 18), (15, 16), (12, 15), (10, 12), (7, 10), (5, 7), (3, 5), (12, 13), (18, 19), (5, 6), (6, 8), (8, 9), (9, 12), (12, 14), (14, 18), (7, 9), (9, 10), (18, 20), (13, 14), (20, 21), (10, 13), (8, 10), (13, 15), (15, 16), (1, 3), (3, 8), (8, 9), (9, 11), (11, 13), (13, 15), (15, 18), (18, 20), (20, 23), (14, 15), (12, 14), (7, 9), (17, 18)))
db_wiring_CEL[(23, 304, 1)] = ((23, 22, 21, 14, 8, 20, 2, 11, 4, 19, 6, 10, 1, 18, 9, 7, 5, 13, 12, 15, 16, 17, 3), ((3, 4), (6, 8), (10, 12), (19, 21), (17, 19), (16, 17), (8, 10), (14, 16), (12, 14), (10, 12), (16, 18), (9, 10), (7, 9), (4, 7), (14, 16), (18, 20), (16, 18), (12, 14), (7, 8), (14, 16), (20, 22), (13, 14), (18, 20), (2, 4), (16, 18), (14, 16), (20, 21), (8, 14), (18, 20), (6, 8), (14, 15), (15, 18), (4, 6), (8, 9), (18, 19), (6, 8), (13, 15), (8, 10), (12, 13), (1, 2), (10, 12), (12, 14), (14, 16), (16, 18), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 17), (17, 23), (15, 17), (13, 15), (11, 13), (9, 11), (11, 12), (7, 9), (5, 7), (12, 14), (3, 5), (7, 8), (14, 16), (5, 7), (1, 3), (7, 12), (16, 18), (12, 14), (18, 19), (14, 16), (16, 18), (11, 12), (18, 20), (12, 14), (10, 12), (14, 16)))
db_wiring_CEL[(24, 304, 1)] = ((17, 14, 5, 1, 22, 6, 24, 3, 7, 12, 23, 8, 15, 2, 9, 20, 13, 10, 21, 16, 11, 18, 19, 4), ((19, 20), (16, 17), (11, 13), (17, 19), (7, 11), (13, 17), (5, 7), (11, 13), (7, 8), (4, 5), (10, 11), (13, 14), (8, 10), (19, 23), (17, 19), (16, 17), (14, 16), (10, 14), (9, 10), (5, 9), (3, 5), (2, 3), (14, 15), (19, 20), (15, 19), (9, 11), (5, 6), (13, 15), (11, 13), (10, 11), (19, 21), (13, 14), (8, 10), (18, 19), (14, 16), (6, 8), (21, 22), (16, 18), (8, 9), (15, 16), (22, 24), (18, 22), (16, 18), (9, 16), (7, 9), (3, 7), (16, 17), (22, 23), (17, 19), (9, 10), (19, 20), (15, 17), (7, 9), (20, 22), (1, 3), (9, 11), (14, 15), (11, 12), (12, 14), (14, 16), (6, 7), (10, 12), (16, 20), (20, 21), (15, 16), (3, 4), (4, 6), (6, 10), (10, 11), (11, 15), (15, 17), (17, 18), (18, 20), (20, 24), (9, 11), (14, 15), (11, 12)))
db_wiring_CEL[(24, 312, 1)] = ((12, 1, 11, 6, 10, 5, 9, 22, 21, 23, 20, 24, 19, 4, 18, 13, 17, 14, 16, 15, 8, 3, 7, 2), ((14, 16), (16, 18), (18, 20), (7, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 19), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (5, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 17), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (3, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 15), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (1, 2), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 13), (13, 24), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (1, 3), (13, 14), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (2, 3), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (15, 16), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (4, 5), (15, 17), (13, 15), (11, 13)))
db_wiring_CEL[(24, 316, 1)] = ((19, 12, 4, 17, 8, 3, 20, 10, 2, 5, 7, 23, 18, 21, 6, 22, 1, 13, 11, 24, 16, 15, 14, 9), ((7, 8), (16, 18), (4, 5), (13, 16), (5, 7), (12, 13), (16, 17), (7, 12), (17, 19), (12, 14), (14, 17), (17, 18), (6, 7), (11, 12), (18, 20), (12, 14), (20, 21), (14, 15), (1, 2), (2, 6), (6, 8), (8, 9), (9, 12), (12, 14), (14, 18), (18, 20), (20, 22), (22, 24), (7, 9), (13, 14), (9, 10), (17, 18), (5, 7), (7, 9), (9, 13), (13, 15), (15, 17), (4, 5), (12, 13), (8, 9), (13, 15), (21, 22), (17, 21), (15, 17), (14, 15), (11, 14), (9, 11), (5, 9), (3, 5), (1, 3), (9, 10), (10, 12), (3, 4), (17, 18), (12, 13), (4, 6), (6, 7), (7, 10), (10, 12), (12, 17), (5, 7), (7, 8), (17, 19), (11, 12), (19, 20), (8, 11), (16, 17), (6, 8), (11, 13), (13, 14), (20, 23), (17, 20), (14, 17), (12, 14), (10, 12), (8, 10), (7, 8), (2, 7), (14, 15), (12, 14)))
db_wiring_CEL[(24, 320, 1)] = ((24, 23, 22, 14, 8, 21, 2, 20, 4, 11, 6, 10, 1, 19, 9, 7, 5, 13, 18, 12, 15, 16, 17, 3), ((8, 10), (19, 22), (10, 12), (6, 8), (3, 4), (17, 19), (16, 17), (14, 16), (8, 10), (12, 14), (16, 18), (10, 12), (9, 10), (7, 9), (22, 23), (18, 20), (14, 16), (4, 7), (16, 18), (12, 14), (20, 22), (14, 16), (7, 8), (13, 14), (18, 20), (16, 18), (20, 21), (14, 16), (18, 20), (8, 14), (2, 4), (6, 8), (14, 15), (15, 18), (4, 6), (8, 9), (18, 19), (6, 8), (13, 15), (8, 10), (12, 13), (10, 12), (12, 14), (14, 16), (16, 18), (1, 2), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 17), (17, 24), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (11, 12), (5, 7), (12, 14), (3, 5), (7, 8), (17, 18), (14, 17), (5, 7), (13, 14), (1, 3), (7, 13), (17, 19), (13, 15), (19, 20), (15, 17), (17, 19), (12, 13), (6, 7), (13, 15), (19, 21), (11, 13)))
db_wiring_CEL[(25, 320, 1)] = ((6, 7, 3, 9, 12, 24, 4, 17, 16, 19, 13, 18, 21, 5, 25, 20, 15, 22, 1, 8, 11, 10, 23, 14, 2), ((17, 18), (12, 14), (6, 7), (18, 20), (7, 12), (12, 13), (13, 18), (5, 7), (18, 19), (11, 13), (7, 8), (4, 5), (10, 11), (13, 14), (8, 10), (19, 24), (17, 19), (16, 17), (14, 16), (10, 14), (9, 10), (5, 9), (3, 5), (14, 15), (2, 3), (19, 20), (15, 19), (9, 11), (5, 6), (13, 15), (11, 13), (10, 11), (19, 21), (13, 14), (8, 10), (18, 19), (14, 16), (6, 8), (21, 22), (16, 18), (8, 9), (15, 16), (24, 25), (22, 24), (18, 22), (16, 18), (9, 16), (7, 9), (3, 7), (16, 17), (22, 23), (17, 19), (9, 10), (19, 20), (15, 17), (7, 9), (20, 22), (14, 15), (1, 3), (9, 11), (11, 12), (12, 14), (14, 16), (6, 7), (10, 12), (16, 20), (20, 21), (15, 16), (3, 4), (4, 6), (6, 10), (10, 11), (11, 15), (15, 17), (17, 18), (18, 20), (20, 25), (9, 11), (14, 15), (11, 12), (5, 6)))
db_wiring_CEL[(25, 336, 1)] = ((24, 5, 13, 6, 12, 20, 21, 1, 17, 18, 19, 2, 16, 22, 3, 15, 7, 11, 23, 4, 10, 14, 25, 8, 9), ((6, 11), (4, 6), (13, 14), (18, 19), (16, 18), (11, 13), (1, 4), (10, 11), (13, 16), (23, 24), (11, 13), (16, 17), (4, 5), (5, 7), (7, 8), (8, 11), (11, 12), (12, 14), (14, 16), (16, 20), (20, 21), (21, 23), (13, 14), (23, 25), (6, 8), (14, 16), (8, 9), (19, 21), (18, 19), (16, 18), (3, 6), (15, 16), (6, 8), (8, 15), (15, 17), (17, 20), (7, 8), (20, 23), (5, 7), (14, 15), (15, 17), (4, 5), (19, 20), (23, 24), (17, 19), (7, 9), (16, 17), (13, 16), (9, 10), (12, 13), (19, 21), (10, 12), (2, 4), (8, 10), (12, 14), (14, 15), (4, 8), (8, 9), (9, 12), (12, 14), (14, 19), (19, 20), (11, 12), (20, 23), (7, 9), (3, 4), (12, 14), (6, 7), (18, 20), (9, 12), (14, 15), (17, 18), (20, 21), (15, 17), (7, 9), (12, 15), (11, 12), (15, 16), (9, 11), (8, 9), (1, 3), (3, 8), (8, 10), (10, 13), (13, 15)))
db_wiring_CEL[(25, 336, 2)] = ((22, 5, 11, 6, 23, 10, 18, 19, 1, 15, 16, 14, 25, 2, 13, 3, 20, 9, 17, 21, 4, 8, 12, 24, 7), ((13, 14), (18, 19), (4, 5), (14, 18), (12, 14), (7, 12), (18, 20), (14, 15), (5, 7), (17, 18), (12, 14), (20, 21), (14, 17), (11, 12), (12, 14), (1, 5), (5, 6), (6, 8), (8, 9), (9, 12), (12, 13), (13, 15), (15, 16), (16, 20), (20, 22), (22, 23), (23, 25), (7, 9), (14, 16), (9, 10), (19, 20), (4, 7), (16, 17), (7, 9), (9, 14), (14, 16), (6, 7), (16, 19), (13, 14), (3, 4), (7, 9), (14, 16), (19, 23), (18, 19), (16, 18), (15, 16), (12, 15), (11, 12), (9, 11), (8, 9), (4, 8), (11, 13), (2, 4), (13, 14), (18, 20), (8, 11), (11, 13), (1, 2), (4, 5), (7, 8), (10, 11), (13, 18), (20, 21), (23, 24), (5, 7), (11, 13), (18, 20), (7, 11), (17, 18), (11, 12), (6, 7), (12, 14), (20, 23), (14, 15), (18, 20), (15, 18), (13, 15), (10, 13), (9, 10), (7, 9), (18, 19), (15, 16), (2, 7), (13, 15), (19, 21)))
db_wiring_CEL[(25, 336, 3)] = ((20, 21, 4, 10, 25, 5, 22, 17, 1, 14, 15, 13, 24, 12, 2, 18, 9, 6, 16, 19, 3, 8, 11, 23, 7), ((17, 19), (5, 7), (13, 17), (12, 13), (17, 18), (7, 12), (18, 20), (12, 14), (14, 15), (15, 18), (6, 7), (18, 19), (11, 12), (19, 21), (12, 15), (21, 22), (1, 2), (15, 16), (2, 6), (6, 8), (8, 9), (9, 12), (12, 13), (13, 15), (15, 19), (19, 21), (21, 23), (23, 25), (7, 9), (14, 15), (9, 10), (18, 19), (5, 7), (7, 9), (9, 14), (14, 16), (22, 23), (16, 18), (13, 14), (4, 5), (8, 9), (14, 16), (18, 22), (16, 18), (15, 16), (12, 15), (11, 12), (9, 11), (5, 9), (3, 5), (9, 10), (18, 19), (10, 13), (1, 3), (13, 14), (3, 4), (4, 6), (6, 7), (7, 10), (10, 11), (11, 13), (13, 18), (18, 20), (20, 21), (5, 7), (7, 8), (12, 13), (21, 24), (17, 18), (8, 12), (6, 8), (12, 14), (18, 21), (14, 15), (15, 18), (13, 15), (11, 13), (10, 11), (8, 10), (7, 8), (15, 16), (2, 7), (13, 15), (18, 19), (21, 22)))
db_wiring_CEL[(25, 336, 4)] = ((4, 20, 16, 5, 10, 22, 17, 1, 14, 15, 13, 24, 2, 25, 18, 12, 9, 3, 21, 19, 8, 6, 23, 11, 7), ((19, 20), (12, 14), (11, 12), (14, 15), (3, 6), (6, 11), (11, 14), (14, 19), (19, 22), (10, 11), (13, 14), (5, 6), (11, 13), (18, 19), (2, 3), (3, 5), (5, 7), (7, 8), (8, 11), (11, 12), (12, 15), (15, 16), (16, 18), (18, 20), (20, 21), (6, 8), (14, 16), (8, 9), (13, 14), (1, 3), (3, 6), (6, 8), (8, 13), (13, 15), (15, 18), (18, 20), (20, 25), (12, 13), (7, 8), (13, 15), (5, 7), (19, 20), (17, 19), (15, 17), (14, 15), (11, 14), (10, 11), (7, 10), (6, 7), (4, 6), (2, 4), (1, 2), (17, 18), (10, 12), (4, 5), (12, 13), (9, 10), (18, 21), (13, 18), (10, 13), (5, 10), (2, 5), (13, 14), (10, 11), (11, 13), (18, 19), (13, 15), (9, 11), (15, 16), (21, 22), (5, 6), (19, 21), (8, 9), (16, 19), (14, 16), (11, 14), (9, 11), (6, 9), (16, 17), (4, 6), (19, 20), (3, 4), (9, 10), (14, 16), (10, 12)))
db_wiring_CEL[(25, 336, 5)] = ((12, 1, 11, 6, 10, 5, 9, 22, 21, 23, 20, 24, 19, 25, 13, 18, 4, 17, 14, 16, 15, 8, 3, 7, 2), ((17, 19), (19, 21), (7, 8), (8, 10), (10, 12), (12, 15), (15, 17), (17, 19), (19, 20), (6, 8), (8, 10), (10, 12), (12, 13), (13, 15), (15, 17), (17, 19), (5, 6), (6, 8), (8, 10), (10, 13), (13, 15), (15, 17), (17, 18), (4, 6), (6, 8), (8, 10), (10, 11), (11, 13), (13, 15), (15, 17), (3, 4), (4, 6), (6, 8), (8, 11), (11, 13), (13, 15), (15, 16), (2, 4), (4, 6), (6, 8), (8, 9), (9, 11), (11, 13), (13, 15), (1, 2), (2, 4), (4, 6), (6, 9), (9, 11), (11, 13), (13, 14), (14, 25), (12, 14), (10, 12), (8, 10), (7, 8), (5, 7), (3, 5), (1, 3), (14, 15), (12, 14), (10, 12), (7, 10), (5, 7), (3, 5), (2, 3), (14, 16), (12, 14), (10, 12), (9, 10), (7, 9), (5, 7), (3, 5), (16, 17), (14, 16), (12, 14), (9, 12), (7, 9), (5, 7), (4, 5), (16, 18), (14, 16), (12, 14), (11, 12), (9, 11)))
db_wiring_CEL[(25, 336, 6)] = ((10, 25, 18, 16, 13, 23, 19, 8, 6, 12, 3, 7, 24, 11, 9, 15, 4, 2, 5, 14, 22, 20, 17, 21, 1), ((3, 4), (6, 8), (10, 12), (14, 15), (20, 23), (8, 10), (18, 20), (17, 18), (15, 17), (12, 15), (10, 12), (9, 10), (7, 9), (17, 19), (4, 7), (12, 13), (15, 17), (19, 21), (23, 24), (17, 19), (13, 15), (7, 8), (2, 4), (15, 17), (14, 15), (21, 23), (19, 21), (17, 19), (15, 17), (21, 22), (8, 15), (19, 21), (6, 8), (15, 16), (4, 6), (16, 19), (8, 9), (19, 20), (6, 8), (1, 2), (8, 10), (14, 16), (13, 14), (10, 11), (11, 13), (13, 15), (2, 4), (15, 17), (17, 19), (4, 6), (6, 8), (8, 11), (11, 13), (13, 15), (15, 17), (17, 18), (18, 25), (16, 18), (14, 16), (12, 14), (10, 12), (9, 10), (12, 13), (7, 9), (5, 7), (13, 15), (3, 5), (7, 8), (18, 19), (15, 18), (5, 7), (14, 15), (18, 20), (7, 14), (1, 3), (14, 16), (20, 21), (16, 18), (18, 20), (13, 14), (20, 22), (14, 16), (6, 7), (12, 14), (16, 18)))
db_wiring_CEL[(25, 360, 1)] = ((14, 22, 15, 17, 19, 12, 20, 10, 23, 7, 9, 16, 3, 8, 5, 6, 1, 11, 21, 4, 18, 24, 13, 2, 25), ((14, 15), (20, 21), (10, 14), (21, 23), (8, 10), (14, 16), (10, 11), (13, 14), (6, 8), (11, 13), (16, 18), (3, 6), (8, 11), (13, 16), (18, 21), (11, 13), (6, 8), (8, 9), (16, 18), (5, 6), (15, 16), (9, 11), (1, 3), (13, 15), (18, 19), (3, 5), (5, 9), (9, 10), (10, 13), (13, 14), (14, 18), (18, 20), (20, 22), (4, 5), (8, 10), (12, 14), (17, 18), (7, 8), (14, 15), (5, 7), (15, 17), (10, 12), (22, 25), (17, 20), (7, 10), (2, 5), (12, 15), (15, 17), (10, 12), (5, 7), (12, 13), (9, 10), (13, 15), (7, 9), (9, 13), (13, 14), (8, 9), (20, 22), (19, 20), (17, 19), (14, 17), (12, 14), (11, 12), (9, 11), (6, 9), (4, 6), (3, 4), (17, 18), (1, 3), (22, 23), (18, 22), (6, 7), (16, 18), (14, 16), (11, 14), (9, 11), (18, 19), (7, 9), (3, 7), (22, 24), (21, 22), (2, 3), (19, 21), (7, 8), (16, 19), (14, 16), (13, 14), (11, 13), (8, 11), (16, 17), (6, 8)))
db_wiring_CEL[(26, 364, 1)] = ((24, 5, 13, 25, 6, 12, 20, 21, 1, 17, 18, 16, 2, 15, 3, 22, 11, 7, 19, 23, 4, 10, 14, 26, 8, 9), ((17, 19), (14, 17), (12, 14), (7, 12), (5, 7), (14, 15), (19, 20), (1, 5), (12, 14), (17, 19), (11, 12), (14, 17), (24, 25), (12, 14), (17, 18), (5, 6), (6, 8), (8, 9), (9, 12), (12, 13), (13, 15), (15, 17), (17, 21), (21, 22), (22, 24), (14, 15), (24, 26), (7, 9), (15, 17), (9, 10), (20, 22), (19, 20), (17, 19), (4, 7), (16, 17), (7, 9), (9, 16), (16, 18), (18, 21), (8, 9), (21, 24), (6, 8), (15, 16), (16, 18), (5, 6), (20, 21), (24, 25), (18, 20), (8, 10), (17, 18), (14, 17), (10, 11), (13, 14), (20, 22), (11, 13), (3, 5), (9, 11), (13, 15), (15, 16), (5, 9), (9, 10), (10, 13), (13, 15), (15, 20), (20, 21), (2, 3), (12, 13), (21, 24), (8, 10), (3, 5), (13, 15), (5, 6), (19, 21), (6, 8), (8, 9), (9, 13), (13, 14), (14, 16), (16, 17), (17, 19), (19, 20), (20, 22), (15, 17), (7, 9), (22, 23), (12, 15), (11, 12), (15, 16), (9, 11), (8, 9), (1, 3), (3, 8)))
db_wiring_CEL[(26, 364, 2)] = ((23, 5, 12, 24, 6, 11, 19, 20, 1, 16, 17, 15, 2, 26, 14, 3, 21, 10, 7, 18, 22, 4, 9, 13, 25, 8), ((18, 20), (14, 18), (12, 14), (7, 12), (18, 19), (5, 7), (14, 15), (19, 21), (17, 19), (12, 14), (21, 22), (14, 17), (11, 12), (17, 18), (12, 14), (1, 5), (5, 6), (6, 8), (8, 9), (9, 12), (12, 13), (13, 15), (15, 17), (17, 21), (21, 23), (23, 24), (14, 15), (24, 26), (7, 9), (15, 17), (9, 10), (20, 21), (4, 7), (17, 18), (7, 9), (9, 15), (15, 17), (17, 20), (6, 7), (20, 24), (14, 15), (15, 17), (7, 9), (19, 20), (17, 19), (3, 4), (16, 17), (9, 10), (13, 16), (12, 13), (10, 12), (24, 25), (8, 10), (19, 21), (12, 14), (4, 8), (14, 15), (8, 9), (9, 12), (12, 14), (21, 22), (2, 4), (14, 19), (11, 12), (7, 9), (19, 21), (4, 5), (12, 14), (5, 7), (21, 24), (7, 8), (18, 19), (8, 12), (12, 13), (19, 21), (13, 15), (15, 16), (1, 2), (6, 8), (16, 19), (14, 16), (19, 20), (11, 14), (20, 22), (16, 17), (10, 11), (8, 10), (7, 8), (14, 16), (22, 23), (13, 14), (2, 7)))
db_wiring_CEL[(26, 364, 3)] = ((1, 3, 5, 7, 9, 11, 13, 21, 17, 25, 2, 16, 26, 20, 22, 24, 18, 15, 14, 19, 23, 4, 6, 8, 10, 12), ((11, 13), (13, 15), (15, 17), (17, 19), (19, 21), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 19), (19, 20), (6, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 19), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 18), (4, 5), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (3, 5), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 16), (2, 3), (3, 5), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (1, 3), (3, 5), (5, 7), (7, 9), (9, 11), (11, 13), (13, 14), (14, 26), (12, 14), (10, 12), (8, 10), (6, 8), (4, 6), (2, 4), (1, 2), (14, 15), (12, 14), (10, 12), (8, 10), (6, 8), (4, 6), (2, 4), (14, 16), (12, 14), (10, 12), (8, 10), (6, 8), (4, 6), (3, 4), (16, 17), (14, 16), (12, 14), (10, 12), (8, 10), (6, 8), (4, 6), (16, 18), (14, 16), (12, 14), (10, 12), (8, 10), (6, 8), (5, 6), (18, 19), (16, 18)))
db_wiring_CEL[(26, 380, 1)] = ((18, 11, 22, 4, 1, 6, 5, 8, 3, 16, 9, 24, 7, 10, 21, 12, 20, 17, 15, 23, 14, 19, 26, 2, 13, 25), ((12, 13), (7, 8), (2, 4), (8, 12), (12, 14), (14, 16), (6, 8), (11, 12), (16, 19), (8, 9), (9, 11), (1, 2), (19, 21), (11, 14), (21, 22), (4, 6), (24, 25), (14, 16), (16, 17), (6, 9), (13, 14), (17, 19), (9, 11), (11, 13), (22, 24), (13, 17), (17, 18), (12, 13), (18, 22), (16, 18), (15, 16), (13, 15), (10, 13), (8, 10), (7, 8), (5, 7), (2, 5), (10, 11), (18, 19), (22, 23), (5, 6), (6, 10), (10, 12), (12, 14), (14, 18), (18, 20), (9, 10), (4, 6), (6, 7), (13, 14), (20, 22), (7, 9), (9, 13), (13, 15), (15, 16), (8, 9), (16, 18), (22, 26), (12, 13), (21, 22), (18, 21), (17, 18), (13, 17), (11, 13), (9, 11), (5, 9), (3, 5), (13, 14), (17, 19), (16, 17), (14, 16), (1, 3), (5, 6), (11, 14), (21, 23), (9, 11), (8, 9), (19, 21), (6, 8), (11, 12), (16, 19), (14, 16), (23, 24), (12, 14), (8, 12), (7, 8), (12, 13), (3, 7), (7, 9), (9, 10), (10, 12), (12, 15), (15, 17), (17, 18), (18, 20), (20, 23)))
db_wiring_CEL[(27, 392, 1)] = ((24, 5, 13, 25, 6, 12, 20, 21, 1, 17, 18, 16, 2, 27, 15, 3, 22, 19, 7, 11, 23, 4, 10, 14, 26, 8, 9), ((14, 18), (12, 14), (7, 12), (18, 19), (5, 7), (14, 15), (19, 21), (17, 19), (12, 14), (25, 26), (21, 22), (14, 17), (11, 12), (1, 5), (12, 14), (17, 18), (5, 6), (6, 8), (8, 9), (9, 12), (12, 13), (13, 15), (15, 17), (17, 21), (21, 23), (23, 25), (14, 15), (25, 27), (7, 9), (22, 23), (15, 17), (9, 10), (20, 22), (19, 20), (4, 7), (17, 19), (16, 17), (7, 9), (9, 16), (16, 18), (18, 21), (8, 9), (6, 8), (21, 25), (15, 16), (5, 6), (16, 18), (20, 21), (8, 10), (18, 20), (17, 18), (3, 5), (10, 11), (14, 17), (25, 26), (13, 14), (11, 13), (9, 11), (20, 22), (13, 15), (5, 9), (15, 16), (9, 10), (2, 3), (10, 13), (22, 23), (13, 15), (15, 20), (3, 5), (12, 13), (8, 10), (20, 22), (5, 6), (13, 15), (22, 25), (6, 8), (19, 20), (8, 9), (9, 13), (20, 22), (13, 14), (14, 16), (16, 17), (17, 20), (7, 9), (20, 21), (15, 17), (1, 3), (12, 15), (21, 23), (17, 18), (11, 12), (23, 24), (9, 11), (15, 17), (8, 9), (14, 15), (3, 8), (8, 10)))
db_wiring_CEL[(27, 392, 2)] = ((24, 23, 5, 12, 25, 6, 11, 19, 20, 1, 16, 17, 15, 27, 2, 14, 3, 21, 10, 7, 26, 13, 9, 4, 22, 18, 8), ((14, 15), (21, 26), (19, 21), (15, 19), (13, 15), (8, 13), (6, 8), (19, 20), (15, 16), (20, 22), (2, 6), (13, 15), (18, 20), (12, 13), (15, 18), (22, 23), (6, 7), (13, 15), (18, 19), (7, 9), (9, 10), (10, 13), (13, 14), (14, 16), (16, 18), (18, 22), (8, 10), (15, 16), (22, 24), (10, 11), (5, 8), (16, 18), (24, 25), (8, 10), (21, 22), (18, 19), (10, 16), (7, 8), (16, 18), (4, 5), (18, 21), (25, 27), (15, 16), (8, 10), (16, 18), (10, 11), (21, 25), (20, 21), (18, 20), (17, 18), (14, 17), (13, 14), (11, 13), (9, 11), (5, 9), (3, 5), (13, 15), (9, 10), (15, 16), (20, 22), (10, 13), (13, 15), (1, 3), (3, 4), (4, 6), (6, 7), (7, 10), (10, 11), (11, 13), (13, 14), (14, 20), (20, 21), (21, 23), (23, 24), (24, 26), (26, 27), (5, 7), (12, 14), (19, 21), (7, 8), (8, 12), (18, 19), (21, 24), (12, 13), (13, 15), (19, 21), (15, 16), (6, 8), (16, 19), (14, 16), (19, 20), (11, 14), (10, 11), (16, 17), (20, 22), (8, 10), (7, 8), (14, 16), (22, 23)))
db_wiring_CEL[(27, 392, 3)] = ((23, 5, 12, 6, 25, 11, 19, 20, 1, 16, 17, 15, 2, 27, 14, 3, 21, 24, 10, 7, 18, 22, 4, 9, 13, 26, 8), ((4, 5), (18, 21), (14, 18), (12, 14), (7, 12), (18, 19), (21, 22), (19, 21), (14, 15), (5, 7), (17, 19), (12, 14), (21, 23), (14, 17), (11, 12), (17, 18), (12, 14), (1, 5), (5, 6), (6, 8), (8, 9), (9, 12), (12, 13), (13, 15), (15, 17), (17, 21), (21, 22), (22, 24), (24, 25), (25, 27), (14, 15), (7, 9), (15, 17), (9, 10), (20, 22), (4, 7), (17, 18), (7, 9), (9, 15), (15, 17), (6, 7), (17, 20), (20, 21), (14, 15), (21, 25), (15, 17), (7, 9), (3, 4), (19, 21), (17, 19), (9, 10), (16, 17), (13, 16), (12, 13), (10, 12), (19, 20), (8, 10), (4, 8), (12, 14), (20, 22), (14, 15), (8, 9), (2, 4), (9, 12), (12, 14), (25, 26), (22, 23), (14, 20), (11, 12), (7, 9), (4, 5), (1, 2), (5, 7), (12, 14), (7, 8), (20, 22), (8, 12), (14, 15), (19, 20), (12, 14), (22, 25), (14, 16), (6, 8), (16, 17), (11, 12), (20, 22), (17, 20), (15, 17), (12, 15), (10, 12), (20, 21), (8, 10), (17, 18), (7, 8), (21, 23), (15, 17), (10, 11), (2, 7), (14, 15)))
db_wiring_CEL[(27, 400, 1)] = ((13, 26, 4, 19, 23, 11, 1, 6, 5, 8, 3, 17, 9, 25, 7, 10, 16, 22, 12, 21, 18, 15, 24, 14, 20, 27, 2), ((14, 15), (1, 3), (9, 10), (10, 14), (14, 16), (8, 10), (16, 17), (13, 14), (17, 19), (10, 11), (11, 13), (19, 22), (6, 8), (13, 17), (22, 24), (8, 11), (17, 19), (11, 13), (19, 20), (16, 17), (13, 14), (24, 25), (20, 22), (14, 16), (3, 6), (16, 20), (15, 16), (6, 8), (20, 21), (8, 9), (9, 11), (11, 15), (15, 17), (17, 18), (18, 20), (5, 6), (10, 11), (20, 24), (6, 10), (10, 12), (4, 6), (12, 13), (19, 20), (9, 10), (13, 15), (6, 7), (24, 26), (15, 19), (7, 9), (14, 15), (19, 21), (9, 14), (23, 24), (8, 9), (2, 4), (14, 16), (16, 17), (26, 27), (21, 23), (17, 19), (13, 14), (4, 8), (8, 10), (10, 11), (11, 13), (13, 17), (17, 18), (18, 21), (12, 13), (3, 4), (16, 18), (21, 22), (15, 16), (13, 15), (9, 13), (22, 26), (7, 9), (6, 7), (20, 22), (4, 6), (9, 10), (18, 20), (15, 18), (13, 15), (12, 13), (10, 12), (6, 10), (22, 23), (5, 6), (10, 11), (1, 5), (5, 7), (7, 8), (8, 10), (10, 14), (14, 16), (16, 17), (17, 19), (19, 22), (22, 24), (24, 25)))
db_wiring_CEL[(28, 420, 1)] = ((25, 24, 5, 13, 26, 6, 12, 20, 21, 1, 17, 18, 16, 2, 28, 15, 3, 22, 11, 7, 19, 23, 4, 10, 14, 27, 8, 9), ((19, 21), (15, 19), (13, 15), (8, 13), (6, 8), (19, 20), (15, 16), (20, 22), (2, 6), (13, 15), (18, 20), (12, 13), (15, 18), (22, 23), (6, 7), (13, 15), (7, 9), (18, 19), (9, 10), (10, 13), (13, 14), (14, 16), (16, 18), (26, 27), (18, 22), (8, 10), (15, 16), (22, 24), (10, 11), (5, 8), (16, 18), (24, 26), (8, 10), (23, 24), (21, 23), (20, 21), (18, 20), (17, 18), (10, 17), (9, 10), (7, 9), (6, 7), (17, 19), (4, 6), (19, 22), (26, 28), (16, 17), (9, 11), (3, 4), (17, 19), (11, 12), (22, 26), (21, 22), (19, 21), (18, 19), (15, 18), (14, 15), (12, 14), (10, 12), (6, 10), (4, 6), (14, 16), (10, 11), (16, 17), (21, 23), (11, 14), (14, 16), (1, 4), (4, 5), (5, 7), (7, 8), (8, 11), (11, 12), (12, 14), (14, 15), (15, 21), (21, 22), (22, 24), (24, 25), (25, 27), (27, 28), (6, 8), (13, 15), (20, 22), (8, 9), (9, 13), (19, 20), (22, 25), (13, 14), (14, 16), (20, 22), (16, 17), (7, 9), (17, 20), (15, 17), (20, 21), (12, 15), (11, 12), (17, 18), (21, 23), (9, 11), (8, 9), (15, 17), (23, 24), (14, 15), (3, 8)))
db_wiring_CEL[(28, 420, 2)] = ((24, 23, 5, 12, 25, 6, 11, 19, 20, 1, 16, 17, 15, 27, 2, 28, 14, 3, 21, 10, 7, 26, 13, 9, 4, 22, 18, 8), ((14, 16), (13, 14), (22, 27), (20, 22), (16, 20), (14, 16), (8, 14), (6, 8), (2, 6), (20, 21), (16, 17), (21, 23), (14, 16), (8, 9), (19, 21), (13, 14), (6, 8), (16, 19), (8, 10), (14, 16), (23, 24), (10, 11), (19, 20), (11, 14), (14, 15), (5, 6), (15, 17), (17, 19), (9, 11), (19, 23), (16, 17), (11, 12), (6, 9), (23, 25), (17, 19), (9, 11), (25, 26), (22, 23), (19, 20), (11, 17), (8, 9), (4, 6), (17, 19), (19, 22), (9, 11), (16, 17), (17, 19), (26, 28), (11, 12), (22, 26), (21, 22), (19, 21), (18, 19), (15, 18), (14, 15), (12, 14), (10, 12), (6, 10), (5, 6), (3, 5), (14, 16), (10, 11), (16, 17), (21, 23), (11, 14), (14, 16), (1, 3), (3, 4), (4, 7), (7, 8), (8, 11), (11, 12), (12, 14), (14, 15), (15, 21), (21, 22), (22, 24), (24, 25), (25, 27), (27, 28), (6, 8), (13, 15), (20, 22), (8, 9), (5, 6), (9, 13), (19, 20), (22, 25), (13, 14), (14, 16), (20, 22), (16, 17), (6, 9), (17, 20), (15, 17), (20, 21), (9, 10), (12, 15), (21, 23), (17, 18), (10, 12), (8, 10), (23, 24), (15, 17), (7, 8), (12, 13)))
db_wiring_CEL[(28, 420, 3)] = ((24, 23, 5, 12, 25, 6, 28, 11, 19, 20, 1, 16, 17, 15, 2, 27, 14, 3, 21, 10, 7, 18, 22, 4, 9, 13, 26, 8), ((7, 8), (20, 22), (16, 20), (14, 16), (8, 14), (20, 21), (6, 8), (14, 15), (15, 17), (21, 23), (17, 18), (18, 21), (13, 15), (21, 22), (2, 6), (12, 13), (15, 18), (22, 24), (13, 15), (18, 19), (6, 7), (7, 9), (24, 25), (9, 10), (10, 13), (13, 14), (14, 16), (16, 18), (18, 22), (8, 10), (15, 16), (22, 24), (24, 26), (10, 11), (16, 18), (5, 8), (21, 22), (8, 10), (18, 19), (10, 16), (7, 8), (26, 28), (16, 18), (18, 21), (4, 5), (15, 16), (25, 26), (8, 10), (16, 18), (21, 25), (10, 11), (20, 21), (18, 20), (17, 18), (14, 17), (13, 14), (11, 13), (9, 11), (5, 9), (13, 15), (3, 5), (9, 10), (15, 16), (20, 22), (10, 13), (13, 15), (1, 3), (3, 4), (4, 6), (6, 7), (7, 10), (10, 11), (11, 13), (13, 14), (14, 20), (20, 21), (21, 23), (23, 24), (24, 27), (27, 28), (5, 7), (12, 14), (7, 8), (19, 21), (8, 12), (18, 19), (21, 24), (12, 13), (13, 15), (6, 8), (15, 16), (19, 21), (16, 19), (14, 16), (11, 14), (19, 20), (24, 25), (10, 11), (16, 17), (8, 10), (20, 22), (7, 8), (14, 16), (22, 24), (13, 14), (2, 7)))
db_wiring_CEL[(28, 420, 4)] = ((9, 1, 11, 3, 13, 5, 8, 7, 10, 18, 27, 23, 22, 28, 17, 19, 26, 24, 21, 15, 16, 20, 25, 2, 12, 4, 14, 6), ((9, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (22, 23), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (7, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 21), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (5, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 19), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (3, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 17), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (1, 2), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 15), (15, 28), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (1, 3), (15, 16), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (2, 3), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (17, 18), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (4, 5), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7)))
db_wiring_CEL[(28, 420, 5)] = ((20, 24, 12, 1, 6, 8, 5, 3, 18, 9, 7, 26, 10, 17, 23, 13, 22, 19, 16, 25, 11, 15, 21, 28, 2, 4, 27, 14), ((7, 11), (26, 28), (11, 13), (20, 21), (13, 14), (5, 7), (14, 16), (10, 11), (7, 8), (16, 20), (8, 10), (15, 16), (20, 22), (10, 15), (22, 23), (3, 5), (9, 10), (25, 26), (15, 17), (17, 18), (5, 9), (14, 15), (18, 20), (9, 11), (23, 25), (11, 12), (12, 14), (14, 18), (18, 19), (4, 5), (13, 14), (19, 23), (17, 19), (16, 17), (14, 16), (10, 14), (23, 24), (8, 10), (7, 8), (5, 7), (19, 20), (10, 11), (1, 5), (5, 6), (6, 10), (10, 12), (12, 13), (13, 15), (15, 19), (19, 21), (21, 23), (23, 27), (9, 10), (14, 15), (4, 6), (6, 7), (7, 9), (22, 23), (27, 28), (9, 14), (14, 16), (16, 17), (17, 19), (19, 22), (13, 14), (8, 9), (18, 19), (14, 18), (12, 14), (11, 12), (9, 11), (22, 24), (18, 20), (14, 15), (5, 9), (17, 18), (15, 17), (3, 5), (24, 25), (11, 15), (2, 3), (20, 22), (5, 6), (9, 11), (17, 20), (8, 9), (11, 12), (15, 17), (6, 8), (14, 15), (12, 14), (8, 12), (25, 27), (7, 8), (12, 13), (22, 25), (20, 22), (19, 20), (17, 19), (13, 17), (11, 13), (10, 11), (8, 10), (3, 8), (17, 18), (22, 23)))
db_wiring_CEL[(28, 420, 6)] = ((14, 27, 20, 4, 24, 11, 1, 13, 6, 5, 8, 3, 18, 9, 7, 26, 10, 17, 23, 12, 22, 19, 16, 25, 15, 21, 28, 2), ((10, 11), (3, 4), (11, 15), (15, 17), (17, 18), (9, 11), (18, 20), (14, 15), (11, 12), (20, 23), (12, 14), (8, 9), (23, 25), (14, 18), (6, 8), (25, 26), (18, 20), (8, 12), (20, 21), (17, 18), (12, 14), (21, 23), (14, 15), (1, 3), (15, 17), (11, 12), (17, 21), (21, 22), (16, 17), (3, 6), (6, 8), (8, 9), (9, 11), (11, 16), (16, 18), (18, 19), (19, 21), (21, 25), (25, 27), (27, 28), (10, 11), (5, 6), (15, 16), (6, 10), (20, 21), (10, 12), (12, 13), (13, 15), (4, 6), (9, 10), (15, 20), (6, 7), (24, 25), (20, 22), (7, 9), (14, 15), (9, 14), (19, 20), (22, 24), (14, 16), (8, 9), (16, 17), (17, 19), (13, 14), (2, 4), (4, 8), (8, 10), (10, 11), (11, 13), (13, 17), (17, 18), (18, 22), (22, 23), (12, 13), (16, 18), (23, 27), (15, 16), (13, 15), (3, 4), (9, 13), (21, 23), (7, 9), (6, 7), (20, 21), (4, 6), (18, 20), (9, 10), (15, 18), (13, 15), (12, 13), (10, 12), (23, 24), (6, 10), (5, 6), (10, 11), (1, 5), (5, 7), (7, 8), (8, 10), (10, 14), (14, 16), (16, 17), (17, 19), (19, 23), (23, 25), (13, 14)))
db_wiring_CEL[(29, 440, 1)] = ((21, 25, 12, 1, 14, 6, 8, 5, 3, 19, 9, 7, 27, 10, 18, 24, 13, 23, 20, 17, 26, 11, 16, 22, 29, 2, 15, 28, 4), ((8, 12), (12, 14), (21, 22), (14, 15), (6, 8), (15, 17), (11, 12), (8, 9), (17, 21), (9, 11), (5, 6), (16, 17), (21, 23), (11, 16), (23, 24), (26, 27), (10, 11), (3, 5), (16, 18), (18, 19), (5, 10), (15, 16), (19, 21), (24, 26), (10, 12), (12, 13), (13, 15), (15, 19), (19, 20), (4, 5), (9, 10), (14, 15), (20, 24), (18, 20), (17, 18), (15, 17), (10, 15), (24, 25), (8, 10), (7, 8), (5, 7), (20, 21), (15, 16), (10, 11), (1, 5), (5, 6), (6, 10), (10, 12), (12, 13), (13, 15), (15, 20), (20, 22), (22, 24), (24, 28), (9, 10), (14, 15), (4, 6), (19, 20), (6, 7), (7, 9), (23, 24), (28, 29), (9, 14), (14, 16), (16, 17), (17, 19), (19, 23), (13, 14), (8, 9), (18, 19), (14, 18), (12, 14), (11, 12), (9, 11), (23, 25), (18, 20), (14, 15), (5, 9), (17, 18), (15, 17), (3, 5), (25, 26), (22, 23), (11, 15), (2, 3), (20, 22), (5, 6), (9, 11), (17, 20), (8, 9), (11, 12), (15, 17), (6, 8), (14, 15), (12, 14), (8, 12), (7, 8), (12, 13), (26, 28), (22, 26), (20, 22), (19, 20), (17, 19), (13, 17), (11, 13), (10, 11), (8, 10), (3, 8), (1, 3), (17, 18), (22, 23)))
db_wiring_CEL[(29, 448, 1)] = ((25, 24, 5, 13, 6, 27, 12, 20, 21, 1, 17, 18, 16, 2, 29, 15, 3, 22, 26, 11, 7, 19, 23, 4, 10, 14, 28, 8, 9), ((5, 6), (19, 22), (15, 19), (13, 15), (8, 13), (19, 20), (22, 23), (20, 22), (15, 16), (6, 8), (27, 28), (18, 20), (13, 15), (22, 24), (15, 18), (12, 13), (18, 19), (13, 15), (2, 6), (6, 7), (7, 9), (9, 10), (10, 13), (13, 14), (14, 16), (16, 18), (18, 22), (22, 23), (23, 25), (15, 16), (25, 27), (8, 10), (10, 11), (16, 18), (24, 25), (5, 8), (21, 24), (27, 29), (20, 21), (8, 10), (18, 20), (17, 18), (10, 17), (9, 10), (17, 19), (7, 9), (6, 7), (19, 22), (22, 23), (16, 17), (4, 6), (9, 11), (17, 19), (23, 27), (3, 4), (11, 12), (21, 23), (19, 21), (18, 19), (15, 18), (14, 15), (12, 14), (10, 12), (21, 22), (6, 10), (14, 16), (4, 6), (10, 11), (16, 17), (22, 24), (11, 14), (14, 16), (1, 4), (4, 5), (5, 7), (7, 8), (8, 11), (11, 12), (12, 14), (14, 15), (15, 22), (22, 23), (23, 25), (25, 26), (26, 28), (28, 29), (6, 8), (13, 15), (8, 9), (21, 23), (9, 13), (15, 16), (20, 21), (13, 15), (23, 26), (15, 17), (7, 9), (17, 18), (12, 13), (21, 23), (18, 21), (16, 18), (13, 16), (21, 22), (11, 13), (9, 11), (18, 19), (8, 9), (22, 24), (16, 18), (11, 12), (3, 8), (15, 16)))
db_wiring_CEL[(29, 448, 2)] = ((25, 24, 5, 13, 26, 6, 12, 20, 21, 1, 17, 18, 29, 16, 2, 28, 15, 3, 22, 11, 7, 27, 14, 10, 4, 23, 19, 8, 9), ((22, 27), (20, 22), (16, 20), (14, 16), (8, 14), (6, 8), (2, 6), (20, 21), (16, 17), (21, 23), (14, 16), (8, 9), (19, 21), (13, 14), (6, 8), (16, 19), (8, 10), (14, 16), (23, 24), (10, 11), (19, 20), (11, 14), (14, 15), (5, 6), (15, 17), (17, 19), (9, 11), (19, 23), (16, 17), (11, 12), (6, 9), (23, 25), (17, 19), (27, 28), (9, 11), (25, 27), (24, 25), (22, 24), (21, 22), (19, 21), (18, 19), (11, 18), (10, 11), (8, 10), (7, 8), (4, 7), (18, 20), (3, 4), (20, 23), (10, 12), (17, 18), (18, 20), (12, 13), (27, 29), (23, 27), (22, 23), (20, 22), (19, 20), (16, 19), (15, 16), (13, 15), (11, 13), (7, 11), (6, 7), (4, 6), (15, 17), (11, 12), (17, 18), (22, 24), (12, 15), (15, 17), (1, 4), (4, 5), (5, 8), (8, 9), (9, 12), (12, 13), (13, 15), (15, 16), (16, 22), (22, 23), (23, 25), (25, 26), (26, 28), (28, 29), (7, 9), (14, 16), (21, 23), (9, 10), (6, 7), (10, 14), (20, 21), (23, 26), (14, 15), (15, 17), (21, 23), (17, 18), (7, 10), (18, 21), (16, 18), (21, 22), (10, 11), (13, 16), (22, 24), (18, 19), (11, 13), (9, 11), (24, 25), (16, 18), (8, 9), (13, 14), (14, 16), (16, 17)))
db_wiring_CEL[(29, 448, 3)] = ((25, 24, 5, 13, 26, 6, 12, 29, 20, 21, 1, 17, 18, 16, 2, 28, 15, 3, 22, 11, 7, 19, 23, 4, 10, 14, 27, 8, 9), ((20, 22), (16, 20), (14, 16), (8, 14), (20, 21), (6, 8), (14, 15), (15, 17), (21, 23), (17, 18), (18, 21), (13, 15), (2, 6), (21, 22), (12, 13), (15, 18), (22, 24), (13, 15), (6, 7), (18, 19), (27, 28), (7, 9), (9, 10), (24, 25), (10, 13), (13, 14), (14, 16), (16, 18), (18, 22), (8, 10), (15, 16), (22, 24), (10, 11), (24, 27), (16, 18), (5, 8), (23, 24), (8, 10), (21, 23), (20, 21), (18, 20), (17, 18), (10, 17), (9, 10), (7, 9), (6, 7), (17, 19), (27, 29), (19, 22), (4, 6), (16, 17), (9, 11), (26, 27), (3, 4), (17, 19), (22, 26), (11, 12), (21, 22), (19, 21), (18, 19), (15, 18), (14, 15), (12, 14), (10, 12), (6, 10), (14, 16), (4, 6), (10, 11), (16, 17), (21, 23), (11, 14), (14, 16), (1, 4), (4, 5), (5, 7), (7, 8), (8, 11), (11, 12), (12, 14), (14, 15), (15, 21), (21, 22), (22, 24), (24, 25), (25, 28), (28, 29), (6, 8), (13, 15), (20, 22), (8, 9), (9, 13), (19, 20), (22, 25), (13, 14), (14, 16), (7, 9), (16, 17), (20, 22), (17, 20), (15, 17), (12, 15), (20, 21), (25, 26), (11, 12), (17, 18), (21, 23), (9, 11), (8, 9), (15, 17), (23, 25), (14, 15), (3, 8), (22, 23)))
db_wiring_CEL[(29, 448, 4)] = ((9, 1, 11, 3, 13, 5, 8, 7, 18, 23, 27, 28, 22, 10, 17, 19, 29, 26, 24, 21, 2, 16, 15, 25, 20, 12, 4, 14, 6), ((21, 23), (23, 25), (14, 16), (16, 17), (17, 19), (8, 10), (10, 12), (19, 21), (12, 14), (21, 23), (23, 24), (14, 17), (7, 8), (8, 10), (17, 19), (10, 12), (19, 21), (12, 14), (21, 23), (14, 15), (6, 8), (15, 17), (8, 10), (17, 19), (10, 12), (19, 21), (21, 22), (12, 15), (5, 6), (6, 8), (15, 17), (8, 10), (17, 19), (19, 21), (10, 12), (12, 13), (13, 15), (4, 6), (15, 17), (6, 8), (17, 19), (19, 20), (8, 10), (10, 13), (13, 15), (15, 17), (3, 4), (4, 6), (17, 19), (6, 8), (8, 10), (10, 11), (11, 13), (13, 15), (15, 17), (17, 18), (2, 4), (4, 6), (6, 8), (8, 11), (11, 13), (13, 15), (15, 17), (1, 2), (2, 4), (4, 6), (6, 8), (8, 9), (9, 11), (11, 13), (13, 15), (15, 16), (16, 29), (14, 16), (12, 14), (10, 12), (7, 10), (5, 7), (3, 5), (1, 3), (16, 17), (14, 16), (12, 14), (10, 12), (9, 10), (7, 9), (5, 7), (3, 5), (2, 3), (16, 18), (14, 16), (12, 14), (9, 12), (7, 9), (5, 7), (3, 5), (18, 19), (16, 18), (14, 16), (12, 14), (11, 12), (9, 11), (7, 9), (18, 20), (16, 18), (5, 7), (4, 5), (14, 16), (11, 14), (20, 21), (9, 11), (18, 20), (16, 18)))
db_wiring_CEL[(30, 460, 1)] = ((22, 26, 13, 3, 1, 15, 7, 9, 6, 4, 20, 10, 8, 28, 11, 19, 25, 14, 24, 21, 18, 27, 12, 17, 23, 30, 2, 16, 29, 5), ((9, 13), (13, 15), (22, 23), (15, 16), (7, 9), (16, 18), (12, 13), (9, 10), (18, 22), (10, 12), (6, 7), (17, 18), (22, 24), (3, 4), (12, 17), (24, 25), (27, 28), (11, 12), (4, 6), (17, 19), (19, 20), (6, 11), (16, 17), (20, 22), (25, 27), (11, 13), (13, 14), (14, 16), (16, 20), (20, 21), (5, 6), (10, 11), (15, 16), (21, 25), (19, 21), (18, 19), (16, 18), (11, 16), (25, 26), (9, 11), (8, 9), (6, 8), (21, 22), (16, 17), (11, 12), (1, 6), (6, 7), (7, 11), (11, 13), (13, 14), (14, 16), (16, 21), (21, 23), (23, 25), (25, 29), (10, 11), (15, 16), (5, 7), (20, 21), (7, 8), (8, 10), (29, 30), (10, 15), (24, 25), (15, 17), (17, 18), (18, 20), (20, 24), (14, 15), (9, 10), (4, 5), (19, 20), (15, 19), (13, 15), (12, 13), (10, 12), (24, 26), (19, 21), (15, 16), (5, 10), (18, 19), (16, 18), (3, 5), (10, 11), (26, 27), (23, 24), (11, 16), (2, 3), (21, 23), (16, 17), (5, 6), (9, 11), (17, 21), (8, 9), (11, 12), (15, 17), (6, 8), (14, 15), (21, 22), (12, 14), (8, 12), (7, 8), (27, 29), (12, 13), (22, 27), (20, 22), (19, 20), (17, 19), (13, 17), (11, 13), (10, 11), (8, 10), (3, 8), (17, 18), (1, 3), (22, 23)))
db_wiring_CEL[(30, 476, 1)] = ((25, 24, 5, 13, 27, 6, 12, 20, 21, 1, 17, 18, 16, 30, 2, 29, 15, 3, 22, 26, 11, 7, 19, 23, 4, 10, 14, 28, 8, 9), ((13, 14), (20, 23), (16, 20), (14, 16), (8, 14), (20, 21), (6, 8), (23, 24), (16, 17), (21, 23), (19, 21), (14, 16), (2, 6), (8, 9), (13, 14), (16, 19), (23, 25), (14, 16), (6, 8), (19, 20), (28, 29), (8, 10), (10, 11), (11, 14), (14, 15), (15, 17), (17, 19), (19, 23), (9, 11), (23, 24), (16, 17), (24, 26), (5, 6), (11, 12), (26, 28), (17, 19), (6, 9), (25, 26), (9, 11), (22, 25), (21, 22), (19, 21), (18, 19), (11, 18), (10, 11), (8, 10), (7, 8), (18, 20), (28, 30), (20, 23), (4, 7), (23, 24), (17, 18), (10, 12), (3, 4), (18, 20), (24, 28), (12, 13), (22, 24), (20, 22), (19, 20), (16, 19), (15, 16), (13, 15), (11, 13), (7, 11), (22, 23), (6, 7), (15, 17), (4, 6), (11, 12), (17, 18), (23, 25), (12, 15), (15, 17), (1, 4), (4, 5), (5, 8), (8, 9), (9, 12), (12, 13), (13, 15), (15, 16), (16, 23), (23, 24), (24, 26), (26, 27), (27, 29), (29, 30), (7, 9), (14, 16), (22, 24), (9, 10), (6, 7), (10, 14), (16, 17), (21, 22), (24, 27), (14, 16), (16, 18), (7, 10), (18, 19), (22, 24), (13, 14), (19, 22), (10, 11), (17, 19), (14, 17), (22, 23), (11, 14), (19, 20), (23, 25), (9, 11), (8, 9), (17, 19), (14, 15), (11, 12), (25, 26), (15, 17), (3, 8)))
db_wiring_CEL[(30, 480, 1)] = ((9, 5, 1, 12, 8, 4, 15, 11, 7, 3, 21, 29, 28, 22, 20, 30, 27, 23, 19, 16, 26, 24, 18, 17, 25, 14, 10, 6, 2, 13), ((10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (22, 24), (24, 25), (9, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (22, 24), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (22, 23), (7, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 21), (5, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 19), (3, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 17), (1, 2), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 30), (15, 16), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (1, 3), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (2, 3), (17, 18), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (4, 5), (19, 20), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7)))
db_wiring_CEL[(31, 480, 1)] = ((12, 19, 3, 28, 7, 24, 16, 13, 6, 27, 31, 9, 22, 1, 21, 30, 10, 26, 5, 14, 17, 23, 8, 2, 11, 20, 18, 15, 4, 25, 29), ((11, 12), (3, 4), (21, 22), (12, 16), (16, 18), (26, 27), (18, 19), (10, 12), (19, 21), (15, 16), (12, 13), (21, 26), (13, 15), (9, 10), (6, 7), (20, 21), (26, 28), (15, 20), (7, 9), (14, 15), (28, 29), (25, 26), (20, 22), (9, 14), (22, 23), (1, 3), (14, 16), (19, 20), (8, 9), (16, 17), (23, 25), (17, 19), (13, 14), (19, 23), (23, 24), (18, 19), (3, 8), (8, 10), (10, 11), (11, 13), (13, 18), (18, 20), (20, 21), (21, 23), (23, 28), (28, 30), (12, 13), (30, 31), (7, 8), (17, 18), (8, 12), (22, 23), (12, 14), (14, 15), (15, 17), (6, 8), (11, 12), (17, 22), (8, 9), (27, 28), (22, 24), (9, 11), (16, 17), (24, 25), (5, 6), (11, 16), (21, 22), (25, 27), (16, 18), (10, 11), (18, 19), (2, 3), (19, 21), (15, 16), (3, 5), (5, 10), (10, 12), (12, 13), (13, 15), (15, 19), (19, 20), (20, 25), (25, 26), (14, 15), (9, 10), (18, 20), (26, 30), (17, 18), (15, 17), (4, 5), (10, 15), (20, 21), (24, 26), (8, 10), (15, 16), (7, 8), (23, 24), (5, 7), (21, 23), (10, 11), (16, 21), (14, 16), (13, 14), (11, 13), (26, 27), (7, 11), (6, 7), (21, 22), (16, 17), (11, 12), (1, 6), (6, 8), (8, 9), (9, 11), (11, 16), (16, 18), (18, 19), (19, 21), (21, 26), (26, 28), (15, 16)))
db_wiring_CEL[(31, 504, 1)] = ((28, 6, 14, 7, 29, 13, 8, 12, 23, 24, 25, 1, 19, 20, 21, 18, 2, 17, 3, 31, 16, 4, 26, 11, 22, 27, 5, 10, 15, 30, 9), ((4, 5), (24, 25), (20, 24), (18, 20), (16, 18), (9, 16), (7, 9), (24, 26), (20, 21), (5, 7), (16, 17), (23, 24), (17, 20), (26, 27), (20, 23), (15, 17), (9, 10), (19, 20), (14, 15), (17, 19), (15, 17), (1, 5), (5, 6), (6, 9), (9, 11), (11, 12), (12, 15), (15, 16), (16, 18), (18, 21), (21, 22), (22, 26), (26, 28), (28, 29), (29, 31), (10, 12), (17, 18), (8, 10), (12, 13), (7, 8), (20, 22), (10, 12), (25, 26), (18, 20), (4, 7), (22, 23), (7, 10), (10, 11), (11, 18), (18, 19), (19, 22), (22, 25), (6, 7), (9, 11), (17, 19), (21, 22), (25, 29), (19, 21), (16, 17), (7, 9), (3, 4), (17, 19), (24, 25), (11, 12), (21, 24), (19, 21), (18, 19), (15, 18), (14, 15), (12, 14), (9, 12), (8, 9), (21, 22), (14, 16), (4, 8), (12, 14), (16, 17), (24, 26), (11, 12), (14, 16), (2, 4), (8, 11), (22, 24), (29, 30), (11, 14), (14, 15), (26, 27), (15, 22), (10, 11), (7, 8), (4, 5), (22, 23), (13, 15), (1, 2), (5, 7), (11, 13), (23, 26), (7, 11), (15, 16), (21, 23), (26, 29), (11, 12), (20, 21), (12, 15), (25, 26), (6, 7), (15, 17), (23, 25), (17, 18), (21, 23), (18, 21), (16, 18), (14, 16), (13, 14), (21, 22), (10, 13), (22, 24), (18, 19), (9, 10), (7, 9), (16, 18), (24, 27), (13, 16), (2, 7), (12, 13), (16, 17), (23, 24)))
db_wiring_CEL[(31, 504, 2)] = ((25, 24, 5, 13, 27, 6, 31, 12, 20, 21, 1, 17, 18, 30, 16, 2, 29, 15, 3, 22, 26, 11, 7, 19, 23, 4, 10, 14, 28, 8, 9), ((7, 8), (21, 24), (17, 21), (15, 17), (8, 15), (21, 22), (24, 25), (15, 16), (6, 8), (16, 18), (22, 24), (18, 19), (19, 22), (14, 16), (22, 23), (29, 30), (23, 26), (16, 19), (13, 14), (8, 9), (2, 6), (14, 16), (19, 20), (26, 27), (6, 8), (8, 10), (10, 11), (11, 14), (14, 15), (15, 17), (17, 19), (19, 23), (23, 24), (24, 26), (16, 17), (9, 11), (26, 29), (5, 6), (11, 12), (17, 19), (6, 9), (25, 26), (22, 25), (9, 11), (21, 22), (19, 21), (18, 19), (11, 18), (29, 31), (10, 11), (8, 10), (18, 20), (7, 8), (20, 23), (4, 7), (23, 24), (28, 29), (17, 18), (10, 12), (18, 20), (3, 4), (24, 28), (12, 13), (22, 24), (20, 22), (19, 20), (16, 19), (15, 16), (13, 15), (11, 13), (7, 11), (22, 23), (6, 7), (15, 17), (4, 6), (11, 12), (17, 18), (23, 25), (12, 15), (15, 17), (1, 4), (4, 5), (5, 8), (8, 9), (9, 12), (12, 13), (13, 15), (15, 16), (16, 23), (23, 24), (24, 26), (26, 27), (27, 30), (30, 31), (7, 9), (14, 16), (9, 10), (22, 24), (6, 7), (10, 14), (16, 17), (21, 22), (14, 16), (24, 27), (16, 18), (7, 10), (18, 19), (22, 24), (13, 14), (19, 22), (10, 11), (17, 19), (14, 17), (22, 23), (11, 14), (27, 28), (19, 20), (9, 11), (8, 9), (23, 25), (17, 19), (14, 15), (11, 12), (3, 8), (15, 17), (25, 27), (17, 18)))
db_wiring_CEL[(32, 544, 1)] = ((4, 13, 1, 10, 6, 15, 3, 12, 8, 9, 26, 31, 21, 20, 5, 25, 32, 30, 27, 19, 22, 24, 17, 29, 28, 18, 23, 14, 2, 11, 7, 16), ((15, 17), (17, 19), (19, 21), (21, 23), (23, 25), (25, 27), (10, 11), (11, 13), (13, 15), (15, 17), (17, 19), (19, 21), (21, 23), (23, 25), (25, 26), (9, 11), (11, 13), (13, 15), (15, 17), (17, 19), (19, 21), (21, 23), (23, 25), (8, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 19), (19, 21), (21, 23), (23, 24), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 19), (19, 21), (21, 23), (6, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 19), (19, 21), (21, 22), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 19), (19, 21), (4, 5), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 19), (19, 20), (3, 5), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 19), (2, 3), (3, 5), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 18), (1, 3), (3, 5), (5, 7), (7, 9), (9, 11), (11, 13), (13, 15), (15, 17), (17, 32), (16, 17), (14, 16), (12, 14), (10, 12), (8, 10), (6, 8), (4, 6), (2, 4), (1, 2), (16, 18), (14, 16), (12, 14), (10, 12), (8, 10), (6, 8), (4, 6), (2, 4), (18, 19), (16, 18), (14, 16), (12, 14), (10, 12), (8, 10), (6, 8), (4, 6), (3, 4), (18, 20), (16, 18), (14, 16), (12, 14), (10, 12), (8, 10), (6, 8), (4, 6), (20, 21), (18, 20), (16, 18), (14, 16), (12, 14), (10, 12), (8, 10), (6, 8), (5, 6), (20, 22), (18, 20)))
db_wiring_CEL[(33, 576, 1)] = ((4, 13, 1, 10, 6, 15, 3, 12, 8, 9, 26, 31, 21, 20, 32, 25, 27, 30, 5, 33, 19, 22, 24, 14, 29, 17, 18, 28, 23, 2, 11, 7, 16), ((10, 11), (24, 26), (11, 13), (19, 20), (26, 28), (13, 15), (20, 22), (28, 29), (15, 17), (22, 24), (9, 11), (11, 13), (17, 20), (24, 26), (13, 15), (26, 28), (20, 22), (8, 9), (15, 17), (9, 11), (22, 24), (17, 18), (11, 13), (18, 20), (24, 26), (26, 27), (13, 15), (7, 9), (20, 22), (9, 11), (15, 18), (22, 24), (11, 13), (24, 26), (18, 20), (6, 7), (13, 15), (7, 9), (20, 22), (15, 16), (9, 11), (22, 24), (16, 18), (24, 25), (11, 13), (18, 20), (5, 7), (7, 9), (13, 16), (20, 22), (22, 24), (9, 11), (16, 18), (18, 20), (11, 13), (4, 5), (20, 22), (13, 14), (5, 7), (22, 23), (14, 16), (7, 9), (16, 18), (9, 11), (18, 20), (20, 22), (11, 14), (3, 5), (14, 16), (5, 7), (16, 18), (7, 9), (18, 20), (20, 21), (9, 11), (11, 12), (12, 14), (14, 16), (2, 3), (16, 18), (3, 5), (18, 20), (5, 7), (7, 9), (9, 12), (12, 14), (14, 16), (16, 18), (18, 19), (1, 3), (3, 5), (5, 7), (7, 9), (9, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 33), (17, 18), (15, 17), (13, 15), (11, 13), (8, 11), (6, 8), (4, 6), (2, 4), (1, 2), (17, 19), (15, 17), (13, 15), (11, 13), (10, 11), (8, 10), (6, 8), (4, 6), (19, 20), (2, 4), (17, 19), (15, 17), (13, 15), (10, 13), (8, 10), (19, 21), (17, 19), (6, 8), (15, 17), (4, 6), (3, 4), (13, 15), (12, 13), (21, 22), (19, 21), (10, 12), (17, 19), (8, 10), (15, 17), (6, 8), (21, 23), (12, 15), (4, 6), (19, 21), (17, 19), (10, 12), (15, 17)))
db_wiring_CEL[(34, 612, 1)] = ((27, 11, 5, 16, 30, 10, 24, 25, 1, 22, 23, 13, 32, 14, 2, 12, 26, 33, 9, 3, 17, 29, 20, 15, 6, 8, 21, 31, 19, 18, 28, 4, 34, 7), ((16, 18), (28, 30), (13, 14), (7, 11), (14, 16), (22, 24), (16, 17), (30, 33), (11, 14), (4, 5), (14, 16), (29, 30), (27, 29), (10, 11), (26, 27), (24, 26), (23, 24), (16, 23), (15, 16), (13, 15), (11, 13), (9, 11), (8, 9), (5, 8), (13, 14), (23, 25), (3, 5), (25, 28), (14, 17), (8, 10), (22, 23), (23, 25), (17, 18), (1, 3), (3, 4), (4, 6), (6, 8), (8, 9), (9, 14), (14, 15), (15, 17), (17, 19), (19, 20), (20, 23), (23, 24), (24, 26), (26, 27), (27, 31), (31, 32), (32, 34), (18, 20), (13, 15), (5, 6), (20, 21), (15, 18), (25, 27), (18, 20), (14, 15), (6, 9), (12, 14), (17, 18), (20, 25), (9, 10), (27, 28), (10, 12), (12, 13), (18, 20), (8, 10), (30, 32), (25, 27), (13, 18), (7, 8), (18, 19), (24, 25), (10, 13), (19, 21), (27, 30), (21, 22), (8, 10), (25, 27), (22, 25), (13, 14), (20, 22), (17, 20), (16, 17), (14, 16), (25, 26), (12, 14), (22, 23), (10, 12), (9, 10), (30, 31), (26, 28), (20, 22), (16, 18), (12, 13), (2, 9), (18, 20), (28, 30), (17, 18), (20, 21), (27, 28), (9, 12), (12, 17), (17, 20), (20, 27), (11, 12), (8, 9), (16, 17), (9, 11), (27, 29), (19, 20), (17, 19), (11, 13), (15, 17), (7, 9), (13, 15), (19, 21), (12, 13), (15, 16), (6, 7), (9, 12), (16, 19), (21, 22), (29, 33), (19, 21), (7, 9), (26, 27), (21, 23), (18, 19), (12, 16), (9, 10), (5, 7), (23, 24), (27, 29), (16, 18), (10, 12), (24, 27), (22, 24), (15, 16), (12, 13), (18, 22), (4, 5), (27, 28), (16, 18), (24, 25), (13, 16), (11, 13), (1, 2), (7, 11), (16, 17), (22, 24), (28, 30), (13, 14)))
db_wiring_CEL[(34, 612, 2)] = ((1, 3, 5, 7, 9, 11, 13, 15, 17, 2, 29, 25, 33, 21, 20, 34, 24, 30, 28, 26, 32, 22, 19, 18, 23, 31, 27, 4, 6, 8, 10, 12, 14, 16), ((10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (22, 24), (24, 26), (26, 27), (9, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (22, 24), (24, 26), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (22, 24), (24, 25), (7, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (22, 24), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (22, 23), (5, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 21), (3, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 19), (1, 2), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 34), (17, 18), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (1, 3), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (2, 3), (19, 20), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (19, 21), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (4, 5), (21, 22), (19, 21), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (21, 23), (19, 21), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (6, 7), (23, 24), (21, 23), (19, 21), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9)))
db_wiring_CEL[(35, 680, 1)] = ((22, 12, 25, 18, 4, 20, 19, 10, 11, 34, 31, 14, 33, 8, 27, 13, 32, 15, 16, 24, 26, 3, 21, 2, 29, 7, 6, 5, 28, 1, 30, 23, 9, 35, 17), ((31, 32), (22, 26), (15, 18), (26, 27), (9, 13), (18, 19), (21, 22), (13, 15), (27, 31), (15, 16), (16, 18), (25, 27), (24, 25), (22, 24), (2, 5), (18, 22), (8, 9), (27, 28), (22, 23), (17, 18), (14, 17), (23, 27), (12, 14), (27, 29), (11, 12), (21, 23), (5, 6), (9, 11), (29, 30), (26, 27), (23, 24), (20, 21), (6, 9), (24, 26), (9, 10), (10, 13), (13, 15), (15, 16), (16, 20), (4, 6), (20, 24), (24, 25), (14, 16), (25, 29), (12, 14), (16, 17), (6, 7), (14, 16), (11, 12), (3, 4), (29, 33), (19, 20), (16, 19), (7, 11), (19, 21), (11, 14), (18, 19), (21, 22), (14, 16), (33, 34), (16, 18), (18, 21), (13, 14), (10, 11), (21, 25), (11, 13), (28, 29), (25, 26), (26, 28), (20, 21), (1, 3), (3, 7), (7, 8), (8, 11), (11, 12), (12, 16), (16, 18), (18, 20), (20, 22), (22, 23), (23, 26), (26, 27), (27, 30), (30, 31), (31, 33), (33, 35), (21, 23), (19, 21), (15, 16), (25, 27), (16, 19), (6, 8), (14, 16), (8, 9), (13, 14), (16, 17), (19, 20), (29, 31), (23, 25), (9, 13), (5, 6), (20, 23), (13, 16), (6, 9), (23, 24), (16, 20), (27, 29), (20, 21), (12, 13), (4, 6), (21, 23), (15, 16), (9, 10), (13, 15), (31, 33), (23, 27), (10, 13), (13, 14), (22, 23), (19, 22), (18, 19), (8, 10), (27, 28), (14, 18), (6, 8), (12, 14), (2, 4), (10, 12), (18, 20), (28, 31), (8, 10), (20, 21), (7, 8), (26, 28), (10, 11), (17, 18), (25, 26), (21, 25), (4, 7), (18, 21), (28, 29), (16, 18), (14, 16), (11, 14), (7, 11), (3, 4), (14, 15), (21, 22), (25, 28), (11, 12), (31, 32), (6, 7), (12, 14), (14, 17), (4, 6), (17, 19), (24, 25), (19, 21), (13, 14), (21, 24), (16, 17), (24, 26), (17, 19), (26, 27), (19, 21), (1, 4), (18, 19), (23, 24), (14, 18), (27, 31), (21, 23)))
db_wiring_CEL[(36, 684, 1)] = ((18, 1, 17, 9, 16, 8, 15, 7, 14, 33, 32, 6, 31, 34, 30, 35, 29, 36, 28, 19, 27, 13, 26, 20, 25, 21, 24, 22, 23, 5, 12, 4, 11, 3, 10, 2), ((12, 14), (22, 24), (14, 16), (24, 26), (16, 18), (26, 28), (28, 29), (18, 20), (9, 10), (10, 12), (20, 22), (12, 14), (22, 24), (14, 16), (24, 26), (26, 28), (16, 18), (18, 20), (8, 10), (10, 12), (20, 22), (12, 14), (22, 24), (14, 16), (24, 26), (26, 27), (16, 18), (7, 8), (8, 10), (18, 20), (10, 12), (20, 22), (12, 14), (22, 24), (24, 26), (14, 16), (16, 18), (6, 8), (8, 10), (18, 20), (10, 12), (20, 22), (22, 24), (24, 25), (12, 14), (14, 16), (16, 18), (5, 6), (6, 8), (18, 20), (8, 10), (20, 22), (22, 24), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (4, 6), (20, 22), (22, 23), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 22), (3, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (20, 21), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 20), (1, 2), (2, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 16), (16, 18), (18, 19), (19, 36), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (1, 3), (19, 20), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (2, 3), (19, 21), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (5, 7), (3, 5), (21, 22), (19, 21), (17, 19), (15, 17), (13, 15), (11, 13), (9, 11), (7, 9), (21, 23), (19, 21), (5, 7), (4, 5), (17, 19), (15, 17), (13, 15), (11, 13), (23, 24), (21, 23), (9, 11), (19, 21), (7, 9), (17, 19), (5, 7), (15, 17), (13, 15), (23, 25), (21, 23), (11, 13), (19, 21), (9, 11), (7, 9), (17, 19), (6, 7), (15, 17), (25, 26), (23, 25), (13, 15), (21, 23), (11, 13), (19, 21), (9, 11), (7, 9), (17, 19), (15, 17), (25, 27)))
db_wiring_CEL[(37, 720, 1)] = ((8, 29, 30, 13, 5, 25, 12, 37, 6, 33, 26, 18, 17, 1, 21, 20, 22, 35, 2, 34, 16, 23, 3, 15, 32, 27, 7, 11, 36, 24, 28, 4, 10, 14, 31, 19, 9), ((25, 26), (8, 10), (20, 22), (16, 18), (28, 29), (26, 28), (10, 12), (24, 26), (15, 16), (22, 24), (6, 8), (21, 22), (12, 13), (18, 21), (16, 18), (24, 25), (13, 16), (28, 30), (11, 13), (16, 17), (25, 28), (17, 19), (8, 11), (19, 20), (13, 14), (7, 8), (30, 31), (20, 25), (11, 13), (28, 30), (5, 7), (18, 20), (25, 26), (26, 28), (10, 11), (28, 29), (13, 18), (20, 21), (2, 3), (24, 26), (29, 32), (23, 24), (18, 20), (11, 13), (20, 23), (3, 5), (17, 18), (23, 25), (5, 6), (18, 20), (32, 33), (6, 11), (11, 12), (12, 14), (14, 15), (15, 18), (18, 19), (19, 21), (21, 23), (23, 24), (24, 29), (29, 30), (30, 32), (32, 34), (20, 21), (13, 15), (34, 35), (15, 16), (10, 13), (21, 24), (28, 30), (13, 15), (9, 10), (24, 25), (1, 3), (3, 6), (6, 7), (7, 9), (9, 11), (11, 13), (13, 14), (14, 21), (21, 22), (22, 24), (24, 26), (26, 28), (28, 29), (29, 32), (32, 34), (34, 37), (10, 11), (25, 26), (20, 22), (22, 25), (11, 14), (5, 7), (19, 20), (33, 34), (20, 22), (31, 33), (14, 15), (30, 31), (25, 30), (24, 25), (22, 24), (21, 22), (18, 21), (17, 18), (15, 17), (13, 15), (12, 13), (7, 12), (6, 7), (4, 6), (17, 19), (2, 4), (12, 14), (19, 20), (24, 26), (14, 17), (30, 32), (17, 19), (1, 2), (13, 14), (26, 27), (4, 5), (19, 24), (16, 17), (11, 13), (29, 30), (24, 26), (5, 8), (17, 19), (8, 9), (26, 29), (9, 11), (23, 24), (11, 12), (12, 17), (24, 26), (29, 31), (17, 18), (7, 9), (18, 20), (20, 21), (21, 24), (24, 25), (6, 7), (25, 27), (19, 21), (9, 12), (27, 29), (16, 19), (29, 30), (21, 22), (12, 13), (15, 16), (26, 27), (7, 9), (13, 15), (19, 21), (11, 13), (15, 17), (30, 35), (17, 19), (9, 11), (19, 20), (8, 9), (16, 17), (11, 12), (27, 30), (20, 27), (17, 20), (12, 17), (9, 12), (2, 9), (17, 18), (20, 21), (27, 28), (30, 31), (18, 20)))
db_wiring_CEL[(37, 720, 2)] = ((18, 1, 17, 9, 16, 8, 15, 7, 14, 6, 32, 33, 31, 34, 30, 35, 29, 36, 28, 37, 19, 27, 13, 26, 20, 25, 21, 24, 22, 23, 5, 12, 4, 11, 3, 10, 2), ((23, 25), (10, 12), (12, 14), (25, 27), (14, 16), (27, 29), (29, 30), (16, 18), (18, 21), (21, 23), (9, 10), (10, 12), (23, 25), (12, 14), (25, 27), (27, 29), (14, 16), (16, 18), (18, 19), (19, 21), (21, 23), (8, 10), (10, 12), (23, 25), (12, 14), (25, 27), (27, 28), (14, 16), (16, 19), (19, 21), (7, 8), (21, 23), (8, 10), (10, 12), (23, 25), (25, 27), (12, 14), (14, 16), (16, 17), (17, 19), (19, 21), (6, 8), (21, 23), (8, 10), (23, 25), (25, 26), (10, 12), (12, 14), (14, 17), (17, 19), (19, 21), (21, 23), (23, 25), (5, 6), (6, 8), (8, 10), (10, 12), (12, 14), (14, 15), (15, 17), (17, 19), (19, 21), (21, 23), (23, 24), (4, 6), (6, 8), (8, 10), (10, 12), (12, 15), (15, 17), (17, 19), (19, 21), (21, 23), (3, 4), (4, 6), (6, 8), (8, 10), (10, 12), (12, 13), (13, 15), (15, 17), (17, 19), (19, 21), (21, 22), (2, 4), (4, 6), (6, 8), (8, 10), (10, 13), (13, 15), (15, 17), (17, 19), (19, 21), (1, 2), (2, 4), (4, 6), (6, 8), (8, 10), (10, 11), (11, 13), (13, 15), (15, 17), (17, 19), (19, 20), (20, 37), (18, 20), (16, 18), (14, 16), (12, 14), (9, 12), (7, 9), (5, 7), (3, 5), (1, 3), (20, 21), (18, 20), (16, 18), (14, 16), (12, 14), (11, 12), (9, 11), (7, 9), (5, 7), (3, 5), (2, 3), (20, 22), (18, 20), (16, 18), (14, 16), (11, 14), (9, 11), (7, 9), (5, 7), (3, 5), (22, 23), (20, 22), (18, 20), (16, 18), (14, 16), (13, 14), (11, 13), (9, 11), (7, 9), (5, 7), (4, 5), (22, 24), (20, 22), (18, 20), (16, 18), (13, 16), (11, 13), (9, 11), (7, 9), (24, 25), (22, 24), (5, 7), (20, 22), (18, 20), (16, 18), (15, 16), (13, 15), (11, 13), (24, 26), (9, 11), (22, 24), (7, 9), (6, 7), (20, 22), (18, 20), (15, 18), (13, 15), (26, 27), (24, 26), (11, 13), (22, 24), (9, 11), (7, 9), (20, 22), (18, 20), (17, 18), (15, 17)))
db_wiring_CEL[(37, 720, 3)] = ((37, 23, 12, 24, 27, 13, 29, 28, 35, 31, 30, 3, 25, 21, 4, 36, 33, 5, 22, 16, 6, 1, 34, 7, 19, 14, 8, 2, 18, 9, 15, 32, 10, 20, 26, 17, 11), ((13, 17), (19, 23), (31, 32), (29, 31), (7, 11), (17, 19), (25, 29), (16, 17), (11, 13), (19, 20), (13, 14), (23, 25), (14, 16), (22, 23), (20, 22), (10, 11), (16, 20), (25, 26), (15, 16), (20, 21), (11, 15), (21, 25), (9, 11), (1, 2), (8, 9), (25, 27), (6, 8), (11, 12), (15, 17), (27, 28), (19, 21), (24, 25), (28, 30), (17, 19), (2, 6), (6, 7), (7, 11), (11, 13), (13, 15), (15, 17), (17, 18), (18, 20), (20, 22), (22, 24), (24, 28), (28, 29), (29, 33), (14, 15), (19, 20), (33, 34), (12, 14), (20, 22), (5, 7), (10, 12), (22, 24), (27, 29), (7, 8), (12, 13), (21, 22), (26, 27), (8, 10), (24, 26), (34, 37), (29, 30), (22, 24), (10, 12), (4, 5), (32, 34), (30, 32), (26, 30), (24, 26), (23, 24), (12, 23), (11, 12), (9, 11), (5, 9), (3, 5), (1, 3), (30, 31), (23, 25), (11, 13), (5, 6), (25, 27), (9, 11), (27, 28), (22, 23), (28, 30), (13, 14), (23, 25), (8, 9), (34, 35), (11, 13), (6, 8), (21, 23), (13, 15), (30, 34), (20, 21), (15, 16), (29, 30), (25, 29), (23, 25), (21, 23), (19, 21), (18, 19), (16, 18), (14, 16), (12, 14), (8, 12), (7, 8), (18, 20), (3, 7), (29, 31), (25, 26), (28, 29), (20, 22), (16, 18), (12, 13), (26, 28), (7, 9), (9, 10), (22, 26), (10, 12), (12, 16), (21, 22), (26, 27), (16, 17), (17, 21), (2, 3), (21, 23), (11, 12), (23, 24), (15, 17), (24, 26), (14, 15), (20, 21), (26, 30), (17, 18), (12, 14), (18, 20), (30, 32), (8, 12), (32, 33), (20, 24), (14, 18), (29, 30), (6, 8), (24, 26), (33, 36), (26, 27), (18, 20), (12, 14), (5, 6), (23, 24), (8, 9), (20, 21), (17, 18), (11, 12), (14, 15), (30, 33), (27, 30), (24, 27), (21, 24), (18, 21), (15, 18), (12, 15), (9, 12), (6, 9), (24, 25), (27, 28), (21, 22), (18, 19), (30, 31), (15, 16), (33, 34), (25, 27), (19, 21), (12, 13), (3, 6), (13, 15), (31, 33), (9, 10)))
