# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['beanie']

package_data = \
{'': ['*']}

install_requires = \
['motor>=2.1.0,<3.0.0', 'pydantic>=1.5.1,<2.0.0']

setup_kwargs = {
    'name': 'beanie',
    'version': '0.1.1',
    'description': 'Pydantic-based Mongo ORM',
    'long_description': '============\nDescription\n============\n\n\nBeanie - is asynchronous ORM for MongoDB, based on `Motor <https://motor.readthedocs.io/en/stable/>`_ and `Pydantic <https://pydantic-docs.helpmanual.io/>`_.\n\nBeanie uses an abstraction over Pydantic Models Motor collections to work with mongo. Document and Collection classes allow to create, replace, update, get, find and aggregate.\n\nOne collection can be associated with only one Document and it helps to keep it structured.\n\nHere you can see, how to use Beanie in simple examples:\n\n============\nInstallation\n============\n\n----\nPIP\n----\n\n.. code-block:: bash\n\n    pip install beanie\n\n\n------\nPoetry\n------\n\n.. code-block:: bash\n\n    poetry add beanie\n\n============\nUsage\n============\n\n-----\nInit\n-----\n\n.. code-block:: python\n\n    from typing import List\n\n    import motor\n    from pydantic import BaseSettings, BaseModel\n\n    from collections import Collection\n    from documents import Document\n\n    # CREATE MOTOR CLIENT AND DB\n\n    client = motor.motor_asyncio.AsyncIOMotorClient(\n        "mongodb://user:pass@host:27017/db",\n        serverSelectionTimeoutMS=100\n    )\n    db = client.beanie_db\n\n\n    # CREATE BEANIE DOCUMENT STRUCTURE\n\n    class SubDocument(BaseModel):\n        test_str: str\n\n\n    class DocumentTestModel(Document):\n        test_int: int\n        test_list: List[SubDocument]\n        test_str: str\n\n\n    # CREATE BEANIE COLLECTION WITH DocumentTestModel STRUCTURE\n\n    test_collection = Collection(\n        name="test_collection", db=db, document_model=DocumentTestModel\n    )\n\n\n---------\nDocuments\n---------\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^\nCreate a document (Insert)\n^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    document = DocumentTestModel(\n        test_int=42,\n        test_list=[SubDocument(test_str="foo"), SubDocument(test_str="bar")],\n        test_str="kipasa",\n    )\n\n    await document.create()\n\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nReplace the document (full update)\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    document.test_str = "REPLACED_VALUE"\n    await document.replace()\n\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nUpdate the document (partial update)\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nin this example, I\'ll add an item to the document\'s "test_list" field\n\n.. code-block:: python\n\n    to_insert = SubDocument(test_str="test")\n    await document.update(update_query={"$push": {"test_list": to_insert.dict()}})\n\n^^^^^^^^^^^^^^^^^^^^^^^\nGet the document\n^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    document = await DocumentTestModel.get(DOCUMENT_ID)\n\n^^^^^^^^^^^^^^^^^^^^^^^\nFind one document\n^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    document = await DocumentTestModel.find_one({"test_str": "kipasa"})\n\n^^^^^^^^^^^^^^^^^^^^^^^\nFind the documents\n^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    async for document in DocumentTestModel.find({"test_str": "uno"}):\n        print(document)\n\nOR\n\n.. code-block:: python\n\n    documents =  await DocumentTestModel.find({"test_str": "uno"}).to_list()\n\n^^^^^^^^^^^^^^^^^^^^^^^\nGet all the documents\n^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    async for document in DocumentTestModel.all()\n        print(document)\n\nOR\n\n.. code-block:: python\n\n    documents = await DocumentTestModel.all().to_list()\n\n^^^^^^^^^^^^^^^^^^^^^^^\nDelete the document\n^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    await document.delete()\n\n^^^^^^^^^^^^^^^^^^^^^^^\nDelete many documents\n^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    await DocumentTestModel.delete_many({"test_str": "wrong"})\n\n^^^^^^^^^^^^^^^^^^^^^^^^^\nDelete all the documents\n^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    await DocumentTestModel.delete_all()\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nAggregate from the document model\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    async for item in DocumentTestModel.aggregate(\n        [{"$group": {"_id": "$test_str", "total": {"$sum": "$test_int"}}}]\n    ):\n        print(item)\n\nOR\n\n.. code-block:: python\n\n    class OutputItem(BaseModel):\n        id: str = Field(None, alias="_id")\n        total: int\n\n    async for item in DocumentTestModel.aggregate(\n        [{"$group": {"_id": "$test_str", "total": {"$sum": "$test_int"}}}],\n        item_model=OutputModel\n    ):\n        print(item)\n\nOR\n\n.. code-block:: python\n\n    results = await DocumentTestModel.aggregate(\n        [{"$group": {"_id": "$test_str", "total": {"$sum": "$test_int"}}}],\n        item_model=OutputModel\n    ).to_list()\n\n\n\n------------\nCollections\n------------\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nInsert the document into the collection\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    inserted_document = await collection.insert_one(document)\n\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nReplace the document\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n     await collection.replace_one(document)\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nUpdate the document\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    to_insert = SubDocument(test_str="test")\n    await collection.update_one(\n        document, update_query={"$push": {"test_list": to_insert.dict()}}\n    )\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nUpdate many documents\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    await collection.update_many(\n        update_query={"$set": {"test_int": 100}}, filter_query={"test_str": "kipasa"},\n    )\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nDelete the document\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    await collection.delete_one(document)\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nDelete many documents\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    await collection.delete_many({"test_str": "uno"})\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nDelete all the documents\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    await collection.delete_all()\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nGet the document\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    document = await collection.get_one(DOCUMENT_ID)\n\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nFind the document\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    document = await collection.find_one({"test_str": "one"})\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nFind many documents\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    async for document in collection.find({"test_str": "uno"}):\n        print(document)\n\nOR\n\n.. code-block:: python\n\n    documents = await collection.find({"test_str": "uno"}).to_list()\n\nOR\n\n.. code-block:: python\n\n    documents = await collection.find({"test_str": "uno"}).to_list(length=10)\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nGet all the documents from the collection\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    async for document in collection.all():\n        print(document)\n\nOR\n\n.. code-block:: python\n\n    documents = await collection.all().to_list()\n\n\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\nAggregate\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n.. code-block:: python\n\n    async for item in collection.aggregate(\n        [{"$group": {"_id": "$test_str", "total": {"$sum": "$test_int"}}}]\n    ):\n        print(item)\n\nOR\n\n.. code-block:: python\n\n    class OutputItem(BaseModel):\n        id: str = Field(None, alias="_id")\n        total: int\n\n    async for item in collection.aggregate(\n        [{"$group": {"_id": "$test_str", "total": {"$sum": "$test_int"}}}],\n        item_model=OutputModel\n    ):\n        print(item)\n\nOR\n\n.. code-block:: python\n\n\n    results = await collection.aggregate(\n        [{"$group": {"_id": "$test_str", "total": {"$sum": "$test_int"}}}],\n        item_model=OutputModel\n    ).to_list():\n',
    'author': 'Roman',
    'author_email': 'roman-right@protonmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/roman-right/beanie',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
