"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedCodeDeployedFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_codedeploy_1 = require("aws-cdk-lib/aws-codedeploy");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_ecs_patterns_1 = require("aws-cdk-lib/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_synthetics_1 = require("aws-cdk-lib/aws-synthetics");
const api_canary_1 = require("../api-canary");
const ecs_deployment_1 = require("../ecs-deployment");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer and deployed by CodeDeploy.
 */
class ApplicationLoadBalancedCodeDeployedFargateService extends aws_ecs_patterns_1.ApplicationLoadBalancedFargateService {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedCodeDeployedFargateService class.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            deploymentController: {
                type: aws_ecs_1.DeploymentControllerType.CODE_DEPLOY,
            },
        });
        if (props.deregistrationDelay) {
            this.targetGroup.setAttribute('deregistration_delay.timeout_seconds', props.deregistrationDelay.toSeconds().toString());
        }
        if (props.targetHealthCheck) {
            this.targetGroup.configureHealthCheck(props.targetHealthCheck);
        }
        this.accessLogBucket = props.accessLogBucket ??
            new aws_s3_1.Bucket(this, 'AccessLogBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                enforceSSL: true,
            });
        this.loadBalancer.logAccessLogs(this.accessLogBucket, props.accessLogPrefix);
        const alarms = [];
        if (props.responseTimeAlarmThreshold) {
            const responseTimeAlarm = new aws_cloudwatch_1.Alarm(this, 'ResponseTimeAlarm', {
                metric: this.loadBalancer.metrics.targetResponseTime({
                    period: aws_cdk_lib_1.Duration.minutes(1),
                    statistic: 'p95',
                }),
                evaluationPeriods: 2,
                threshold: props.responseTimeAlarmThreshold.toSeconds(),
                comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            });
            alarms.push(responseTimeAlarm);
        }
        const protocol = props.protocol ?? (props.certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
        const testHostName = props.domainName ? props.domainName : this.loadBalancer.loadBalancerDnsName;
        if (props.apiTestSteps?.length) {
            this.apiCanary = new api_canary_1.ApiCanary(this, 'Canary', {
                baseUrl: `${protocol.toLowerCase()}://${testHostName}`,
                durationAlarmThreshold: props.apiCanaryTimeout,
                schedule: aws_synthetics_1.Schedule.rate(props.apiCanarySchedule ?? aws_cdk_lib_1.Duration.minutes(5)),
                threadCount: props.apiCanaryThreadCount,
                steps: props.apiTestSteps,
            });
            this.apiCanary.node.addDependency(this.service);
            alarms.push(this.apiCanary.successAlarm);
            if (this.apiCanary.durationAlarm) {
                alarms.push(this.apiCanary.durationAlarm);
            }
        }
        if (alarms.length > 0) {
            this.healthAlarm = new aws_cloudwatch_1.CompositeAlarm(this, 'HealthAlarm', {
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...alarms),
            });
        }
        else {
            this.healthAlarm = undefined;
        }
        let testPort;
        if (props.testPort) {
            testPort = props.testPort;
        }
        else if (props.listenerPort) {
            testPort = props.listenerPort + 1;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            testPort = 8080;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            testPort = 8443;
        }
        else {
            throw new Error('Unable to determine port for test listener');
        }
        let certificates;
        if (props.certificate) {
            certificates = [props.certificate];
        }
        this.testListener = this.loadBalancer.addListener('TestListener', {
            protocol,
            port: testPort,
            open: props.openListener ?? true,
            sslPolicy: props.sslPolicy,
            certificates: certificates,
        });
        this.greenTargetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(this, 'GreenTargetGroup', {
            vpc: this.cluster.vpc,
            port: testPort,
            protocol: props.targetProtocol ?? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            protocolVersion: props.protocolVersion,
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.targetHealthCheck,
            targetType: aws_elasticloadbalancingv2_1.TargetType.IP,
        });
        this.listener.node.addDependency(this.greenTargetGroup);
        this.testListener.addTargetGroups('ECS', {
            targetGroups: [this.greenTargetGroup],
        });
        this.application = new aws_codedeploy_1.EcsApplication(this, 'Application', {
            applicationName: props.applicationName,
        });
        this.deploymentGroup = new aws_codedeploy_1.EcsDeploymentGroup(this, 'DeploymentGroup', {
            application: this.application,
            deploymentGroupName: props.deploymentGroupName,
            alarms: this.healthAlarm ? [this.healthAlarm] : undefined,
            service: this.service,
            blueGreenDeploymentConfig: {
                blueTargetGroup: this.targetGroup,
                greenTargetGroup: this.greenTargetGroup,
                listener: this.listener,
                testListener: this.testListener,
                terminationWaitTime: props.terminationWaitTime ?? aws_cdk_lib_1.Duration.minutes(10),
            },
            deploymentConfig: props.deploymentConfig ?? aws_codedeploy_1.EcsDeploymentConfig.ALL_AT_ONCE,
            autoRollback: {
                stoppedDeployment: true,
            },
        });
        this.deployment = new ecs_deployment_1.EcsDeployment({
            deploymentGroup: this.deploymentGroup,
            timeout: props.deploymentTimeout ?? aws_cdk_lib_1.Duration.minutes(60),
            targetService: {
                taskDefinition: this.taskDefinition,
                containerName: this.taskDefinition.defaultContainer.containerName,
                containerPort: this.taskDefinition.defaultContainer.containerPort,
            },
            hooks: props.hooks,
        });
    }
    addServiceAsTarget(service) {
        super.addServiceAsTarget(service);
    }
}
exports.ApplicationLoadBalancedCodeDeployedFargateService = ApplicationLoadBalancedCodeDeployedFargateService;
_a = JSII_RTTI_SYMBOL_1;
ApplicationLoadBalancedCodeDeployedFargateService[_a] = { fqn: "@cdklabs/cdk-ecs-codedeploy.ApplicationLoadBalancedCodeDeployedFargateService", version: "0.0.339" };
//# sourceMappingURL=data:application/json;base64,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