"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const construct_1 = require("./construct");
const lazy_1 = require("./lazy");
const token_1 = require("./token");
const CFN_ELEMENT_SYMBOL = Symbol.for('@aws-cdk/core.CfnElement');
/**
 * An element of a CloudFormation stack.
 */
class CfnElement extends construct_1.Construct {
    /**
     * Creates an entity and binds it to a tree.
     * Note that the root of the tree must be a Stack object (not just any Root).
     *
     * @param scope The parent construct
     * @param props Construct properties
     */
    constructor(scope, id) {
        super(scope, id);
        Object.defineProperty(this, CFN_ELEMENT_SYMBOL, { value: true });
        this.stack = stack_1.Stack.of(this);
        this.logicalId = lazy_1.Lazy.stringValue({ produce: () => this.synthesizeLogicalId() }, {
            displayHint: `${notTooLong(this.node.path)}.LogicalID`
        });
        this.node.addMetadata(cxapi.LOGICAL_ID_METADATA_KEY, this.logicalId, this.constructor);
    }
    /**
     * Returns `true` if a construct is a stack element (i.e. part of the
     * synthesized cloudformation template).
     *
     * Uses duck-typing instead of `instanceof` to allow stack elements from different
     * versions of this library to be included in the same stack.
     *
     * @returns The construct as a stack element or undefined if it is not a stack element.
     */
    static isCfnElement(x) {
        return CFN_ELEMENT_SYMBOL in x;
    }
    /**
     * Overrides the auto-generated logical ID with a specific ID.
     * @param newLogicalId The new logical ID to use for this stack element.
     */
    overrideLogicalId(newLogicalId) {
        this._logicalIdOverride = newLogicalId;
    }
    /**
     * @returns the stack trace of the point where this Resource was created from, sourced
     *      from the +metadata+ entry typed +aws:cdk:logicalId+, and with the bottom-most
     *      node +internal+ entries filtered.
     */
    get creationStack() {
        const trace = this.node.metadata.find(md => md.type === cxapi.LOGICAL_ID_METADATA_KEY).trace;
        if (!trace) {
            return [];
        }
        return filterStackTrace(trace);
        function filterStackTrace(stack) {
            const result = Array.of(...stack);
            while (result.length > 0 && shouldFilter(result[result.length - 1])) {
                result.pop();
            }
            // It's weird if we filtered everything, so return the whole stack...
            return result.length === 0 ? stack : result;
        }
        function shouldFilter(str) {
            return str.match(/[^(]+\(internal\/.*/) !== null;
        }
    }
    /**
     * Called during synthesize to render the logical ID of this element. If
     * `overrideLogicalId` was it will be used, otherwise, we will allocate the
     * logical ID through the stack.
     */
    synthesizeLogicalId() {
        if (this._logicalIdOverride) {
            return this._logicalIdOverride;
        }
        else {
            return this.stack.getLogicalId(this);
        }
    }
}
exports.CfnElement = CfnElement;
/**
 * Base class for referenceable CloudFormation constructs which are not Resources
 *
 * These constructs are things like Conditions and Parameters, can be
 * referenced by taking the `.ref` attribute.
 *
 * Resource constructs do not inherit from CfnRefElement because they have their
 * own, more specific types returned from the .ref attribute. Also, some
 * resources aren't referenceable at all (such as BucketPolicies or GatewayAttachments).
 */
class CfnRefElement extends CfnElement {
    /**
     * Return a string that will be resolved to a CloudFormation `{ Ref }` for this element.
     *
     * If, by any chance, the intrinsic reference of a resource is not a string, you could
     * coerce it to an IResolvable through `Lazy.any({ produce: resource.ref })`.
     */
    get ref() {
        return token_1.Token.asString(cfn_reference_1.CfnReference.for(this, 'Ref'));
    }
}
exports.CfnRefElement = CfnRefElement;
function notTooLong(x) {
    if (x.length < 100) {
        return x;
    }
    return x.substr(0, 47) + '...' + x.substr(x.length - 47);
}
const cfn_reference_1 = require("./private/cfn-reference");
const stack_1 = require("./stack");
//# sourceMappingURL=data:application/json;base64,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