"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const constructs_1 = require("constructs");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * @summary The ApiGatewayToSageMakerEndpoint class.
 */
class ApiGatewayToSageMakerEndpoint extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToSageMakerEndpoint class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToSageMakerEndpointProps} props - user provided props for the construct.
     * @since 1.68.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // CheckSagemakerProps is not called because this construct can't create a Sagemaker resource
        // Setup the API Gateway
        const globalRestApiResponse = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        this.apiGateway = globalRestApiResponse.api;
        this.apiGatewayCloudWatchRole = globalRestApiResponse.role;
        this.apiGatewayLogGroup = globalRestApiResponse.logGroup;
        // Setup the API Gateway role
        if (props.apiGatewayExecutionRole !== undefined) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
            });
            defaults.addCfnGuardSuppressRules(this.apiGatewayRole, ["IAM_NO_INLINE_POLICY_CHECK"]);
            // Setup the IAM policy for SageMaker endpoint
            const invokePolicy = new iam.Policy(this, 'InvokeEndpointPolicy', {
                statements: [
                    new iam.PolicyStatement({
                        actions: ['sagemaker:InvokeEndpoint'],
                        resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:sagemaker:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:endpoint/${props.endpointName}`]
                    })
                ]
            });
            invokePolicy.attachToRole(this.apiGatewayRole);
        }
        // Setup request validation
        const requestValidator = this.apiGateway.addRequestValidator('request-validator', {
            requestValidatorName: 'request-param-validator',
            // Setting this property to true makes sure the following are validated:
            // - Required request parameters in the URI
            // - Query string
            // - Headers
            validateRequestParameters: true
        });
        // Setup method and integration responses
        const methodResponses = [
            { statusCode: '200' },
            { statusCode: '500' },
            { statusCode: '400' }
        ];
        const integResponses = [];
        if (props.responseMappingTemplate !== undefined) {
            integResponses.push({
                statusCode: '200',
                responseTemplates: { 'application/json': props.responseMappingTemplate }
            });
        }
        else {
            integResponses.push({ statusCode: '200' });
        }
        integResponses.push({ statusCode: '500', selectionPattern: '5\\d{2}' }, { statusCode: '400', selectionPattern: '4\\d{2}' });
        // The SageMaker integration can be added either at the root of the API (GET https://execute-api.amazonaws.com/{some-param}),
        // or as a sub-resource (GET https://execute-api.amazonaws.com/inference/{some-param}).
        // The following lines will make sure only the necessary resources are created.
        let apiResource = this.apiGateway.root;
        if (props.resourceName !== undefined) {
            apiResource = apiResource.addResource(props.resourceName);
        }
        apiResource = apiResource.addResource(props.resourcePath);
        // Setup API Gateway method
        defaults.addProxyMethodToApiResource({
            service: 'runtime.sagemaker',
            path: `endpoints/${props.endpointName}/invocations`,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'GET',
            apiResource,
            requestTemplate: props.requestMappingTemplate,
            additionalRequestTemplates: props.additionalRequestTemplates,
            awsIntegrationProps: {
                options: { integrationResponses: integResponses }
            },
            methodOptions: {
                methodResponses,
                requestValidator,
            }
        });
    }
}
exports.ApiGatewayToSageMakerEndpoint = ApiGatewayToSageMakerEndpoint;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToSageMakerEndpoint[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-sagemakerendpoint.ApiGatewayToSageMakerEndpoint", version: "2.65.0" };
//# sourceMappingURL=data:application/json;base64,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