"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const iam = require("aws-cdk-lib/aws-iam");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename));
stack.templateOptions.description = 'Integration Test for aws-apigateway-sagemakerendpoint';
const existingRole = new iam.Role(stack, 'api-gateway-role', {
    assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
    description: 'existing role for SageMaker integration',
    inlinePolicies: {
        InvokePolicy: new iam.PolicyDocument({
            statements: [new iam.PolicyStatement({
                    resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:sagemaker:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:endpoint/my-endpoint`],
                    actions: ['sagemaker:InvokeEndpoint']
                })]
        })
    }
});
(0, core_1.addCfnGuardSuppressRules)(existingRole, ["IAM_NO_INLINE_POLICY_CHECK"]);
// Definitions
const requestTemplate = `{
    "instances": [
#set( $user_id = $input.params("user_id") )
#set( $items = $input.params("items") )
#foreach( $item in $items.split(",") )
    {"in0": [$user_id], "in1": [$item]}#if( $foreach.hasNext ),#end
    $esc.newline
#end
    ]
}`;
const responseTemplate = `{
    "ratings": [
#set( $predictions = $input.path("$.predictions") )
#foreach( $item in $predictions )
    $item.scores[0]#if( $foreach.hasNext ),#end
    $esc.newline
#end
    ]
}`;
const props = {
    endpointName: 'my-endpoint',
    resourcePath: '{user_id}',
    resourceName: 'predicted-ratings',
    requestMappingTemplate: requestTemplate,
    responseMappingTemplate: responseTemplate,
    apiGatewayExecutionRole: existingRole
};
new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'test-apigateway-sagemakerendpoint-overwrite', props);
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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