"""
Tests for GitHub Commit Message Generator

Tests commit message generation logic and git status collection.
"""

import pytest
from pathlib import Path
from unittest.mock import patch, MagicMock
from tempfile import TemporaryDirectory

from devpulse.commands.github_commit import (
    GitStatus,
    GitStatusCollector,
    ConventionalCommitGenerator,
    CommitMessageValidator,
    apply_commit_message,
)


class TestGitStatusCollector:
    """Test git status collection."""

    @patch("devpulse.commands.github_commit.subprocess.run")
    def test_is_git_repo_valid(self, mock_run):
        """Test valid git repository detection."""
        mock_run.return_value = MagicMock(returncode=0)
        
        is_valid = GitStatusCollector._is_git_repo(Path("."))
        
        assert is_valid is True

    @patch("devpulse.commands.github_commit.subprocess.run")
    def test_is_git_repo_invalid(self, mock_run):
        """Test invalid git repository detection."""
        mock_run.return_value = MagicMock(returncode=1)
        
        is_valid = GitStatusCollector._is_git_repo(Path("."))
        
        assert is_valid is False

    @patch("devpulse.commands.github_commit.subprocess.run")
    def test_collect_no_changes(self, mock_run):
        """Test collecting status with no changes."""
        def side_effect(*args, **kwargs):
            response = MagicMock()
            response.returncode = 0
            
            if "rev-parse" in str(args):
                response.stdout = ".git"
            else:
                response.stdout = ""
                response.stderr = ""
            
            return response
        
        mock_run.side_effect = side_effect
        
        # Create temporary git repo
        with TemporaryDirectory() as tmpdir:
            git_dir = Path(tmpdir) / ".git"
            git_dir.mkdir()
            
            with patch("devpulse.commands.github_commit.GitStatusCollector._is_git_repo", return_value=True):
                with patch("devpulse.commands.github_commit.GitStatusCollector._get_staged_files", return_value=[]):
                    with patch("devpulse.commands.github_commit.GitStatusCollector._get_unstaged_files", return_value=[]):
                        status = GitStatusCollector.collect(tmpdir)
                        
                        assert status.is_git_repo is True
                        assert status.has_changes is False
                        assert status.has_staged is False
                        assert status.has_unstaged is False

    def test_collect_not_git_repo(self):
        """Test collecting from non-git directory raises error."""
        with TemporaryDirectory() as tmpdir:
            with patch("devpulse.commands.github_commit.GitStatusCollector._is_git_repo", return_value=False):
                with pytest.raises(ValueError, match="Not a git repository"):
                    GitStatusCollector.collect(tmpdir)

    @patch("devpulse.commands.github_commit.GitStatusCollector._is_git_repo", return_value=True)
    def test_collect_with_changes(self, mock_is_git):
        """Test collecting status with staged and unstaged changes."""
        with TemporaryDirectory() as tmpdir:
            with patch("devpulse.commands.github_commit.GitStatusCollector._get_staged_files") as mock_staged:
                with patch("devpulse.commands.github_commit.GitStatusCollector._get_unstaged_files") as mock_unstaged:
                    with patch("devpulse.commands.github_commit.GitStatusCollector._get_diff_stat") as mock_diff:
                        mock_staged.return_value = ["file1.py", "file2.py"]
                        mock_unstaged.return_value = ["file3.js"]
                        mock_diff.return_value = "2 files changed, 10 insertions"
                        
                        status = GitStatusCollector.collect(tmpdir)
                        
                        assert status.has_changes is True
                        assert status.has_staged is True
                        assert status.has_unstaged is True
                        assert len(status.changed_files) == 3


class TestConventionalCommitGenerator:
    """Test conventional commit message generation."""

    def test_generate_feat_message(self):
        """Test generating feat message."""
        git_status = GitStatus(
            is_git_repo=True,
            has_changes=True,
            has_staged=True,
            has_unstaged=False,
            changed_files=["auth.py"],
            staged_files=["auth.py"],
            unstaged_files=[],
            diff_stat="1 file changed"
        )
        
        generator = ConventionalCommitGenerator()
        message = generator.generate(git_status, commit_type="feat")
        
        assert "feat:" in message
        assert len(message) > 0

    def test_generate_fix_message(self):
        """Test generating fix message."""
        git_status = GitStatus(
            is_git_repo=True,
            has_changes=True,
            has_staged=True,
            has_unstaged=False,
            changed_files=["bug_fix.py"],
            staged_files=["bug_fix.py"],
            unstaged_files=[],
            diff_stat="1 file changed"
        )
        
        generator = ConventionalCommitGenerator()
        message = generator.generate(git_status, commit_type="fix")
        
        assert "fix:" in message

    def test_generate_with_scope(self):
        """Test generating message with scope."""
        git_status = GitStatus(
            is_git_repo=True,
            has_changes=True,
            has_staged=True,
            has_unstaged=False,
            changed_files=["auth.py"],
            staged_files=["auth.py"],
            unstaged_files=[],
            diff_stat="1 file changed"
        )
        
        generator = ConventionalCommitGenerator()
        message = generator.generate(git_status, commit_type="feat", scope="auth")
        
        assert "feat(auth):" in message

    def test_generate_breaking_change(self):
        """Test generating breaking change message."""
        git_status = GitStatus(
            is_git_repo=True,
            has_changes=True,
            has_staged=True,
            has_unstaged=False,
            changed_files=["api.py"],
            staged_files=["api.py"],
            unstaged_files=[],
            diff_stat="1 file changed"
        )
        
        generator = ConventionalCommitGenerator()
        message = generator.generate(git_status, commit_type="feat", breaking=True)
        
        assert "BREAKING CHANGE:" in message

    def test_invalid_commit_type(self):
        """Test that invalid commit type raises error."""
        git_status = GitStatus(
            is_git_repo=True,
            has_changes=True,
            has_staged=True,
            has_unstaged=False,
            changed_files=[],
            staged_files=[],
            unstaged_files=[],
            diff_stat=""
        )
        
        generator = ConventionalCommitGenerator()
        
        with pytest.raises(ValueError, match="Invalid commit type"):
            generator.generate(git_status, commit_type="invalid")

    def test_summary_line_length(self):
        """Test that summary line is reasonable length."""
        git_status = GitStatus(
            is_git_repo=True,
            has_changes=True,
            has_staged=True,
            has_unstaged=False,
            changed_files=["file.py"],
            staged_files=["file.py"],
            unstaged_files=[],
            diff_stat="1 file changed"
        )
        
        generator = ConventionalCommitGenerator()
        message = generator.generate(git_status, commit_type="feat")
        
        summary_line = message.split("\n")[0]
        assert len(summary_line) < 100  # Reasonable limit

    def test_body_includes_changed_files(self):
        """Test that body includes changed files."""
        git_status = GitStatus(
            is_git_repo=True,
            has_changes=True,
            has_staged=True,
            has_unstaged=False,
            changed_files=["file1.py", "file2.py"],
            staged_files=["file1.py", "file2.py"],
            unstaged_files=[],
            diff_stat="2 files changed"
        )
        
        generator = ConventionalCommitGenerator()
        message = generator.generate(git_status, commit_type="feat")
        
        assert "file1.py" in message or "file2.py" in message


class TestCommitMessageValidator:
    """Test commit message validation."""

    def test_validate_format_valid_message(self):
        """Test validation of valid message."""
        message = "feat: add authentication\n\n- added JWT token support"
        is_valid, error = CommitMessageValidator.validate_format(message)
        
        assert is_valid is True
        assert error == ""

    def test_validate_format_empty_message(self):
        """Test validation of empty message."""
        message = ""
        is_valid, error = CommitMessageValidator.validate_format(message)
        
        assert is_valid is False
        assert "empty" in error.lower()

    def test_validate_format_too_long_summary(self):
        """Test validation of overly long summary line."""
        message = "a" * 100
        is_valid, error = CommitMessageValidator.validate_format(message)
        
        assert is_valid is False
        assert "too long" in error.lower()

    def test_validate_format_too_short_summary(self):
        """Test validation of too short summary line."""
        message = "fix"
        is_valid, error = CommitMessageValidator.validate_format(message)
        
        assert is_valid is False
        assert "too short" in error.lower()

    def test_validate_conventional_valid(self):
        """Test conventional format validation for valid message."""
        message = "feat(auth): add login feature"
        is_valid, error = CommitMessageValidator.validate_conventional(message)
        
        assert is_valid is True

    def test_validate_conventional_invalid_type(self):
        """Test conventional format validation for invalid type."""
        message = "invalid(scope): some message"
        is_valid, error = CommitMessageValidator.validate_conventional(message)
        
        assert is_valid is False
        assert "Invalid commit type" in error


class TestApplyCommitMessage:
    """Test applying commit messages."""

    @patch("devpulse.commands.github_commit.subprocess.run")
    def test_apply_commit_success(self, mock_run):
        """Test successful commit application."""
        mock_run.return_value = MagicMock(
            returncode=0,
            stdout="",
            stderr=""
        )
        
        success = apply_commit_message(".", "feat: test message")
        
        assert success is True

    @patch("devpulse.commands.github_commit.subprocess.run")
    def test_apply_commit_failure(self, mock_run):
        """Test failed commit application."""
        mock_run.return_value = MagicMock(
            returncode=1,
            stdout="",
            stderr="Error: nothing to commit"
        )
        
        with patch("devpulse.commands.github_commit.console.print"):
            success = apply_commit_message(".", "feat: test message")
        
        assert success is False

    @patch("devpulse.commands.github_commit.subprocess.run")
    def test_apply_commit_exception(self, mock_run):
        """Test exception handling during commit."""
        mock_run.side_effect = Exception("Test error")
        
        with patch("devpulse.commands.github_commit.console.print"):
            success = apply_commit_message(".", "feat: test message")
        
        assert success is False


class TestCommitTypeValidation:
    """Test commit type validation."""

    def test_valid_types(self):
        """Test all valid commit types."""
        valid_types = ["feat", "fix", "docs", "style", "refactor", "perf", "test", "chore", "ci", "build"]
        
        for commit_type in valid_types:
            assert commit_type in ConventionalCommitGenerator.VALID_TYPES

    def test_invalid_type_rejected(self):
        """Test that invalid types are rejected."""
        git_status = GitStatus(
            is_git_repo=True,
            has_changes=True,
            has_staged=True,
            has_unstaged=False,
            changed_files=[],
            staged_files=[],
            unstaged_files=[],
            diff_stat=""
        )
        
        generator = ConventionalCommitGenerator()
        
        with pytest.raises(ValueError):
            generator.generate(git_status, commit_type="invalid_type")
