"""
Tests for GitHub README Generator

Tests README generation logic, context collection, and validation.
"""

import pytest
from pathlib import Path
from unittest.mock import patch, MagicMock
from tempfile import TemporaryDirectory

from devpulse.commands.github_readme import (
    RepoContext,
    ReadmeGenerator,
    RepositoryContextCollector,
    validate_readme_not_exists,
    save_readme,
)


class TestReadmeGenerator:
    """Test README generation logic."""

    def test_generate_minimal_style(self):
        """Test minimal README generation."""
        context = RepoContext(
            name="MyProject",
            description="A test project",
            languages=["Python"],
            has_package_json=False,
            has_pyproject_toml=False,
            has_requirements_txt=True,
            has_license=True,
            license_type="MIT",
            top_level_files=["README.md", "setup.py"],
            remote_url="https://github.com/user/repo",
            is_local=True
        )
        
        generator = ReadmeGenerator()
        content = generator.generate(context, style="minimal")
        
        assert "# MyProject" in content
        assert "A test project" in content
        assert "## Installation" in content
        assert "## License" in content
        # Minimal should not have features
        assert "## Features" not in content

    def test_generate_standard_style(self):
        """Test standard README generation."""
        context = RepoContext(
            name="TestApp",
            description="Testing application",
            languages=["JavaScript", "TypeScript"],
            has_package_json=True,
            has_pyproject_toml=False,
            has_requirements_txt=False,
            has_license=False,
            license_type=None,
            top_level_files=["package.json"],
            remote_url=None,
            is_local=True
        )
        
        generator = ReadmeGenerator()
        content = generator.generate(context, style="standard")
        
        assert "# TestApp" in content
        assert "## Features" in content
        assert "## Installation" in content
        assert "## Usage" in content
        assert "npm install" in content

    def test_generate_detailed_style(self):
        """Test detailed README generation."""
        context = RepoContext(
            name="AdvancedApp",
            description="Advanced application",
            languages=["Python"],
            has_package_json=False,
            has_pyproject_toml=True,
            has_requirements_txt=False,
            has_license=True,
            license_type="Apache 2.0",
            top_level_files=["pyproject.toml"],
            remote_url="https://github.com/user/repo",
            is_local=True
        )
        
        generator = ReadmeGenerator()
        content = generator.generate(context, style="detailed")
        
        assert "## API Documentation" in content
        assert "## Tech Stack" in content
        assert "poetry install" in content

    def test_readme_title_section(self):
        """Test title section generation."""
        context = RepoContext(
            name="TestProject",
            description=None,
            languages=[],
            has_package_json=False,
            has_pyproject_toml=False,
            has_requirements_txt=False,
            has_license=False,
            license_type=None,
            top_level_files=[],
            remote_url=None,
            is_local=True
        )
        
        generator = ReadmeGenerator()
        content = generator.generate(context)
        
        assert content.startswith("# TestProject")

    def test_tech_stack_with_multiple_languages(self):
        """Test tech stack detection with multiple languages."""
        context = RepoContext(
            name="Polyglot",
            description="Multi-language project",
            languages=["Python", "JavaScript", "Go"],
            has_package_json=True,
            has_pyproject_toml=True,
            has_requirements_txt=False,
            has_license=False,
            license_type=None,
            top_level_files=[],
            remote_url=None,
            is_local=True
        )
        
        generator = ReadmeGenerator()
        content = generator.generate(context, style="standard")
        
        assert "Python" in content
        assert "JavaScript" in content
        assert "Go" in content

    def test_license_detection(self):
        """Test license type detection in README."""
        context = RepoContext(
            name="Licensed",
            description="Licensed project",
            languages=[],
            has_package_json=False,
            has_pyproject_toml=False,
            has_requirements_txt=False,
            has_license=True,
            license_type="GPL",
            top_level_files=["LICENSE"],
            remote_url=None,
            is_local=True
        )
        
        generator = ReadmeGenerator()
        content = generator.generate(context)
        
        assert "GPL" in content


class TestRepositoryContextCollector:
    """Test repository context collection."""

    def test_collect_local_invalid_repo(self):
        """Test that collecting from non-git directory raises error."""
        with TemporaryDirectory() as tmpdir:
            with pytest.raises(ValueError, match="Not a git repository"):
                RepositoryContextCollector.collect_local(tmpdir)

    @patch("devpulse.commands.github_readme.subprocess.run")
    def test_collect_local_with_remote(self, mock_run):
        """Test collecting context includes remote URL."""
        mock_run.return_value = MagicMock(
            returncode=0,
            stdout="https://github.com/user/repo.git",
            stderr=""
        )
        
        with TemporaryDirectory() as tmpdir:
            # Create minimal git repo
            git_dir = Path(tmpdir) / ".git"
            git_dir.mkdir()
            
            context = RepositoryContextCollector.collect_local(tmpdir)
            
            assert context.is_local is True
            assert context.name == Path(tmpdir).name

    def test_language_detection_python(self):
        """Test Python language detection."""
        with TemporaryDirectory() as tmpdir:
            tmpdir_path = Path(tmpdir)
            
            # Create git repo
            (tmpdir_path / ".git").mkdir()
            
            # Create Python files
            (tmpdir_path / "main.py").write_text("print('hello')")
            (tmpdir_path / "utils.py").write_text("def util(): pass")
            
            context = RepositoryContextCollector.collect_local(tmpdir)
            
            assert "Python" in context.languages

    def test_config_file_detection(self):
        """Test config file detection."""
        with TemporaryDirectory() as tmpdir:
            tmpdir_path = Path(tmpdir)
            
            # Create git repo
            (tmpdir_path / ".git").mkdir()
            
            # Create config files
            (tmpdir_path / "package.json").write_text("{}")
            (tmpdir_path / "requirements.txt").write_text("")
            
            context = RepositoryContextCollector.collect_local(tmpdir)
            
            assert context.has_package_json is True
            assert context.has_requirements_txt is True

    def test_license_detection_mit(self):
        """Test MIT license detection."""
        with TemporaryDirectory() as tmpdir:
            tmpdir_path = Path(tmpdir)
            
            # Create git repo and license
            (tmpdir_path / ".git").mkdir()
            (tmpdir_path / "LICENSE").write_text("MIT License\n\nCopyright (c) 2025")
            
            context = RepositoryContextCollector.collect_local(tmpdir)
            
            assert context.has_license is True
            assert context.license_type == "MIT"


class TestValidateAndSaveReadme:
    """Test validation and saving of README files."""

    def test_validate_readme_not_exists_new_file(self):
        """Test validation when README doesn't exist."""
        with TemporaryDirectory() as tmpdir:
            result = validate_readme_not_exists(tmpdir)
            assert result is True

    def test_validate_readme_exists_user_denies(self):
        """Test validation when README exists and user denies overwrite."""
        with TemporaryDirectory() as tmpdir:
            tmpdir_path = Path(tmpdir)
            (tmpdir_path / "README.md").write_text("# Existing")
            
            # Mock typer.confirm to return False
            with patch("devpulse.commands.github_readme.typer.confirm", return_value=False):
                result = validate_readme_not_exists(tmpdir)
                assert result is False

    def test_save_readme_creates_file(self):
        """Test saving README creates file correctly."""
        with TemporaryDirectory() as tmpdir:
            content = "# Test\n\nTest content"
            path = save_readme(content, tmpdir)
            
            assert path.exists()
            assert path.read_text() == content

    def test_save_readme_custom_path(self):
        """Test saving README to custom path."""
        with TemporaryDirectory() as tmpdir:
            tmpdir_path = Path(tmpdir)
            custom_path = tmpdir_path / "docs" / "CUSTOM.md"
            
            content = "# Custom"
            path = save_readme(content, tmpdir, str(custom_path))
            
            assert custom_path.exists()
            assert custom_path.read_text() == content

    def test_save_readme_creates_directories(self):
        """Test saving README creates parent directories."""
        with TemporaryDirectory() as tmpdir:
            tmpdir_path = Path(tmpdir)
            custom_path = tmpdir_path / "deep" / "nested" / "README.md"
            
            content = "# Nested"
            path = save_readme(content, tmpdir, str(custom_path))
            
            assert custom_path.parent.exists()
            assert custom_path.exists()
