"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedCodeDeployedFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_synthetics_alpha_1 = require("@aws-cdk/aws-synthetics-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_codedeploy_1 = require("aws-cdk-lib/aws-codedeploy");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_ecs_patterns_1 = require("aws-cdk-lib/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const api_canary_1 = require("../api-canary");
const ecs_deployment_1 = require("../ecs-deployment");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer and deployed by CodeDeploy.
 */
class ApplicationLoadBalancedCodeDeployedFargateService extends aws_ecs_patterns_1.ApplicationLoadBalancedFargateService {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedCodeDeployedFargateService class.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            deploymentController: {
                type: aws_ecs_1.DeploymentControllerType.CODE_DEPLOY,
            },
        });
        if (props.deregistrationDelay) {
            this.targetGroup.setAttribute('deregistration_delay.timeout_seconds', props.deregistrationDelay.toSeconds().toString());
        }
        if (props.healthCheck) {
            this.targetGroup.configureHealthCheck(props.healthCheck);
        }
        this.accessLogBucket = props.accessLogBucket ??
            new aws_s3_1.Bucket(this, 'AccessLogBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                enforceSSL: true,
            });
        this.loadBalancer.logAccessLogs(this.accessLogBucket, props.accessLogPrefix);
        const alarms = [];
        if (props.responseTimeAlarmThreshold) {
            const responseTimeAlarm = new aws_cloudwatch_1.Alarm(this, 'ResponseTimeAlarm', {
                metric: this.loadBalancer.metricTargetResponseTime({
                    period: aws_cdk_lib_1.Duration.minutes(1),
                    statistic: 'p95',
                }),
                evaluationPeriods: 2,
                threshold: props.responseTimeAlarmThreshold.toSeconds(),
                comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            });
            alarms.push(responseTimeAlarm);
        }
        const protocol = props.protocol ?? (props.certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
        const testHostName = props.domainName ? props.domainName : this.loadBalancer.loadBalancerDnsName;
        if (props.apiTestSteps?.length) {
            this.apiCanary = new api_canary_1.ApiCanary(this, 'Canary', {
                baseUrl: `${protocol.toLowerCase()}://${testHostName}`,
                durationAlarmThreshold: props.apiCanaryTimeout,
                schedule: aws_synthetics_alpha_1.Schedule.rate(props.apiCanarySchedule ?? aws_cdk_lib_1.Duration.minutes(5)),
                threadCount: props.apiCanaryThreadCount,
                steps: props.apiTestSteps,
            });
            this.apiCanary.node.addDependency(this.service);
            alarms.push(this.apiCanary.successAlarm);
            if (this.apiCanary.durationAlarm) {
                alarms.push(this.apiCanary.durationAlarm);
            }
        }
        this.healthAlarm = new aws_cloudwatch_1.CompositeAlarm(this, 'HealthAlarm', {
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...alarms),
        });
        let testPort;
        if (props.listenerPort) {
            testPort = props.listenerPort + 1;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            testPort = 8080;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            testPort = 8443;
        }
        else {
            throw new Error('Unable to determine port for test listener');
        }
        let certificates;
        if (props.certificate) {
            certificates = [props.certificate];
        }
        this.testListener = this.loadBalancer.addListener('TestListener', {
            protocol,
            port: testPort,
            open: props.openListener ?? true,
            sslPolicy: props.sslPolicy,
            certificates: certificates,
        });
        this.greenTargetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(this, 'GreenTargetGroup', {
            vpc: this.cluster.vpc,
            port: testPort,
            protocol: props.targetProtocol ?? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            protocolVersion: props.protocolVersion,
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            targetType: aws_elasticloadbalancingv2_1.TargetType.IP,
        });
        this.testListener.addTargetGroups('ECS', {
            targetGroups: [this.greenTargetGroup],
        });
        this.application = new aws_codedeploy_1.EcsApplication(this, 'Application');
        this.deploymentGroup = new aws_codedeploy_1.EcsDeploymentGroup(this, 'DeploymentGroup', {
            application: this.application,
            alarms: [this.healthAlarm],
            service: this.service,
            blueGreenDeploymentConfig: {
                blueTargetGroup: this.targetGroup,
                greenTargetGroup: this.greenTargetGroup,
                listener: this.listener,
                testListener: this.testListener,
                terminationWaitTime: props.terminationWaitTime ?? aws_cdk_lib_1.Duration.minutes(10),
            },
            deploymentConfig: props.deploymentConfig ?? aws_codedeploy_1.EcsDeploymentConfig.ALL_AT_ONCE,
            autoRollback: {
                stoppedDeployment: true,
            },
        });
        this.deployment = new ecs_deployment_1.EcsDeployment({
            deploymentGroup: this.deploymentGroup,
            timeout: props.deploymentTimeout ?? aws_cdk_lib_1.Duration.minutes(60),
            targetService: {
                taskDefinition: this.taskDefinition,
                containerName: this.taskDefinition.defaultContainer.containerName,
                containerPort: this.taskDefinition.defaultContainer.containerPort,
            },
        });
    }
    addServiceAsTarget(service) {
        super.addServiceAsTarget(service);
    }
}
exports.ApplicationLoadBalancedCodeDeployedFargateService = ApplicationLoadBalancedCodeDeployedFargateService;
_a = JSII_RTTI_SYMBOL_1;
ApplicationLoadBalancedCodeDeployedFargateService[_a] = { fqn: "@cdklabs/cdk-ecs-codedeploy.ApplicationLoadBalancedCodeDeployedFargateService", version: "0.0.78" };
//# sourceMappingURL=data:application/json;base64,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