import importlib
import traceback
from collections import defaultdict
from pathlib import Path
from types import ModuleType
from typing import Any, Callable, Dict, List, Optional, Type
from datetime import datetime, timedelta
from concurrent.futures import ThreadPoolExecutor
from copy import copy
from glob import glob
from concurrent.futures import Future

from xmpy.包_事件引擎 import 类_事件,类_事件引擎
from xmpy.包_交易核心.模块_主引擎 import 基础引擎,类_主引擎
from xmpy.包_交易核心.模块_对象 import (
    类_委托请求,
    类_订阅请求,
    类_历史数据请求,
    类_撤单请求,
    类_日志数据,
    类_行情数据,
    类_K线数据,
    类_委托数据,
    类_成交数据,
    类_合约数据
)
from xmpy.包_交易核心.模块_事件类型 import (
    事件类型_行情,
    事件类型_订单,
    事件类型_成交
)
from xmpy.包_交易核心.模块_常数 import (
    类_方向,
    类_委托类型,
    类_周期,
    类_交易所,
    类_开平,
    类_状态
)
from xmpy.包_交易核心.模块_工具 import 加载json文件, 保存json文件, 提取合约代码, 四舍五入到指定值
from xmpy.包_交易核心.模块_基础数据库 import 类_基础数据库, 获取数据库, 数据库时区
from xmpy.包_交易核心.模块_数据服务 import 类_基础数据服务, 获取数据服务

from .模块_基础 import (
    应用名称,
    事件类型_CTA日志,
    事件类型_CTA策略,
    事件类型_CTA停止单,
    类_引擎类型,
    类_停止单,
    类_停止单状态,
    停止单前缀
)
from .模块_模板 import 类_CTA策略模板, 类_目标仓位模板      #类_目标持仓模板
from .包_国际化 import _

# 停止单状态映射表
停止单状态映射: Dict[类_状态, 类_停止单状态] = {
    类_状态.提交中: 类_停止单状态.等待中,
    类_状态.未成交: 类_停止单状态.等待中,
    类_状态.部分成交: 类_停止单状态.已触发,
    类_状态.全部成交: 类_停止单状态.已触发,
    类_状态.已撤销: 类_停止单状态.已撤销,
    类_状态.已拒单: 类_停止单状态.已撤销
}


class 类_CTA引擎(基础引擎):
    """CTA策略引擎"""

    引擎类型: 类_引擎类型 = 类_引擎类型.实盘模式  # 交易引擎类型

    配置文件名: str = "cta_参数设置.json"
    数据文件名: str = "cta_变量数据.json"

    def __init__(self, 主引擎: 类_主引擎, 事件引擎: 类_事件引擎) -> None:
        super().__init__(主引擎, 事件引擎, 应用名称)

        # 策略相关存储结构
        self.策略配置字典: dict = {}  # 策略名称: 配置字典
        self.策略数据字典: dict = {}  # 策略名称: 数据字典
        self.策略类字典: dict = {}  # 类名称: 策略类
        self.运行策略字典: dict = {}  # 策略名称: 策略实例

        # 映射关系
        self.合约策略映射: defaultdict = defaultdict(list)  # 合约标识: 策略列表
        self.订单策略映射: dict = {}  # 订单编号: 策略实例
        self.策略订单映射: defaultdict = defaultdict(set)  # 策略名称: 订单编号集合

        # 停止单管理
        self.停止单计数器: int = 0  # 停止单编号生成
        self.停止单字典: Dict[str, 类_停止单] = {}  # 停止单编号: 停止单实例

        # 初始化线程池
        self.初始化执行器: ThreadPoolExecutor = ThreadPoolExecutor(max_workers=1)

        # 数据过滤
        self.已处理成交编号集合: set = set()  # 用于过滤重复成交

        # 外部服务
        self.数据库: 类_基础数据库 = 获取数据库()
        self.数据服务: 类_基础数据服务 = 获取数据服务()

    def 初始化引擎(self) -> None:
        """执行引擎初始化流程"""
        self.初始化数据服务()
        self.加载策略类()
        self.加载策略配置()
        self.加载策略数据()
        self.注册事件()
        self.记录日志(_("CTA策略引擎初始化成功"))

    def 关闭引擎(self) -> None:
        """停止所有策略并关闭引擎"""
        self.停止所有策略()

    def 注册事件(self) -> None:
        """注册事件监听"""
        self.事件引擎.注册类型处理器(事件类型_行情, self.处理行情事件)
        self.事件引擎.注册类型处理器(事件类型_订单, self.处理订单事件)
        self.事件引擎.注册类型处理器(事件类型_成交, self.处理成交事件)

    def 初始化数据服务(self) -> None:
        """初始化数据服务客户端"""
        初始化结果: bool = self.数据服务.初始化(self.记录日志)
        if 初始化结果:
            self.记录日志(_("数据服务初始化成功"))

    def 查询K线数据(self, 合约代码: str, 交易所: 类_交易所, 周期: 类_周期, 开始时间: datetime, 结束时间: datetime) -> List[
        类_K线数据]:
        """从数据服务查询K线数据"""
        查询请求 = 类_历史数据请求(
            代码=合约代码,
            交易所=交易所,
            周期=周期,
            开始时间=开始时间,
            结束时间=结束时间
        )
        K线列表: List[类_K线数据] = self.数据服务.查询K线历史(查询请求, self.记录日志)
        return K线列表

    def 处理行情事件(self, 事件: 类_事件) -> None:
        """处理行情推送事件"""
        行情实例: 类_行情数据 = 事件.数据

        # 获取关联策略列表
        策略列表: list = self.合约策略映射[行情实例.唯一标识]
        if not 策略列表:
            return

        # 检查停止单触发条件
        self.检查停止单(行情实例)

        # 遍历策略执行回调
        for 策略 in 策略列表:
            if 策略.已初始化:
                self.调用策略方法(策略, 策略.行情回调, 行情实例)

    def 处理订单事件(self, 事件: 类_事件) -> None:
        """处理订单状态更新事件"""
        订单实例: 类_委托数据 = 事件.数据

        # 查找关联策略
        策略实例: Optional[type] = self.订单策略映射.get(订单实例.订单唯一标识, None)
        if not 策略实例:
            return

        # 移除已完成订单
        订单集合: set = self.策略订单映射[策略实例.策略名称]
        if 订单实例.订单唯一标识 in 订单集合 and not 订单实例.是否活跃():
            订单集合.remove(订单实例.订单唯一标识)

        # 处理停止单回调
        if 订单实例.类型 == 类_委托类型.止损单:
            停止单实例 = 类_停止单(
                合约标识=订单实例.唯一标识,
                方向=订单实例.方向,
                开平=订单实例.开平,
                价格=订单实例.价格,
                数量=订单实例.数量,
                停止单编号=订单实例.委托编号,
                策略名称=策略实例.策略名称,
                时间戳=订单实例.时间戳,
                状态=停止单状态映射[订单实例.状态],
                订单编号列表=[订单实例.委托编号],
            )
            self.调用策略方法(策略实例, 策略实例.停止单更新, 停止单实例)

        # 执行策略订单回调
        self.调用策略方法(策略实例, 策略实例.委托回调, 订单实例)

    def 处理成交事件(self, 事件: 类_事件) -> None:
        """处理成交回报事件"""
        成交实例: 类_成交数据 = 事件.数据

        # 过滤重复成交
        if 成交实例.成交唯一标识 in self.已处理成交编号集合:
            return
        self.已处理成交编号集合.add(成交实例.成交唯一标识)

        # 查找关联策略
        策略实例: Optional[type] = self.订单策略映射.get(成交实例.订单唯一标识, None)
        if not 策略实例:
            return

        # 更新策略持仓
        if 成交实例.方向 == 类_方向.做多 and 成交实例.开平 == 类_开平.开仓:
            策略实例.多头仓位 += 成交实例.数量
        elif 成交实例.方向 == 类_方向.做空 and 成交实例.开平 == 类_开平.开仓:
            策略实例.空头仓位 += 成交实例.数量
        # 平仓
        elif 成交实例.方向 == 类_方向.做空 and 成交实例.开平.value in ["平", "平今", "平昨"]:
            策略实例.多头仓位 -= 成交实例.数量
        elif 成交实例.方向 == 类_方向.做多 and 成交实例.开平.value in ["平", "平今", "平昨"]:
            策略实例.空头仓位 -= 成交实例.数量

        # 更新策略持仓
        # if 成交实例.方向 == 类_方向.做多:
        #     策略实例.当前仓位 += 成交实例.数量
        # else:
        #     策略实例.当前仓位 -= 成交实例.数量

        # 执行策略成交回调
        self.调用策略方法(策略实例, 策略实例.成交回调, 成交实例)

        # 同步策略数据
        self.同步策略数据(策略实例)

        # 更新策略状态事件
        self.推送策略事件(策略实例)

    def 检查停止单(self, 行情实例: 类_行情数据) -> None:
        """检查停止单触发条件"""
        for 停止单实例 in list(self.停止单字典.values()):
            if 停止单实例.合约标识 != 行情实例.唯一标识:
                continue

            # 触发条件判断
            多单触发 = 停止单实例.方向 == 类_方向.做多 and 行情实例.最新价 >= 停止单实例.价格
            空单触发 = 停止单实例.方向 == 类_方向.做空 and 行情实例.最新价 <= 停止单实例.价格

            if 多单触发 or 空单触发:
                策略实例: 类_CTA策略模板 = self.运行策略字典[停止单实例.策略名称]

                # 确定委托价格
                if 停止单实例.方向 == 类_方向.做多:
                    委托价格 = 行情实例.涨停价 if 行情实例.涨停价 else 行情实例.卖五价
                else:
                    委托价格 = 行情实例.跌停价 if 行情实例.跌停价 else 行情实例.买五价

                # 获取合约信息
                合约信息: Optional[类_合约数据] = self.主引擎.获取合约详情(停止单实例.合约标识)

                # 发送限价单
                订单编号列表: list = self.发送限价单(
                    策略实例,
                    合约信息,
                    停止单实例.方向,
                    停止单实例.开平,
                    委托价格,
                    停止单实例.数量,
                    停止单实例.锁定模式,
                    停止单实例.净仓模式
                )

                if 订单编号列表:
                    # 清理停止单记录
                    self.停止单字典.pop(停止单实例.停止单编号)
                    策略订单集合: set = self.策略订单映射[策略实例.策略名称]
                    if 停止单实例.停止单编号 in 策略订单集合:
                        策略订单集合.remove(停止单实例.停止单编号)

                    # 更新停止单状态
                    停止单实例.状态 = 类_停止单状态.已触发
                    停止单实例.关联订单列表 = 订单编号列表

                    # 推送停止单事件
                    self.调用策略方法(策略实例, 策略实例.停止单回调, 停止单实例)
                    self.推送停止单事件(停止单实例)

    def 发送服务器委托(
            self,
            策略实例: 类_CTA策略模板,
            合约信息: 类_合约数据,
            方向: 类_方向,
            开平: 类_开平,
            价格: float,
            数量: float,
            类型: 类_委托类型,
            锁定: bool,
            净仓: bool
    ) -> list:
        """向交易服务器发送原始订单"""
        # 创建原始订单请求
        原始请求 = 类_委托请求(
            代码=合约信息.代码,
            交易所=合约信息.交易所,
            方向=方向,
            开平=开平,
            类型=类型,
            价格=价格,
            数量=数量,
            参考号=f"{应用名称}_{策略实例.策略名称}"
        )

        # 使用仓位转换器处理请求
        转换后请求列表: List[类_委托请求] = self.主引擎.转换委托请求(
            原始请求,
            合约信息.网关名称,
            锁定,
            净仓
        )

        订单编号列表 = []
        for 请求 in 转换后请求列表:
            订单编号: str = self.主引擎.发送委托(请求, 合约信息.网关名称)

            if 订单编号:
                订单编号列表.append(订单编号)
                self.主引擎.更新委托请求(请求, 订单编号, 合约信息.网关名称)

                # 维护订单映射关系
                self.订单策略映射[订单编号] = 策略实例
                self.策略订单映射[策略实例.策略名称].add(订单编号)

        return 订单编号列表

    def 发送限价单(
            self,
            策略实例: 类_CTA策略模板,
            合约信息: 类_合约数据,
            方向: 类_方向,
            开平: 类_开平,
            价格: float,
            数量: float,
            锁定: bool,
            净仓: bool
    ) -> list:
        """发送限价单到服务器"""
        return self.发送服务器委托(
            策略实例,
            合约信息,
            方向,
            开平,
            价格,
            数量,
            类_委托类型.限价单,
            锁定,
            净仓
        )

    def 发送服务器停止单(
            self,
            策略实例: 类_CTA策略模板,
            合约信息: 类_合约数据,
            方向: 类_方向,
            开平: 类_开平,
            价格: float,
            数量: float,
            锁定: bool,
            净仓: bool
    ) -> list:
        """发送服务器端停止单"""
        return self.发送服务器委托(
            策略实例,
            合约信息,
            方向,
            开平,
            价格,
            数量,
            类_委托类型.止损单,
            锁定,
            净仓
        )

    def 发送本地停止单(
            self,
            策略实例: 类_CTA策略模板,
            方向: 类_方向,
            开平: 类_开平,
            价格: float,
            数量: float,
            锁定: bool,
            净仓: bool
    ) -> list:
        """创建本地停止单"""
        self.停止单计数器 += 1
        停止单编号: str = f"{停止单前缀}.{self.停止单计数器}"

        停止单实例 = 类_停止单(
            合约标识=策略实例.合约标识,
            方向=方向,
            开平=开平,
            价格=价格,
            数量=数量,
            停止单编号=停止单编号,
            策略名称=策略实例.策略名称,
            时间戳=datetime.now(数据库时区),
            锁定模式=锁定,
            净仓模式=净仓
        )

        # 登记停止单
        self.停止单字典[停止单编号] = 停止单实例
        self.策略订单映射[策略实例.策略名称].add(停止单编号)

        # 推送停止单事件
        self.调用策略方法(策略实例, 策略实例.停止单回调, 停止单实例)
        self.推送停止单事件(停止单实例)

        return [停止单编号]

    def 撤消服务器订单(self, 策略实例: 类_CTA策略模板, 订单编号: str) -> None:
        """撤销服务器端订单"""
        订单实例: Optional[类_委托数据] = self.主引擎.获取订单详情(订单编号)
        if not 订单实例:
            self.记录日志(_("撤单失败，找不到委托{}").format(订单编号), 策略实例)
            return

        撤单请求: 类_撤单请求 = 订单实例.创建撤单请求()
        self.主引擎.撤消订单(撤单请求, 订单实例.网关名称)

    def 撤消本地停止单(self, 策略实例: 类_CTA策略模板, 停止单编号: str) -> None:
        """撤销本地停止单"""
        停止单实例: Optional[类_停止单] = self.停止单字典.get(停止单编号, None)
        if not 停止单实例:
            return

        策略实例:类_CTA策略模板 = self.运行策略字典[停止单实例.策略名称]

        # 清理映射关系
        self.停止单字典.pop(停止单编号)
        策略订单集合: set = self.策略订单映射[策略实例.策略名称]
        if 停止单编号 in 策略订单集合:
            策略订单集合.remove(停止单编号)

        # 更新停止单状态
        停止单实例.状态 = 类_停止单状态.已撤销
        self.调用策略方法(策略实例, 策略实例.停止单回调, 停止单实例)
        self.推送停止单事件(停止单实例)

    def 发送委托(
            self,
            策略实例: 类_CTA策略模板,
            方向: 类_方向,
            开平: 类_开平,
            价格: float,
            数量: float,
            停止单标志: bool,
            锁定: bool,
            净仓: bool
    ) -> list:
        """统一委托方法"""
        合约实例: Optional[类_合约数据] = self.主引擎.获取合约详情(策略实例.合约标识)
        if not 类_委托类型:
            self.记录日志(_("委托失败，找不到合约：{}").format(策略实例.合约标识), 策略实例)
            return []

        # 价格和数量取整
        价格 = 四舍五入到指定值(价格, 合约实例.最小价位)
        数量 = 四舍五入到指定值(数量, 合约实例.最小数量)

        if 停止单标志:
            if 合约实例.支持止损单:
                return self.发送服务器停止单(
                    策略实例, 合约实例, 方向, 开平, 价格, 数量, 锁定, 净仓
                )
            else:
                return self.发送本地停止单(
                    策略实例, 方向, 开平, 价格, 数量, 锁定, 净仓
                )
        else:
            return self.发送限价单(
                策略实例, 合约实例, 方向, 开平, 价格, 数量, 锁定, 净仓
            )

    def 撤消订单(self, 策略实例: 类_CTA策略模板, 订单编号: str) -> None:
        """统一撤单方法"""
        if 订单编号.startswith(停止单前缀):
            self.撤消本地停止单(策略实例, 订单编号)
        else:
            self.撤消服务器订单(策略实例, 订单编号)

    def 撤消全部订单(self, 策略实例: 类_CTA策略模板) -> None:
        """撤销策略所有订单"""
        订单集合: set = self.策略订单映射[策略实例.策略名称]

        if not 订单集合:
            return

        for 订单编号 in copy(订单集合):
            self.撤消订单(策略实例, 订单编号)

    def 获取引擎类型(self) -> 引擎类型:
        """获取引擎类型"""
        return self.引擎类型

    def 获取最小价位(self, 策略实例: 类_CTA策略模板) -> float:
        """获取合约最小价格变动单位"""
        合约实例: Optional[类_合约数据] = self.主引擎.获取合约详情(策略实例.合约标识)
        return 合约实例.最小价位 if 合约实例 else None

    def 获取合约乘数(self, 策略实例: 类_CTA策略模板) -> int:
        """获取合约乘数"""
        合约实例: Optional[类_合约数据] = self.主引擎.获取合约详情(策略实例.合约标识)
        return 合约实例.合约乘数 if 合约实例 else None

    def 加载K线数据(
            self,
            合约标识: str,
            天数: int,
            周期: 类_周期,
            回调函数: Callable[[类_K线数据], None],
            使用数据库: bool
    ) -> List[类_K线数据]:
        """加载历史K线数据"""
        代码, 交易所 = 提取合约代码(合约标识)
        结束时间: datetime = datetime.now(数据库时区)
        开始时间: datetime = 结束时间 - timedelta(days=天数)
        K线列表: List[类_K线数据] = []

        if not 使用数据库:
            # 从网关或数据服务查询
            合约信息: Optional[类_合约数据] = self.主引擎.获取合约详情(合约标识)
            if 合约信息 and 合约信息.支持历史数据:
                查询请求 = 类_历史数据请求(
                    代码=代码,
                    交易所=交易所.value,
                    周期=周期,
                    开始时间=开始时间,
                    结束时间=结束时间
                )
                K线列表 = self.主引擎.查询历史(查询请求, 合约信息.网关名称)
            else:
                K线列表 = self.查询K线数据(代码, 交易所, 周期, 开始时间, 结束时间)

        if not K线列表:
            K线列表 = self.数据库.加载K线数据(
                代码=代码,
                交易所=交易所,
                周期=周期,
                开始时间=开始时间,
                结束时间=结束时间,
            )

        return K线列表

    def 加载Tick数据(
            self,
            合约标识: str,
            天数: int,
            回调函数: Callable[[类_行情数据], None]
    ) -> List[类_行情数据]:
        """加载历史Tick数据"""
        代码, 交易所 = 提取合约代码(合约标识)
        结束时间: datetime = datetime.now(数据库时区)
        开始时间: datetime = 结束时间 - timedelta(days=天数)

        Tick列表: List[类_行情数据] = self.数据库.加载Tick数据(
            代码=代码,
            交易所=交易所.value,
            开始时间=开始时间,
            结束时间=结束时间,
        )

        return Tick列表

    def 调用策略方法(
            self, 策略实例: 类_CTA策略模板, 方法: Callable, 参数: Any = None
    ) -> None:
        """安全执行策略方法并捕获异常"""
        try:
            if 参数:
                方法(参数)
            else:
                方法()
        except Exception:
            策略实例.运行中 = False
            策略实例.已初始化 = False

            错误信息: str = _("触发异常已停止\n{}").format(traceback.format_exc())
            self.记录日志(错误信息, 策略实例)

    def 添加策略(self, 类名: str, 策略名称: str, 合约标识: str, 配置字典: dict) -> None:
        """添加新策略"""
        if 策略名称 in self.运行策略字典:
            self.记录日志(f"策略名称重复: {策略名称}")
            return

        # 验证策略类
        策略类: Optional[Type[类_CTA策略模板]] = self.策略类字典.get(类名, None)
        if not 策略类:
            self.记录日志(f"创建策略失败，找不到策略类{类名}")
            return

        # 验证合约标识格式
        if "." not in 合约标识:
            self.记录日志("创建策略失败，本地代码缺失交易所后缀")
            return

        _, 交易所后缀 = 合约标识.split(".")
        if 交易所后缀 not in 类_交易所.__members__:
            self.记录日志("创建策略失败，本地代码的交易所后缀不正确")
            return

        # 创建策略实例
        策略实例 = 策略类(self, 策略名称, 合约标识, 配置字典)
        self.运行策略字典[策略名称] = 策略实例

        strategies: list = self.合约策略映射[合约标识]
        # 维护合约映射
        strategies.append(策略实例)

        # 更新配置文件
        self.更新策略配置(策略名称, 配置字典)
        self.推送策略事件(策略实例)

    def 初始化策略(self, 策略名称: str) -> Future:
        """异步初始化策略"""
        return self.初始化执行器.submit(self._执行策略初始化, 策略名称)

    def _执行策略初始化(self, 策略名称: str) -> None:
        """策略初始化核心逻辑"""
        策略实例:类_CTA策略模板 = self.运行策略字典[策略名称]

        if 策略实例.已初始化:
            self.记录日志(_("{}已经完成初始化，禁止重复操作").format(策略名称))
            return

        self.记录日志(_("{}:开始执行初始化").format(策略名称))

        # 执行策略初始化方法
        self.调用策略方法(策略实例, 策略实例.初始化回调)

        # 恢复策略变量
        策略数据: Optional[dict] = self.策略数据字典.get(策略名称, None)
        if 策略数据:
            for 变量名 in 策略实例.变量列表:
                值 = 策略数据.get(变量名, None)
                if 值 is not None:
                    setattr(策略实例, 变量名, 值)

        代码, 交易所 = 提取合约代码(策略实例.合约标识)
        策略实例.合约标识 = f"{代码}.{交易所.value}"
        # 订阅行情
        合约信息: Optional[类_合约数据] = self.主引擎.获取合约详情(策略实例.合约标识)
        if 合约信息:
            订阅请求 = 类_订阅请求(
                代码=合约信息.代码,
                交易所=合约信息.交易所
            )
            self.主引擎.订阅行情(订阅请求, 合约信息.网关名称)
        else:
            self.记录日志(_("行情订阅失败，找不到合约{}").format(策略实例.合约标识), 策略实例)

        # 更新初始化状态
        策略实例.已初始化 = True
        self.推送策略事件(策略实例)
        self.记录日志(_("{}初始化完成").format(策略名称))

    def 启动策略(self, 策略名称: str) -> None:
        """启动策略交易"""
        策略实例 = self.运行策略字典[策略名称]
        if not 策略实例.已初始化:
            self.记录日志(_("策略:{},启动失败，请先初始化").format(策略实例.策略名称))
            return

        if 策略实例.运行中:
            self.记录日志(_("{}已经启动，请勿重复操作").format(策略名称))
            return

        self.调用策略方法(策略实例, 策略实例.启动回调)
        策略实例.运行中 = True
        self.推送策略事件(策略实例)

    def 停止策略(self, 策略名称: str) -> None:
        """停止策略交易"""
        策略实例 = self.运行策略字典[策略名称]
        if not 策略实例.运行中:
            return

        # 执行策略停止方法
        self.调用策略方法(策略实例, 策略实例.停止)

        # 更新交易状态
        策略实例.运行中 = False

        # 撤消所有订单
        self.撤消全部订单(策略实例)

        # 同步数据
        self.同步策略数据(策略实例)
        self.推送策略事件(策略实例)

    def 编辑策略(self, 策略名称: str, 配置字典: dict) -> None:
        """修改策略参数"""
        策略实例 = self.运行策略字典[策略名称]
        策略实例.更新配置(配置字典)
        self.更新策略配置(策略名称, 配置字典)
        self.推送策略事件(策略实例)

    def 移除策略(self, 策略名称: str) -> bool:
        """移除策略实例"""
        策略实例 = self.运行策略字典[策略名称]
        if 策略实例.运行中:
            self.记录日志(_("策略{}移除失败，请先停止").format(策略实例.策略名称))
            return

        # 清理配置
        self.移除策略配置(策略名称)

        # 清理合约映射
        strategies: list = self.合约策略映射[策略实例.合约标识]
        strategies.remove(策略实例)

        # 清理订单映射
        if 策略名称 in self.策略订单映射:
            订单集合 = self.策略订单映射.pop(策略名称)
            for 订单编号 in 订单集合:
                if 订单编号 in self.订单策略映射:
                    self.订单策略映射.pop(订单编号)

        # 移除策略实例
        self.运行策略字典.pop(策略名称)
        self.记录日志(_("策略{}移除成功").format(策略实例.策略名称))
        return True

    def 加载策略类(self) -> None:
        """加载策略类文件"""
        路径1 = Path(__file__).parent.joinpath("策略文件夹")
        self.记录日志(f'路径1：{路径1}')
        self.从目录加载策略类(路径1, "xmpy_ctastrategy.策略文件夹")

        路径2 = Path.cwd().joinpath("策略文件夹")
        self.记录日志(f'路径2：{路径2}')
        self.从目录加载策略类(路径2, "策略文件夹")

    def 从目录加载策略类(self, 路径: Path, 模块前缀: str = "") -> None:
        """从指定目录加载策略类"""
        for 后缀 in ["py", "pyd", "so"]:
            匹配模式 = str(路径.joinpath(f"*.{后缀}"))
            for 文件路径 in glob(匹配模式):
                文件名 = Path(文件路径).stem
                模块名 = f"{模块前缀}.{文件名}"
                self.从模块加载策略类(模块名)

    def 从模块加载策略类(self, 模块名: str) -> None:
        """从Python模块加载策略类"""
        try:
            模块: ModuleType = importlib.import_module(模块名)
            importlib.reload(模块)  # 确保修改后重新加载

            for 属性名 in dir(模块):
                属性值 = getattr(模块, 属性名)
                if (
                        isinstance(属性值, type)
                        and issubclass(属性值, 类_CTA策略模板)
                        and 属性值 not in {类_CTA策略模板, 类_目标仓位模板}
                ):
                    self.策略类字典[属性值.__name__] = 属性值
        except Exception:
            错误信息 = _("策略文件{}加载失败，触发异常：\n{}").format(模块名, traceback.format_exc())
            self.记录日志(错误信息)

    def 加载策略数据(self) -> None:
        """从文件加载策略数据"""
        self.策略数据字典 = 加载json文件(self.数据文件名)

    def 同步策略数据(self, 策略实例: 类_CTA策略模板) -> None:
        """将策略变量保存到文件"""
        数据字典 = 策略实例.获取变量()
        数据字典.pop("已初始化")  # 不保存状态信息
        数据字典.pop("运行中")

        self.策略数据字典[策略实例.策略名称] = 数据字典
        保存json文件(self.数据文件名, self.策略数据字典)

    def 获取所有策略类名称(self) -> list:
        """获取已加载的策略类名称列表"""
        return list(self.策略类字典.keys())

    def 获取策略类参数(self, 类名: str) -> dict:
        """获取策略类的默认参数"""
        策略类 = self.策略类字典[类名]
        return {参数名: getattr(策略类, 参数名) for 参数名 in 策略类.参数列表}

    def 获取策略参数(self, 策略名称: str) -> dict:
        """获取运行中策略的参数"""
        策略实例 = self.运行策略字典[策略名称]
        return 策略实例.获取参数()

    def 初始化所有策略(self) -> Dict[str, Future]:
        """批量初始化所有策略"""
        futures:Dict[str,Future] = {}
        for 策略名称 in self.运行策略字典.keys():
            futures[策略名称] = self.初始化策略(策略名称)
        return futures

    def 启动所有策略(self) -> None:
        """批量启动所有策略"""
        for 策略名称 in self.运行策略字典.keys():
            self.启动策略(策略名称)

    def 停止所有策略(self) -> None:
        """批量停止所有策略"""
        for 策略名称 in self.运行策略字典.keys():
            self.停止策略(策略名称)

    def 加载策略配置(self) -> None:
        """加载策略配置文件"""
        self.策略配置字典 = 加载json文件(self.配置文件名)

        for 策略名称, 配置 in self.策略配置字典.items():
            self.添加策略(
                配置["class_name"],
                策略名称,
                配置["vt_symbol"],
                配置["setting"]
            )

    def 更新策略配置(self, 策略名称: str, 配置字典: dict) -> None:
        """更新策略配置到文件"""
        策略实例 = self.运行策略字典[策略名称]
        self.策略配置字典[策略名称] = {
            "class_name": 策略实例.__class__.__name__,
            "vt_symbol": 策略实例.合约标识,
            "setting": 配置字典,
        }
        保存json文件(self.配置文件名, self.策略配置字典)

    def 移除策略配置(self, 策略名称: str) -> None:
        """从配置文件中移除策略"""
        if 策略名称 in self.策略配置字典:
            del self.策略配置字典[策略名称]
            保存json文件(self.配置文件名, self.策略配置字典)

        if 策略名称 in self.策略数据字典:
            del self.策略数据字典[策略名称]
            保存json文件(self.数据文件名, self.策略数据字典)

    def 推送停止单事件(self, 停止单实例: 类_停止单) -> None:
        """推送停止单状态更新事件"""
        事件 = 类_事件(事件类型_CTA停止单, 停止单实例)
        self.事件引擎.放入事件(事件)

    def 推送策略事件(self, 策略实例: 类_CTA策略模板) -> None:
        """推送策略状态更新事件"""
        数据字典 = 策略实例.获取数据()
        self.记录日志(f'策略运行情况：{数据字典}')
        事件 = 类_事件(事件类型_CTA策略, 数据字典)
        self.事件引擎.放入事件(事件)

    def 记录日志(self, 内容: str, 策略实例: 类_CTA策略模板 = None) -> None:
        """记录引擎日志"""
        if 策略实例:
            内容 = f"[{策略实例.策略名称}]  {内容}"

        日志实例 = 类_日志数据(消息内容=内容, 网关名称=应用名称)
        事件 = 类_事件(类型=事件类型_CTA日志, 数据=日志实例)
        self.事件引擎.放入事件(事件)

    def 发送邮件(self, 内容: str, 策略实例: 类_CTA策略模板 = None) -> None:
        """发送邮件通知"""
        主题 = 策略实例.策略名称 if 策略实例 else _("CTA策略引擎")
        self.主引擎.发送邮件(主题, 内容)