"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerConfigSecret = exports.TlsSecret = exports.ServiceAccountTokenSecret = exports.SshAuthSecret = exports.BasicAuthSecret = exports.Secret = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const base = require("./base");
const container_1 = require("./container");
const k8s = require("./imports/k8s");
class ImportedSecret extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this.resourceType = 'secrets';
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubeSecret.GVK.apiVersion;
    }
    get apiGroup() {
        return '';
    }
    get kind() {
        return k8s.KubeSecret.GVK.kind;
    }
    get resourceName() {
        return this.name;
    }
    envValue(key, options) {
        return container_1.EnvValue.fromSecretValue({ secret: this, key }, options);
    }
}
/**
 * Kubernetes Secrets let you store and manage sensitive information, such as
 * passwords, OAuth tokens, and ssh keys. Storing confidential information in a
 * Secret is safer and more flexible than putting it verbatim in a Pod
 * definition or in a container image.
 *
 * @see https://kubernetes.io/docs/concepts/configuration/secret
 */
class Secret extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'secrets';
        this.stringData = props.stringData ?? {};
        this.immutable = props.immutable ?? false;
        this.apiObject = new k8s.KubeSecret(this, 'Resource', {
            metadata: props.metadata,
            type: props.type,
            stringData: this.stringData,
            immutable: this.immutable,
        });
    }
    /**
     * Imports a secret from the cluster as a reference.
     */
    static fromSecretName(scope, id, name) {
        return new ImportedSecret(scope, id, name);
    }
    /**
     * Adds a string data field to the secert.
     * @param key Key
     * @param value Value
     */
    addStringData(key, value) {
        this.stringData[key] = value;
    }
    /**
     * Gets a string data by key or undefined
     * @param key Key
     */
    getStringData(key) {
        return this.stringData[key];
    }
    envValue(key, options) {
        return container_1.EnvValue.fromSecretValue({ secret: this, key }, options);
    }
}
exports.Secret = Secret;
_a = JSII_RTTI_SYMBOL_1;
Secret[_a] = { fqn: "cdk8s-plus-23.Secret", version: "2.4.30" };
/**
 * Create a secret for basic authentication.
 *
 * @see https://kubernetes.io/docs/concepts/configuration/secret/#basic-authentication-secret
 */
class BasicAuthSecret extends Secret {
    constructor(scope, id, props) {
        super(scope, id, {
            type: 'kubernetes.io/basic-auth',
            stringData: {
                username: props.username,
                password: props.password,
            },
            immutable: props.immutable,
            metadata: props.metadata,
        });
    }
}
exports.BasicAuthSecret = BasicAuthSecret;
_b = JSII_RTTI_SYMBOL_1;
BasicAuthSecret[_b] = { fqn: "cdk8s-plus-23.BasicAuthSecret", version: "2.4.30" };
/**
 * Create a secret for ssh authentication.
 *
 * @see https://kubernetes.io/docs/concepts/configuration/secret/#ssh-authentication-secrets
 */
class SshAuthSecret extends Secret {
    constructor(scope, id, props) {
        super(scope, id, {
            type: 'kubernetes.io/ssh-auth',
            stringData: {
                'ssh-privatekey': props.sshPrivateKey,
            },
            immutable: props.immutable,
            metadata: props.metadata,
        });
    }
}
exports.SshAuthSecret = SshAuthSecret;
_c = JSII_RTTI_SYMBOL_1;
SshAuthSecret[_c] = { fqn: "cdk8s-plus-23.SshAuthSecret", version: "2.4.30" };
/**
 * Create a secret for a service account token.
 *
 * @see https://kubernetes.io/docs/concepts/configuration/secret/#service-account-token-secrets
 */
class ServiceAccountTokenSecret extends Secret {
    constructor(scope, id, props) {
        super(scope, id, {
            type: 'kubernetes.io/service-account-token',
            metadata: props.metadata,
            immutable: props.immutable,
        });
        this.metadata.addAnnotation('kubernetes.io/service-account.name', props.serviceAccount.name);
    }
}
exports.ServiceAccountTokenSecret = ServiceAccountTokenSecret;
_d = JSII_RTTI_SYMBOL_1;
ServiceAccountTokenSecret[_d] = { fqn: "cdk8s-plus-23.ServiceAccountTokenSecret", version: "2.4.30" };
/**
 * Create a secret for storing a TLS certificate and its associated key.
 *
 * @see https://kubernetes.io/docs/concepts/configuration/secret/#tls-secrets
 */
class TlsSecret extends Secret {
    constructor(scope, id, props) {
        super(scope, id, {
            type: 'kubernetes.io/tls',
            stringData: {
                'tls.crt': props.tlsCert,
                'tls.key': props.tlsKey,
            },
            immutable: props.immutable,
            metadata: props.metadata,
        });
    }
}
exports.TlsSecret = TlsSecret;
_e = JSII_RTTI_SYMBOL_1;
TlsSecret[_e] = { fqn: "cdk8s-plus-23.TlsSecret", version: "2.4.30" };
/**
 * Create a secret for storing credentials for accessing a container image
 * registry.
 *
 * @see https://kubernetes.io/docs/concepts/configuration/secret/#docker-config-secrets
 */
class DockerConfigSecret extends Secret {
    constructor(scope, id, props) {
        super(scope, id, {
            type: 'kubernetes.io/dockerconfigjson',
            stringData: {
                '.dockerconfigjson': JSON.stringify(props.data),
            },
            immutable: props.immutable,
            metadata: props.metadata,
        });
    }
}
exports.DockerConfigSecret = DockerConfigSecret;
_f = JSII_RTTI_SYMBOL_1;
DockerConfigSecret[_f] = { fqn: "cdk8s-plus-23.DockerConfigSecret", version: "2.4.30" };
//# sourceMappingURL=data:application/json;base64,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