"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f, _g;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Bundling image derived from runtime bundling image (AWS SAM docker image)
        const image = cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
            buildArgs: {
                ...(_b = options.buildArgs) !== null && _b !== void 0 ? _b : {},
                IMAGE: options.runtime.bundlingDockerImage.image,
                PARCEL_VERSION: (_c = options.parcelVersion) !== null && _c !== void 0 ? _c : '2.0.0-beta.1',
            },
        });
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_d = options.externalModules) !== null && _d !== void 0 ? _d : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_e = options.nodeModules) !== null && _e !== void 0 ? _e : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            targets: {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_f = options.sourceMaps) !== null && _f !== void 0 ? _f : false,
                    minify: (_g = options.minify) !== null && _g !== void 0 ? _g : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(projectRoot, path.resolve(options.entry)));
        const distFile = path.basename(options.entry).replace(/\.ts$/, '.js');
        const parcelCommand = chain([
            [
                '$(node -p "require.resolve(\'parcel\')")',
                'build', containerEntryPath.replace(/\\/g, '/'),
                '--target', 'cdk-lambda',
                '--dist-dir', cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
                '--no-autoinstall',
                '--no-scope-hoist',
                ...options.cacheDir
                    ? ['--cache-dir', '/parcel-cache']
                    : [],
            ].join(' '),
            // Always rename dist file to index.js because Lambda doesn't support filenames
            // with multiple dots and we can end up with multiple dots when using automatic
            // entry lookup
            `mv ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${distFile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/index.js`,
        ]);
        let installer = Installer.NPM;
        let lockfile;
        let depsCommand = '';
        if (dependencies) {
            // Create a dummy package.json for dependencies that we need to install
            fs.writeFileSync(path.join(projectRoot, '.package.json'), JSON.stringify({ dependencies }));
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, LockFile.YARN))) {
                installer = Installer.YARN;
                lockfile = LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, LockFile.NPM))) {
                lockfile = LockFile.NPM;
            }
            // Move dummy package.json and lock file then install
            depsCommand = chain([
                `mv ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/.package.json ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/package.json`,
                lockfile ? `cp ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/${lockfile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${lockfile}` : '',
                `cd ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR} && ${installer} install`,
            ]);
        }
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                image,
                command: ['bash', '-c', chain([parcelCommand, depsCommand])],
                environment: options.parcelEnvironment,
                volumes: options.cacheDir
                    ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                    : [],
                workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
            },
        });
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile || (LockFile = {}));
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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