"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxDockerManifestProject = exports.LinuxDockerBuildProject = exports.BaseDockerProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
/**
 * The base for Linux-based Docker build projects.
 */
class BaseDockerProject extends constructs_1.Construct {
    /**
     * Creates a new BaseDockerProject.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { repository, removalPolicy, logRetention } = props;
        if (props.buildEnvironment.buildImage) {
            if (/\b(windows|macos)\b/.test(props.buildEnvironment.buildImage.imageId)) {
                throw new aws_cdk_lib_1.ValidationError("You must use a Linux-based build image", this);
            }
        }
        this.buildSpec = props.buildSpec;
        this.logGroup = new aws_logs_1.LogGroup(this, "Logs", { retention: logRetention });
        const repoEnv = repository.env;
        const projectVariables = {
            AWS_DEFAULT_REGION: { value: repoEnv.region },
            AWS_ECR_HOSTNAME: {
                value: `${repoEnv.account}.dkr.ecr.${repoEnv.region}.amazonaws.com`,
            },
            AWS_ECR_URL: { value: repository.repositoryUri },
        };
        this.project = new aws_codebuild_1.PipelineProject(this, "Project", {
            description: props.description,
            environment: props.buildEnvironment,
            environmentVariables: projectVariables,
            buildSpec: this.buildSpec,
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            securityGroups: props.securityGroups,
            logging: { cloudWatch: { logGroup: this.logGroup } },
        });
        repository.grantPullPush(this.project);
        if (removalPolicy) {
            this.applyRemovalPolicy(removalPolicy);
        }
    }
    get stack() {
        return this.project.stack;
    }
    get env() {
        return this.project.env;
    }
    applyRemovalPolicy(policy) {
        this.project.applyRemovalPolicy(policy);
        this.logGroup.applyRemovalPolicy(policy);
    }
}
exports.BaseDockerProject = BaseDockerProject;
_a = JSII_RTTI_SYMBOL_1;
BaseDockerProject[_a] = { fqn: "shady-island.automation.BaseDockerProject", version: "0.1.66" };
/**
 * Sets up a standardized Docker build project.
 *
 * This project accepts the following optional environment variables:
 *
 * - IMAGE_LABELS: JSON-formatted object of container labels and their values.
 * - BUILD_ARGS: JSON-formatted object of build arguments and their values.
 * - IMAGE_TAG: Optional. The image tag (e.g. Git commit ID) (default: build
 *   number).
 */
class LinuxDockerBuildProject extends BaseDockerProject {
    /**
     * Creates a new LinuxDockerBuildProject.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope, id, props) {
        const { dockerfile = "Dockerfile", buildDirectory = ".", enableCache = false, testCommands = [], pushLatest = true, } = props;
        const buildCommands = [
            'echo "Building the Docker image..."',
            'if [ -z "$IMAGE_TAG" ]; then IMAGE_TAG="$CODEBUILD_BUILD_NUMBER"; fi',
            [
                "build_args=()",
                'if [ -n "$BUILD_ARGS" ]; then for row in $(echo $BUILD_ARGS | jq -r .[]); do build_args+=("$row"); done; fi',
                'build_args_args=""',
                'for build_arg in "${build_args[@]}"; do build_args_args="--build-arg $build_arg $build_args_args"; done',
            ].join("\n"),
            [
                'labels=("org.opencontainers.image.created=$(date --iso-8601=seconds)")',
                'if [ -n "$IMAGE_LABELS" ]; then for row in $(echo $IMAGE_LABELS | jq -r .[]); do labels+=("$row"); done; fi',
                'label_args=""',
                'for label in "${labels[@]}"; do label_args="--label $label $label_args"; done',
            ].join("\n"),
        ];
        const buildArguments = [
            "$label_args",
            "$build_args_args",
            '-t "$AWS_ECR_URL:$IMAGE_TAG"',
            `-f ${dockerfile}`,
        ];
        if (enableCache) {
            buildArguments.push("--cache-from $AWS_ECR_URL:latest");
            buildCommands.push("docker pull $AWS_ECR_URL:latest || true");
        }
        const buildCommand = [
            "docker",
            "build",
            ...buildArguments,
            `"${buildDirectory}"`,
        ].join(" ");
        buildCommands.push(`echo Build command: ${buildCommand}`, buildCommand, 'docker inspect "$AWS_ECR_URL:$IMAGE_TAG"');
        if (testCommands.length > 0) {
            buildCommands.push(...testCommands);
        }
        const postBuildCommands = [
            'if [ "$CODEBUILD_BUILD_SUCCEEDING" == "0" ]; then exit 1; fi',
            'echo "Pushing the Docker image..."',
            'docker push "$AWS_ECR_URL:$IMAGE_TAG"',
        ];
        if (pushLatest) {
            postBuildCommands.push('docker tag "$AWS_ECR_URL:$IMAGE_TAG" $AWS_ECR_URL:latest', "docker push $AWS_ECR_URL:latest");
        }
        const buildSpec = aws_codebuild_1.BuildSpec.fromObjectToYaml({
            version: "0.2",
            env: {
                shell: "bash",
            },
            phases: {
                pre_build: {
                    commands: [
                        'echo "Logging in to Amazon ECR..."',
                        [
                            'aws ecr get-login-password --region "$AWS_DEFAULT_REGION"',
                            'docker login --username AWS --password-stdin "$AWS_ECR_HOSTNAME"',
                        ].join(" | "),
                    ],
                },
                build: {
                    commands: buildCommands,
                },
                post_build: {
                    commands: postBuildCommands,
                },
            },
        });
        super(scope, id, {
            repository: props.repository,
            buildSpec,
            description: props.description ?? "Builds and pushes a Linux Docker container",
            buildEnvironment: {
                buildImage: props.buildImage,
                computeType: props.computeType,
                privileged: true,
            },
            logRetention: props.logRetention,
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            securityGroups: props.securityGroups,
            removalPolicy: props.removalPolicy,
        });
    }
}
exports.LinuxDockerBuildProject = LinuxDockerBuildProject;
_b = JSII_RTTI_SYMBOL_1;
LinuxDockerBuildProject[_b] = { fqn: "shady-island.automation.LinuxDockerBuildProject", version: "0.1.66" };
/**
 * Sets up a standardized Docker manifest build project.
 *
 * This project accepts the following variables:
 *
 * - LATEST_TAG: Optional. The tag to push (default: "latest").
 * - MANIFEST_CUSTOM_TAG: Optional. The tag to push, in addition to $LATEST_TAG.
 */
class LinuxDockerManifestProject extends BaseDockerProject {
    /**
     * Creates a new LinuxDockerManifestProject.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope, id, props) {
        const { tagVariableNames } = props;
        const archTags = tagVariableNames.map((x) => `"$AWS_ECR_URL:$${x}"`);
        const buildCommands = [
            ...tagVariableNames.map((x) => `echo "${x}: $${x}"`),
            [
                'latest_tag="latest"',
                'if [ -n "$LATEST_TAG" ]; then',
                '  latest_tag="$LATEST_TAG"',
                "fi",
            ].join("\n"),
            [
                'if [ -n "$MANIFEST_CUSTOM_TAG" ]; then',
                [
                    '  docker manifest create "$AWS_ECR_URL:$MANIFEST_CUSTOM_TAG"',
                    ...archTags,
                ].join(" "),
                '  docker manifest inspect "$AWS_ECR_URL:$MANIFEST_CUSTOM_TAG"',
                '  docker manifest push "$AWS_ECR_URL:$MANIFEST_CUSTOM_TAG"',
                "fi",
            ].join("\n"),
            ['docker manifest create "$AWS_ECR_URL:$latest_tag"', ...archTags].join(" "),
            'docker manifest inspect "$AWS_ECR_URL:$latest_tag"',
            'docker manifest push "$AWS_ECR_URL:$latest_tag"',
        ];
        const buildSpec = aws_codebuild_1.BuildSpec.fromObjectToYaml({
            version: "0.2",
            env: { shell: "bash" },
            phases: {
                pre_build: {
                    commands: [
                        'echo "Logging in to Amazon ECR..."',
                        [
                            'aws ecr get-login-password --region "$AWS_DEFAULT_REGION"',
                            'docker login --username AWS --password-stdin "$AWS_ECR_HOSTNAME"',
                        ].join(" | "),
                    ],
                },
                build: {
                    commands: buildCommands,
                },
            },
        });
        super(scope, id, {
            repository: props.repository,
            buildSpec,
            description: props.description ?? "Pushes Docker manifest files",
            buildEnvironment: {
                buildImage: props.buildImage,
                computeType: props.computeType,
                privileged: true,
            },
            logRetention: props.logRetention,
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            securityGroups: props.securityGroups,
            removalPolicy: props.removalPolicy,
        });
    }
}
exports.LinuxDockerManifestProject = LinuxDockerManifestProject;
_c = JSII_RTTI_SYMBOL_1;
LinuxDockerManifestProject[_c] = { fqn: "shady-island.automation.LinuxDockerManifestProject", version: "0.1.66" };
//# sourceMappingURL=data:application/json;base64,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