import type { IFileSystem } from "aws-cdk-lib/aws-efs";
import type { IBucket } from "aws-cdk-lib/aws-s3";
import type { ISecret } from "aws-cdk-lib/aws-secretsmanager";
/**
 * Options for the `ShellCommands.outputFile` method.
 */
export interface OutputFileOptions {
    /**
     * The bash heredoc delimiter
     *
     * @default - END_OF_FILE
     */
    readonly delimiter?: string;
    /**
     * Use `true` to enable variable and command substitution inside the heredoc.
     *
     * @default - disabled
     */
    readonly substitution?: boolean;
}
/**
 * Options for the `ShellCommands.addDirectory` method.
 */
export interface AddDirectoryOptions {
    /**
     * The username or numeric user ID to assign as the directory owner.
     */
    readonly owner?: string;
    /**
     * The group name or numeric group ID to assign as the directory group.
     */
    readonly group?: string;
    /**
     * The file mode, e.g. 2755, 0400
     */
    readonly mode?: string;
}
/**
 * Constructor properties for AptPackagesAddOn.
 */
export interface InstallAptPackagesOptions {
    /**
     * Additional Apt Repositories to enable using add-apt-repository.
     */
    readonly repositories?: string[];
    /**
     * Whether to run apt autoremove after installation finishes.
     *
     * @default - true
     */
    readonly autoRemove?: boolean;
}
/**
 * A utility class that provides POSIX shell commands for User Data scripts.
 */
export declare class ShellCommands {
    /**
     * Writes the literal contents of a string to a destination file.
     *
     * @param contents - The file contents
     * @param destination - The filename to output
     * @param options - Configuration options
     * @returns The shell commands.
     */
    static outputFile(contents: string, destination: string, options?: OutputFileOptions): string[];
    /**
     * Gets commands to synchronize objects from an S3 bucket to the filesystem.
     *
     * e.g. `syncFromBucket(bucket, {"nginx-config": "/etc/nginx"})`.
     *
     * Be sure to grant your autoscaling group or EC2 instance read access.
     *
     * @param bucket - The source bucket
     * @param destinations - Record with S3 object keys to filesystem path values
     * @returns The shell commands.
     */
    static syncFromBucket(bucket: IBucket, destinations: Record<string, string>): string[];
    /**
     * Uses either `mkdir` or `install` to create a directory.
     *
     * @param name - The name of the directory to create
     * @param options - Configuration options
     * @returns The shell commands.
     */
    static addDirectory(name: string, options?: AddDirectoryOptions): string[];
    /**
     * Gets the command to download a Secrets Manager secret to the filesystem.
     *
     * Be sure to grant your autoscaling group or EC2 instance read access.
     *
     * @param secret - The secret to download.
     * @param destination - The local filesystem path where the secret is stored.
     * @returns The shell commands.
     */
    static downloadSecret(secret: ISecret, destination: string): string[];
    /**
     * Gets a command to change the ownership and/or group membership of a file.
     *
     * If both `uid` and `gid` are provided, this method returns a single
     * `chown` command to set both values. If just `uid` is provided, this method
     * returns a single `chown` command that sets the owner. If just `gid` is
     * provided, this method returns a single `chgrp` command. If neither are
     * provided, this method returns an empty array.
     *
     * @param filename - The local filesystem path to the file or directory.
     * @param uid - Optional. The owner username or uid.
     * @param gid - Optional. The group name or gid.
     * @returns The shell commands.
     */
    static changeOwnership(filename: string, uid?: string, gid?: string): string[];
    /**
     * Gets the command to mount an EFS filesystem to a destination path.
     *
     * Be sure to grant your autoscaling group or EC2 instance network access.
     *
     * @param filesystem - The EFS filesystem
     * @param destination - The local filesystem path for the mount point
     * @returns The shell commands.
     */
    static mountElasticFileSystem(filesystem: IFileSystem, destination: string): string[];
    /**
     * Gets a command to disable unattended package upgrades on Debian/Ubuntu.
     *
     * @returns The shell commands.
     */
    static disableUnattendedUpgrades(): string[];
    /**
     * Gets commands to setup APT and install packages.
     *
     * @param packages - The packages to install
     * @param options - Configuration options
     * @returns The shell commands.
     */
    static installAptPackages(packages: string[], options?: InstallAptPackagesOptions): string[];
}
