import { Port } from "aws-cdk-lib/aws-ec2";
import { Address } from "../networking/address";
/**
 * Used to configure on-instance firewall rules (e.g. iptables, firewalld)
 */
export interface IFirewallRules {
    /**
     * Declare an inbound rule.
     *
     * Only the following protocols are allowed: TCP, UDP, ICMP, and ICMPv6. The
     * address can be a single address or a range of addresses in CIDR notation.
     *
     * @param port - The ingress port
     * @param address - The source address
     * @returns provides a fluent interface
     */
    inbound(port: Port, address: Address): this;
    /**
     * Declare an inbound rule that covers all IPv4 addresses.
     *
     * Only the following protocols are allowed: TCP, UDP, ICMP, and ICMPv6.
     *
     * @param port - The ingress port
     * @returns provides a fluent interface
     */
    inboundFromAnyIpv4(port: Port): this;
    /**
     * Declare an inbound rule that covers all IPv6 addresses.
     *
     * Only the following protocols are allowed: TCP, UDP, ICMP, and ICMPv6.
     *
     * @param port - The ingress port
     * @returns provides a fluent interface
     */
    inboundFromAnyIpv6(port: Port): this;
    /**
     * Declare an outbound rule.
     *
     * Only the following protocols are allowed: TCP, UDP, ICMP, and ICMPv6. The
     * address can be a single address or a range of addresses in CIDR notation.
     *
     * @param port - The egress port
     * @param address - The target address
     * @returns provides a fluent interface
     */
    outbound(port: Port, address: Address): this;
    /**
     * Declare an outbound rule that covers all IPv4 addresses.
     *
     * Only the following protocols are allowed: TCP, UDP, and ICMP.
     *
     * @param port - The egress port
     * @returns provides a fluent interface
     */
    outboundToAnyIpv4(port: Port): this;
    /**
     * Declare an outbound rule that covers all IPv6 addresses.
     *
     * Only the following protocols are allowed: TCP, UDP, and ICMPv6.
     *
     * @param port - The egress port
     * @returns provides a fluent interface
     */
    outboundToAnyIpv6(port: Port): this;
    /**
     * Retrieves the shell commands used to configure the instance firewall.
     *
     * @returns An array of POSIX shell or PowerShell commands
     */
    buildCommands(): string[];
}
/**
 * Produces the appropriate commands to configure an on-instance firewall.
 */
export declare class InstanceFirewall {
    /**
     * Define an instance firewall using iptables/ip6tables.
     *
     * @returns An iptables-based on-instance firewall
     */
    static iptables(): IFirewallRules;
}
