"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TierTagger = exports.Tier = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const TIER_SYMBOL = Symbol.for("shady-island-tier");
/**
 * A deployment environment with a specific purpose and audience.
 *
 * You can create any Tier you like, but we include those explained by DTAP.
 *
 * @see https://en.wikipedia.org/wiki/Development,_testing,_acceptance_and_production
 */
class Tier {
    /**
     * Finds the deployment tier of the given construct.
     *
     * @param construct - The construct to inspect
     * @returns The assigned deployment tier if found, otherwise undefined
     */
    static of(construct) {
        for (const scope of construct.node.scopes.reverse()) {
            const tier = scope[TIER_SYMBOL];
            if (tier instanceof Tier) {
                return tier;
            }
        }
        return undefined;
    }
    /**
     * Return the deployment tier that corresponds to the provided value.
     *
     * Production: "live", "prod", or "production".
     * Acceptance: "uat", "stage", "staging", or "acceptance".
     * Testing: "qc", "qa", "test", or "testing".
     * Development: anything else.
     *
     * @param value - The value to parse, case-insensitive
     * @returns The matching deployment tier, or else `DEVELOPMENT`.
     */
    static parse(value) {
        let tier;
        switch (value.trim().toLowerCase()) {
            case "production":
            case "prod":
            case "live":
                tier = Tier.PRODUCTION;
                break;
            case "uat":
            case "stage":
            case "staging":
            case "accept":
            case "acceptance":
                tier = Tier.ACCEPTANCE;
                break;
            case "qc":
            case "qa":
            case "test":
            case "testing":
                tier = Tier.TESTING;
                break;
            default:
                tier = Tier.DEVELOPMENT;
        }
        return tier;
    }
    /**
     * Creates a new Tier.
     *
     * @param id - The machine-readable identifier for this tier (e.g. prod)
     * @param label - The human-readable label for this tier (e.g. Production)
     */
    constructor(id, label) {
        this.id = id;
        this.label = label;
    }
    /**
     * Compares this tier to the provided value and tests for equality.
     *
     * @param other - The value to compare.
     * @returns Whether the provided value is equal to this tier.
     */
    matches(other) {
        if (this === other) {
            return true;
        }
        return other instanceof Tier && this.id === other.id;
    }
    /**
     * Assigns this tier to a construct.
     *
     * This method will register an error annotation on the construct if any of
     * the constructs in its parent scopes have a different tier assigned.
     *
     * @param construct - The construct to receive the tier assignment
     */
    assignTo(construct) {
        const existingTier = construct[TIER_SYMBOL];
        if (this.matches(existingTier)) {
            // Return early if the tier is already set.
            return;
        }
        for (const scope of construct.node.scopes.reverse().slice(1)) {
            const tier = scope[TIER_SYMBOL];
            if (this.matches(tier)) {
                // Return early if a parent scope has the same tier.
                return;
            }
            else if (tier instanceof Tier) {
                // Register an error if a parent scope has a different tier.
                aws_cdk_lib_1.Annotations.of(construct).addError(`The tier assigned to this construct is different from the tier assigned to the path: ${scope.node.path}`);
                break;
            }
        }
        Object.defineProperty(construct, TIER_SYMBOL, {
            value: this,
            configurable: true,
            enumerable: false,
        });
    }
    /**
     * Adds the label of this tier as a tag to the provided construct.
     */
    applyTags(construct) {
        aws_cdk_lib_1.Tags.of(construct).add("DeploymentTier", this.label);
    }
}
exports.Tier = Tier;
_a = JSII_RTTI_SYMBOL_1;
Tier[_a] = { fqn: "shady-island.Tier", version: "0.1.66" };
/**
 * A tier that represents a production environment.
 */
Tier.PRODUCTION = new Tier("prod", "Production");
/**
 * A tier that represents an acceptance environment.
 */
Tier.ACCEPTANCE = new Tier("uat", "Acceptance");
/**
 * A tier that represents a testing environment.
 */
Tier.TESTING = new Tier("test", "Testing");
/**
 * A tier that represents a development environment.
 */
Tier.DEVELOPMENT = new Tier("dev", "Development");
/**
 * A CDK Aspect to apply the `DeploymentTier` tag to Stacks.
 */
class TierTagger {
    /**
     * Create a new TierTagger.
     *
     * @param tier - The deployment tier
     */
    constructor(tier) {
        this.tier = tier;
    }
    visit(node) {
        new aws_cdk_lib_1.Tag("DeploymentTier", this.tier.label).visit(node);
    }
}
exports.TierTagger = TierTagger;
_b = JSII_RTTI_SYMBOL_1;
TierTagger[_b] = { fqn: "shady-island.TierTagger", version: "0.1.66" };
//# sourceMappingURL=data:application/json;base64,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