"""
Defines types for temporally sequential data.

record            A dict corresponding to a row such as `{:timestamp TTTTT :value XXXX}`
timeseries        List of records: `[{key val ...}, ...]` with at least one of `timestamp date` as key.
metadata          A dict containing the signal metadata (attribute of timeseries)
dataset           A dict as `{'symbol' timeseries, ...}`.
symbols           A list of symbols (generated by `(.keys some-dataset)` corresponding to rows of a data matrix.)
dates             A list of dates (extracted from a timeseries) corresponding to columns of a data matrix.
data-matrix       A matrix with times aligned to columns and each row corresponding to a signal.

"""

(require hyrule [unless -> ->> as->]
         hyjinx.macros *)

(import collections [UserList UserDict])
(import cytoolz [first])
(import itertools [islice repeat])

(import numpy)


(defclass Timeseries [UserList]
  "A list of records with some extra methods and attributes.
Each record must have at least the following keys, set at init,
time-key : the key for the sequential time index
value-key : the key for the timeseries values."

  (defn __init__ [self [data None] * symbol [time-key "timestamp"] [value-key "value"] [metadata {}]]
    (setv self.data data)
    (setv self.symbol symbol)
    (setv self.time-key time-key)
    (setv self.value-key value-key)
    (setv self.metadata metadata))

  (defn values [self [kw None]]
    "A list of values on kw in the timeseries records."
    (amap (get it (or kw self.value-key)) self))

  (defn vector [self [kw None] * [dtype float]]
    "Convert a timeseries to a vector (on kw)."
    (numpy.array (.values self (or kw self.value-key)) :dtype dtype))

  (defn matrix [self [kw None]]
    "Convert a timeseries to a 1xn matrix (on kw)."
    (.reshape (.vector self (or kw self.value-key)) #(1 -1)))

  (defn dates [self]
    "A list of timestamps/dates in the timeseries records."
    (.values self self.time-key))

  (defn __repr__ [self]
    (import tabulate [tabulate])
    (.join "\n"
             [(tabulate (+ (list (islice self 10))
                           [(dict (zip (.keys (first self)) (repeat "...")))]
                           (list (reversed (list (islice (reversed self) 10)))))
                        :headers "keys")])))


(defclass Dataset [UserDict]
  "A dict of timeseries indexed by symbols.
No guarantees are made of alignment."

  (defn symbols [self]
     (list (.keys self)))

  (defn matrix [self kw * [dtype float]]
    "Convert a dataset to a matrix (samples x instruments, on kw)."
    (numpy.array
      (list (amap (list (.values it kw)) (.values self)))
      :dtype dtype))

  (defn vector [self kw symbol]
    "Convert timeseries with given symbol to a vector (on kw)."
    (.vector (get self symbol) kw))

  (defn time-key [self]
    (.time-key (first (.values self))))

  (defn dates [self]
    "A list of timestamps/dates in the dataset's timeseries records."
    (.dates (first (.values self))))

  (defn __repr__ [self]
    (.join "\n" (.symbols self))))
