"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fingerprint = void 0;
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const options_1 = require("./options");
const utils_1 = require("./utils");
const BUFFER_SIZE = 8 * 1024;
const CTRL_SOH = '\x01';
const CTRL_SOT = '\x02';
const CTRL_ETX = '\x03';
/**
 * Produces fingerprint based on the contents of a single file or an entire directory tree.
 *
 * The fingerprint will also include:
 * 1. An extra string if defined in `options.extra`.
 * 2. The set of exclude patterns, if defined in `options.exclude`
 * 3. The symlink follow mode value.
 *
 * @param fileOrDirectory The directory or file to fingerprint
 * @param options Fingerprinting options
 */
function fingerprint(fileOrDirectory, options = {}) {
    const hash = crypto.createHash('sha256');
    _hashField(hash, 'options.extra', options.extraHash || '');
    const follow = options.follow || options_1.SymlinkFollowMode.EXTERNAL;
    _hashField(hash, 'options.follow', follow);
    const rootDirectory = fs.statSync(fileOrDirectory).isDirectory()
        ? fileOrDirectory
        : path.dirname(fileOrDirectory);
    const exclude = options.exclude || [];
    if (exclude.length) {
        _hashField(hash, 'options.exclude', JSON.stringify(exclude));
    }
    const isDir = fs.statSync(fileOrDirectory).isDirectory();
    _processFileOrDirectory(fileOrDirectory, isDir);
    return hash.digest('hex');
    function _processFileOrDirectory(symbolicPath, isRootDir = false, realPath = symbolicPath) {
        if (!isRootDir && utils_1.shouldExclude(exclude, symbolicPath)) {
            return;
        }
        const stat = fs.lstatSync(realPath);
        const relativePath = path.relative(fileOrDirectory, symbolicPath);
        if (stat.isSymbolicLink()) {
            const linkTarget = fs.readlinkSync(realPath);
            const resolvedLinkTarget = path.resolve(path.dirname(realPath), linkTarget);
            if (utils_1.shouldFollow(follow, rootDirectory, resolvedLinkTarget)) {
                _processFileOrDirectory(symbolicPath, false, resolvedLinkTarget);
            }
            else {
                _hashField(hash, `link:${relativePath}`, linkTarget);
            }
        }
        else if (stat.isFile()) {
            _hashField(hash, `file:${relativePath}`, _contentFingerprint(realPath, stat));
        }
        else if (stat.isDirectory()) {
            for (const item of fs.readdirSync(realPath).sort()) {
                _processFileOrDirectory(path.join(symbolicPath, item), false, path.join(realPath, item));
            }
        }
        else {
            throw new Error(`Unable to hash ${symbolicPath}: it is neither a file nor a directory`);
        }
    }
}
exports.fingerprint = fingerprint;
function _contentFingerprint(file, stat) {
    const hash = crypto.createHash('sha256');
    const buffer = Buffer.alloc(BUFFER_SIZE);
    // tslint:disable-next-line: no-bitwise
    const fd = fs.openSync(file, fs.constants.O_DSYNC | fs.constants.O_RDONLY | fs.constants.O_SYNC);
    try {
        let read = 0;
        // tslint:disable-next-line: no-conditional-assignment
        while ((read = fs.readSync(fd, buffer, 0, BUFFER_SIZE, null)) !== 0) {
            hash.update(buffer.slice(0, read));
        }
    }
    finally {
        fs.closeSync(fd);
    }
    return `${stat.size}:${hash.digest('hex')}`;
}
function _hashField(hash, header, value) {
    hash.update(CTRL_SOH).update(header).update(CTRL_SOT).update(value).update(CTRL_ETX);
}
//# sourceMappingURL=data:application/json;base64,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