"use strict";
const lib_1 = require("../lib");
const encoding_1 = require("../lib/private/encoding");
const intrinsic_1 = require("../lib/private/intrinsic");
const resolve_1 = require("../lib/private/resolve");
const evaluate_cfn_1 = require("./evaluate-cfn");
const util_1 = require("./util");
class Promise2 {
    constructor() {
        this.creationStack = [];
    }
    resolve() {
        return {
            Data: {
                stringProp: 'hello',
                numberProp: 1234,
            },
            Recurse: new intrinsic_1.Intrinsic(42),
        };
    }
}
class Promise1 {
    constructor() {
        this.creationStack = [];
        this.p2 = [new Promise2(), new Promise2()];
    }
    resolve() {
        return this.p2;
    }
}
class BaseDataType {
    constructor(foo) {
        this.foo = foo;
    }
}
class DataType extends BaseDataType {
    constructor() {
        super(12);
        this.goo = 'hello';
    }
}
/**
 * Return Tokens in both flavors that resolve to the given string
 */
function tokensThatResolveTo(value) {
    return [
        new intrinsic_1.Intrinsic(value),
        lib_1.Lazy.anyValue({ produce: () => value }),
    ];
}
/**
 * Wrapper for resolve that creates an throwaway Construct to call it on
 *
 * So I don't have to change all call sites in this file.
 */
function resolve(x) {
    return new lib_1.Stack().resolve(x);
}
module.exports = {
    'resolve a plain old object should just return the object'(test) {
        const obj = { PlainOldObject: 123, Array: [1, 2, 3] };
        test.deepEqual(resolve(obj), obj);
        test.done();
    },
    'if a value is an object with a token value, it will be evaluated'(test) {
        const obj = {
            RegularValue: 'hello',
            LazyValue: new intrinsic_1.Intrinsic('World'),
        };
        test.deepEqual(resolve(obj), {
            RegularValue: 'hello',
            LazyValue: 'World',
        });
        test.done();
    },
    'tokens are evaluated anywhere in the object tree'(test) {
        const obj = new Promise1();
        const actual = resolve({ Obj: obj });
        test.deepEqual(actual, {
            Obj: [
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234,
                    },
                    Recurse: 42,
                },
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234,
                    },
                    Recurse: 42,
                },
            ],
        });
        test.done();
    },
    'tokens are evaluated recursively'(test) {
        const obj = new Promise1();
        const actual = resolve(new intrinsic_1.Intrinsic({ Obj: obj }));
        test.deepEqual(actual, {
            Obj: [
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234,
                    },
                    Recurse: 42,
                },
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234,
                    },
                    Recurse: 42,
                },
            ],
        });
        test.done();
    },
    'empty arrays or objects are kept'(test) {
        test.deepEqual(resolve({}), {});
        test.deepEqual(resolve([]), []);
        const obj = {
            Prop1: 1234,
            Prop2: {},
            Prop3: [],
            Prop4: 'hello',
            Prop5: {
                PropA: {},
                PropB: {
                    PropC: [undefined, undefined],
                    PropD: 'Yoohoo',
                },
            },
        };
        test.deepEqual(resolve(obj), {
            Prop1: 1234,
            Prop2: {},
            Prop3: [],
            Prop4: 'hello',
            Prop5: {
                PropA: {},
                PropB: {
                    PropC: [],
                    PropD: 'Yoohoo',
                },
            },
        });
        test.done();
    },
    'if an object has a "resolve" property that is not a function, it is not considered a token'(test) {
        test.deepEqual(resolve({ a_token: { resolve: () => 78787 } }), { a_token: 78787 });
        test.deepEqual(resolve({ not_a_token: { resolve: 12 } }), { not_a_token: { resolve: 12 } });
        test.done();
    },
    // tslint:disable-next-line:max-line-length
    'if a resolvable object inherits from a class that is also resolvable, the "constructor" function will not get in the way (uses Object.keys instead of "for in")'(test) {
        test.deepEqual(resolve({ prop: new DataType() }), { prop: { foo: 12, goo: 'hello' } });
        test.done();
    },
    'isToken(obj) can be used to determine if an object is a token'(test) {
        test.ok(lib_1.isResolvableObject({ resolve: () => 123 }));
        test.ok(lib_1.isResolvableObject({ a: 1, b: 2, resolve: () => 'hello' }));
        test.ok(!lib_1.isResolvableObject({ a: 1, b: 2, resolve: 3 }));
        test.done();
    },
    'Token can be used to create tokens that contain a constant value'(test) {
        test.equal(resolve(new intrinsic_1.Intrinsic(12)), 12);
        test.equal(resolve(new intrinsic_1.Intrinsic('hello')), 'hello');
        test.deepEqual(resolve(new intrinsic_1.Intrinsic(['hi', 'there'])), ['hi', 'there']);
        test.done();
    },
    'resolving leaves a Date object in working order'(test) {
        const date = new Date('2000-01-01');
        const resolved = resolve(date);
        test.equal(date.toString(), resolved.toString());
        test.done();
    },
    'tokens can be stringified and evaluated to conceptual value'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic('woof woof');
        // WHEN
        const stringified = `The dog says: ${token}`;
        const resolved = resolve(stringified);
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), 'The dog says: woof woof');
        test.done();
    },
    'tokens stringification can be reversed'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic('woof woof');
        // THEN
        test.equal(token, lib_1.Tokenization.reverseString(`${token}`).firstToken);
        test.done();
    },
    'Tokens stringification and reversing of CloudFormation Tokens is implemented using Fn::Join'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic(({ woof: 'woof' }));
        // WHEN
        const stringified = `The dog says: ${token}`;
        const resolved = resolve(stringified);
        // THEN
        test.deepEqual(resolved, {
            'Fn::Join': ['', ['The dog says: ', { woof: 'woof' }]],
        });
        test.done();
    },
    'Doubly nested strings evaluate correctly in scalar context'(test) {
        // GIVEN
        const token1 = new intrinsic_1.Intrinsic('world');
        const token2 = new intrinsic_1.Intrinsic(`hello ${token1}`);
        // WHEN
        const resolved1 = resolve(token2.toString());
        const resolved2 = resolve(token2);
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved1), 'hello world');
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved2), 'hello world');
        test.done();
    },
    'integer Tokens can be stringified and evaluate to conceptual value'(test) {
        // GIVEN
        for (const token of tokensThatResolveTo(1)) {
            // WHEN
            const stringified = `the number is ${token}`;
            const resolved = resolve(stringified);
            // THEN
            test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), 'the number is 1');
        }
        test.done();
    },
    'intrinsic Tokens can be stringified and evaluate to conceptual value'(test) {
        // GIVEN
        for (const bucketName of tokensThatResolveTo({ Ref: 'MyBucket' })) {
            // WHEN
            const resolved = resolve(`my bucket is named ${bucketName}`);
            // THEN
            const context = { MyBucket: 'TheName' };
            test.equal(evaluate_cfn_1.evaluateCFN(resolved, context), 'my bucket is named TheName');
        }
        test.done();
    },
    'tokens resolve properly in initial position'(test) {
        // GIVEN
        for (const token of tokensThatResolveTo('Hello')) {
            // WHEN
            const resolved = resolve(`${token} world`);
            // THEN
            test.equal(evaluate_cfn_1.evaluateCFN(resolved), 'Hello world');
        }
        test.done();
    },
    'side-by-side Tokens resolve correctly'(test) {
        // GIVEN
        for (const token1 of tokensThatResolveTo('Hello ')) {
            for (const token2 of tokensThatResolveTo('world')) {
                // WHEN
                const resolved = resolve(`${token1}${token2}`);
                // THEN
                test.equal(evaluate_cfn_1.evaluateCFN(resolved), 'Hello world');
            }
        }
        test.done();
    },
    'tokens can be used in hash keys but must resolve to a string'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic('I am a string');
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`,
        };
        // THEN
        test.deepEqual(resolve(s), { 'I am a string': 'boom I am a string' });
        test.done();
    },
    'tokens can be nested in hash keys'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic(lib_1.Lazy.stringValue({ produce: () => lib_1.Lazy.stringValue({ produce: (() => 'I am a string') }) }));
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`,
        };
        // THEN
        test.deepEqual(resolve(s), { 'I am a string': 'boom I am a string' });
        test.done();
    },
    'tokens can be nested and concatenated in hash keys'(test) {
        // GIVEN
        const innerToken = new intrinsic_1.Intrinsic('toot');
        const token = new intrinsic_1.Intrinsic(`${innerToken} the woot`);
        // WHEN
        const s = {
            [token.toString()]: 'boom chicago',
        };
        // THEN
        test.deepEqual(resolve(s), { 'toot the woot': 'boom chicago' });
        test.done();
    },
    'can find nested tokens in hash keys'(test) {
        // GIVEN
        const innerToken = new intrinsic_1.Intrinsic('toot');
        const token = new intrinsic_1.Intrinsic(`${innerToken} the woot`);
        // WHEN
        const s = {
            [token.toString()]: 'boom chicago',
        };
        // THEN
        const tokens = resolve_1.findTokens(new lib_1.Stack(), () => s);
        test.ok(tokens.some(t => t === innerToken), 'Cannot find innerToken');
        test.ok(tokens.some(t => t === token), 'Cannot find token');
        test.done();
    },
    'fails if token in a hash key resolves to a non-string'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`,
        };
        // THEN
        test.throws(() => resolve(s), 'The key "${Token[TOKEN.19]}" has been resolved to {"Ref":"Other"} but must be resolvable to a string');
        test.done();
    },
    'list encoding': {
        'can encode Token to string and resolve the encoding'(test) {
            // GIVEN
            const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
            // WHEN
            const struct = {
                XYZ: lib_1.Token.asList(token),
            };
            // THEN
            test.deepEqual(resolve(struct), {
                XYZ: { Ref: 'Other' },
            });
            test.done();
        },
        'cannot add to encoded list'(test) {
            // GIVEN
            const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
            // WHEN
            const encoded = lib_1.Token.asList(token);
            encoded.push('hello');
            // THEN
            test.throws(() => {
                resolve(encoded);
            }, /Cannot add elements to list token/);
            test.done();
        },
        'cannot add to strings in encoded list'(test) {
            // GIVEN
            const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
            // WHEN
            const encoded = lib_1.Token.asList(token);
            encoded[0] += 'hello';
            // THEN
            test.throws(() => {
                resolve(encoded);
            }, /concatenate strings in/);
            test.done();
        },
        'can pass encoded lists to FnSelect'(test) {
            // GIVEN
            const encoded = lib_1.Token.asList(new intrinsic_1.Intrinsic({ Ref: 'Other' }));
            // WHEN
            const struct = lib_1.Fn.select(1, encoded);
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Select': [1, { Ref: 'Other' }],
            });
            test.done();
        },
        'can pass encoded lists to FnJoin'(test) {
            // GIVEN
            const encoded = lib_1.Token.asList(new intrinsic_1.Intrinsic({ Ref: 'Other' }));
            // WHEN
            const struct = lib_1.Fn.join('/', encoded);
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Join': ['/', { Ref: 'Other' }],
            });
            test.done();
        },
        'can pass encoded lists to FnJoin, even if join is stringified'(test) {
            // GIVEN
            const encoded = lib_1.Token.asList(new intrinsic_1.Intrinsic({ Ref: 'Other' }));
            // WHEN
            const struct = lib_1.Fn.join('/', encoded).toString();
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Join': ['/', { Ref: 'Other' }],
            });
            test.done();
        },
    },
    'number encoding': {
        'basic integer encoding works'(test) {
            test.equal(16, encoding_1.extractTokenDouble(encoding_1.createTokenDouble(16)));
            test.done();
        },
        'arbitrary integers can be encoded, stringified, and recovered'(test) {
            for (let i = 0; i < 100; i++) {
                // We can encode all numbers up to 2^48-1
                const x = Math.floor(Math.random() * (Math.pow(2, 48) - 1));
                const encoded = encoding_1.createTokenDouble(x);
                // Roundtrip through JSONification
                const roundtripped = JSON.parse(JSON.stringify({ theNumber: encoded })).theNumber;
                const decoded = encoding_1.extractTokenDouble(roundtripped);
                test.equal(decoded, x, `Fail roundtrip encoding of ${x}`);
            }
            test.done();
        },
        'arbitrary numbers are correctly detected as not being tokens'(test) {
            test.equal(undefined, encoding_1.extractTokenDouble(0));
            test.equal(undefined, encoding_1.extractTokenDouble(1243));
            test.equal(undefined, encoding_1.extractTokenDouble(4835e+532));
            test.done();
        },
        'can number-encode and resolve Token objects'(test) {
            // GIVEN
            const x = new intrinsic_1.Intrinsic(123);
            // THEN
            const encoded = lib_1.Token.asNumber(x);
            test.equal(false, lib_1.isResolvableObject(encoded), 'encoded number does not test as token');
            test.equal(true, lib_1.Token.isUnresolved(encoded), 'encoded number does not test as token');
            // THEN
            const resolved = resolve({ value: encoded });
            test.deepEqual(resolved, { value: 123 });
            test.done();
        },
    },
    'stack trace is captured at token creation'(test) {
        function fn1() {
            function fn2() {
                class ExposeTrace extends intrinsic_1.Intrinsic {
                    get creationTrace() {
                        return this.creationStack;
                    }
                }
                return new ExposeTrace('hello');
            }
            return fn2();
        }
        const previousValue = util_1.reEnableStackTraceCollection();
        const token = fn1();
        util_1.restoreStackTraceColection(previousValue);
        test.ok(token.creationTrace.find(x => x.includes('fn1')));
        test.ok(token.creationTrace.find(x => x.includes('fn2')));
        test.done();
    },
    'newError returns an error with the creation stack trace'(test) {
        function fn1() {
            function fn2() {
                function fn3() {
                    class ThrowingToken extends intrinsic_1.Intrinsic {
                        throwError(message) {
                            throw this.newError(message);
                        }
                    }
                    return new ThrowingToken('boom');
                }
                return fn3();
            }
            return fn2();
        }
        const previousValue = util_1.reEnableStackTraceCollection();
        const token = fn1();
        util_1.restoreStackTraceColection(previousValue);
        test.throws(() => token.throwError('message!'), /Token created:/);
        test.done();
    },
    'type coercion': (() => {
        const tests = {};
        const inputs = [
            'a string',
            1234,
            { an_object: 1234 },
            [1, 2, 3],
            false,
        ];
        for (const input of inputs) {
            // GIVEN
            const stringToken = lib_1.Token.asString(new intrinsic_1.Intrinsic(input));
            const numberToken = lib_1.Token.asNumber(new intrinsic_1.Intrinsic(input));
            const listToken = lib_1.Token.asList(new intrinsic_1.Intrinsic(input));
            // THEN
            const expected = input;
            tests[`${input}<string>.toNumber()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asNumber(new intrinsic_1.Intrinsic(stringToken))), expected);
                test.done();
            };
            tests[`${input}<list>.toNumber()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asNumber(new intrinsic_1.Intrinsic(listToken))), expected);
                test.done();
            };
            tests[`${input}<number>.toNumber()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asNumber(new intrinsic_1.Intrinsic(numberToken))), expected);
                test.done();
            };
            tests[`${input}<string>.toString()`] = (test) => {
                test.deepEqual(resolve(new intrinsic_1.Intrinsic(stringToken).toString()), expected);
                test.done();
            };
            tests[`${input}<list>.toString()`] = (test) => {
                test.deepEqual(resolve(new intrinsic_1.Intrinsic(listToken).toString()), expected);
                test.done();
            };
            tests[`${input}<number>.toString()`] = (test) => {
                test.deepEqual(resolve(new intrinsic_1.Intrinsic(numberToken).toString()), expected);
                test.done();
            };
            tests[`${input}<string>.toList()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asList(new intrinsic_1.Intrinsic(stringToken))), expected);
                test.done();
            };
            tests[`${input}<list>.toList()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asList(new intrinsic_1.Intrinsic(listToken))), expected);
                test.done();
            };
            tests[`${input}<number>.toList()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asList(new intrinsic_1.Intrinsic(numberToken))), expected);
                test.done();
            };
        }
        return tests;
    })(),
    'creation stack is attached to errors emitted during resolve'(test) {
        function showMeInTheStackTrace() {
            return lib_1.Lazy.stringValue({ produce: () => { throw new Error('fooError'); } });
        }
        const previousValue = util_1.reEnableStackTraceCollection();
        const x = showMeInTheStackTrace();
        let message;
        try {
            resolve(x);
        }
        catch (e) {
            message = e.message;
        }
        finally {
            util_1.restoreStackTraceColection(previousValue);
        }
        test.ok(message && message.includes('showMeInTheStackTrace'));
        test.done();
    },
    'stringifyNumber': {
        'converts number to string'(test) {
            test.equal(lib_1.Tokenization.stringifyNumber(100), '100');
            test.done();
        },
        'converts tokenized number to string'(test) {
            test.equal(resolve(lib_1.Tokenization.stringifyNumber({
                resolve: () => 100,
            })), '100');
            test.done();
        },
        'string remains the same'(test) {
            test.equal(lib_1.Tokenization.stringifyNumber('123'), '123');
            test.done();
        },
        'Ref remains the same'(test) {
            const val = { Ref: 'SomeLogicalId' };
            test.deepEqual(lib_1.Tokenization.stringifyNumber(val), val);
            test.done();
        },
        'lazy Ref remains the same'(test) {
            const resolvedVal = { Ref: 'SomeLogicalId' };
            const tokenizedVal = lib_1.Lazy.anyValue({
                produce: () => resolvedVal,
            });
            const res = lib_1.Tokenization.stringifyNumber(tokenizedVal);
            test.notDeepEqual(res, resolvedVal);
            test.deepEqual(resolve(res), resolvedVal);
            test.done();
        },
        'tokenized Ref remains the same'(test) {
            const resolvedVal = { Ref: 'SomeLogicalId' };
            const tokenizedVal = lib_1.Token.asNumber(resolvedVal);
            const res = lib_1.Tokenization.stringifyNumber(tokenizedVal);
            test.notDeepEqual(res, resolvedVal);
            test.deepEqual(resolve(res), resolvedVal);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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