"""
Multimodal Integration Module

This module provides integration between the multimodal tools and the file storage system.
It extends the existing file storage capabilities with multimodal processing.

Requirements covered:
- 4.2: Add multimodal processing capabilities at upload
- 3.1, 3.2, 3.3: Multimodal image analysis and metadata extraction
- 5.4: Update file metadata to indicate multimodal capabilities

v 0.1.0 - Initial implementation
"""

import os
import logging
from typing import Optional, Dict, Any, TYPE_CHECKING
from datetime import datetime

# Import comprehensive error handling system
from agent_framework.monitoring.error_handling import (
    ErrorHandler,
    FileProcessingError,
    FileProcessingErrorType,
    ErrorSeverity,
    MultimodalProcessingError,
    ProcessingIssue,
    ErrorHandlingResult,
    UserFeedbackGenerator
)

if TYPE_CHECKING:
    from agent_framework.storage.file_storages import FileMetadata
    from agent_framework.storage.file_system_management import FileStorageManager

from agent_framework.tools.multimodal_tools import MultimodalProcessor, MultimodalProcessingResult

logger = logging.getLogger(__name__)


class MultimodalFileStorageIntegration:
    """
    Integration layer between file storage and multimodal processing.
    
    This class provides methods to enhance file storage operations with
    multimodal capabilities, particularly for image processing at upload time.
    """
    
    def __init__(self, file_storage_manager: 'FileStorageManager'):
        self.file_storage_manager = file_storage_manager
        self.multimodal_processor = MultimodalProcessor(enable_auto_analysis=False)
        
        logger.info("MultimodalFileStorageIntegration initialized")
    
    async def process_file_with_multimodal_capabilities(self, 
                                                      content: bytes, 
                                                      filename: str, 
                                                      mime_type: str,
                                                      user_id: str,
                                                      session_id: Optional[str] = None,
                                                      agent_id: Optional[str] = None,
                                                      is_generated: bool = False,
                                                      tags: Optional[list] = None,
                                                      custom_metadata: Optional[Dict] = None) -> str:
        """
        Store file with enhanced multimodal processing capabilities.
        
        This method:
        1. Processes the file for multimodal capabilities
        2. Updates metadata with multimodal information
        3. Stores the file using the standard storage workflow
        
        Args:
            content: File content bytes
            filename: Original filename
            mime_type: MIME type of the file
            user_id: User ID for file storage
            session_id: Optional session ID
            agent_id: Optional agent ID
            is_generated: Whether file was generated by agent
            tags: Optional list of tags
            custom_metadata: Optional custom metadata
            
        Returns:
            File ID of the stored file
        """
        try:
            # Process file for multimodal capabilities
            multimodal_result = await self.multimodal_processor.process_image_at_upload(
                content, mime_type, filename
            )
            
            # Prepare enhanced metadata
            enhanced_metadata = custom_metadata or {}
            enhanced_tags = tags or []
            
            # Add multimodal information to metadata
            if multimodal_result.success:
                enhanced_metadata.update({
                    'multimodal_processing': {
                        'has_visual_content': multimodal_result.has_visual_content,
                        'processing_status': multimodal_result.processing_status.value,
                        'capabilities_available': [cap.value for cap in multimodal_result.capabilities_available],
                        'basic_metadata': multimodal_result.basic_metadata,
                        'processing_time_ms': multimodal_result.processing_time_ms,
                        'processed_at': datetime.now().isoformat()
                    }
                })
                
                # Add multimodal tags
                if multimodal_result.has_visual_content:
                    enhanced_tags.extend(['multimodal-ready', 'visual-content'])
                
                logger.info(f"Enhanced {filename} with multimodal capabilities (visual: {multimodal_result.has_visual_content})")
            else:
                enhanced_metadata.update({
                    'multimodal_processing': {
                        'has_visual_content': False,
                        'processing_status': multimodal_result.processing_status.value,
                        'error_message': multimodal_result.error_message,
                        'processing_time_ms': multimodal_result.processing_time_ms,
                        'processed_at': datetime.now().isoformat()
                    }
                })
                
                logger.debug(f"Multimodal processing not applicable for {filename}")
            
            # Store file with enhanced metadata using dual storage workflow
            original_file_id, markdown_file_id = await self.file_storage_manager.store_file_with_markdown_conversion(
                content=content,
                filename=filename,
                user_id=user_id,
                session_id=session_id,
                agent_id=agent_id,
                is_generated=is_generated,
                mime_type=mime_type,
                tags=enhanced_tags,
                custom_metadata=enhanced_metadata
            )
            
            # Update file metadata with multimodal processing results
            if original_file_id:
                await self._update_file_metadata_with_multimodal_info(
                    original_file_id, multimodal_result
                )
            
            logger.info(f"Successfully stored file {filename} with multimodal processing (ID: {original_file_id})")
            return original_file_id
            
        except Exception as e:
            logger.error(f"Failed to process file {filename} with multimodal capabilities: {e}")
            # Fallback to standard storage without multimodal processing
            return await self.file_storage_manager.store_file(
                content=content,
                filename=filename,
                user_id=user_id,
                session_id=session_id,
                agent_id=agent_id,
                is_generated=is_generated,
                mime_type=mime_type,
                tags=tags,
                custom_metadata=custom_metadata
            )
    
    async def _update_file_metadata_with_multimodal_info(self, 
                                                       file_id: str, 
                                                       multimodal_result: MultimodalProcessingResult):
        """Update file metadata with multimodal processing information"""
        try:
            metadata_updates = {
                'has_visual_content': multimodal_result.has_visual_content,
                'multimodal_processing_status': multimodal_result.processing_status.value,
                'updated_at': datetime.now()
            }
            
            # Add processing error if failed
            if not multimodal_result.success and multimodal_result.error_message:
                metadata_updates['processing_errors'] = [multimodal_result.error_message]
            
            # Add basic metadata if available
            if multimodal_result.basic_metadata:
                metadata_updates['custom_metadata'] = {
                    'multimodal_basic_info': multimodal_result.basic_metadata
                }
            
            await self.file_storage_manager.update_metadata(file_id, metadata_updates)
            logger.debug(f"Updated file {file_id} metadata with multimodal information")
            
        except Exception as e:
            logger.error(f"Failed to update file metadata with multimodal info: {e}")
    
    async def get_multimodal_file_info(self, file_id: str) -> Dict[str, Any]:
        """
        Get comprehensive multimodal information for a file.
        
        Args:
            file_id: ID of the file
            
        Returns:
            Dictionary with multimodal information and capabilities
        """
        try:
            # Get file metadata
            metadata = await self.file_storage_manager.get_file_metadata(file_id)
            if not metadata:
                return {
                    'file_id': file_id,
                    'exists': False,
                    'error': 'File not found'
                }
            
            # Check if file has multimodal capabilities
            has_visual_content = metadata.has_visual_content or (
                metadata.mime_type and metadata.mime_type.startswith('image/')
            )
            
            # Get available capabilities
            capabilities = []
            if has_visual_content and self.multimodal_processor.multimodal_enabled:
                from agent_framework.tools.multimodal_tools import MultimodalCapability
                capabilities = [cap.value for cap in MultimodalCapability]
            
            return {
                'file_id': file_id,
                'filename': metadata.filename,
                'mime_type': metadata.mime_type,
                'has_visual_content': has_visual_content,
                'multimodal_processing_status': metadata.multimodal_processing_status,
                'multimodal_enabled': self.multimodal_processor.multimodal_enabled,
                'available_capabilities': capabilities,
                'image_analysis_result': metadata.image_analysis_result,
                'processing_errors': metadata.processing_errors,
                'processing_warnings': metadata.processing_warnings,
                'custom_metadata': metadata.custom_metadata
            }
            
        except Exception as e:
            logger.error(f"Failed to get multimodal file info for {file_id}: {e}")
            return {
                'file_id': file_id,
                'exists': False,
                'error': str(e)
            }
    
    async def analyze_stored_image(self, file_id: str, analysis_prompt: Optional[str] = None) -> Dict[str, Any]:
        """
        Analyze a stored image using multimodal capabilities.
        
        This is a convenience method that uses the ImageAnalysisTool
        to analyze images that are already stored in the system.
        
        Args:
            file_id: ID of the stored image file
            analysis_prompt: Optional specific prompt for analysis
            
        Returns:
            Dictionary with analysis results
        """
        try:
            from agent_framework.tools.multimodal_tools import ImageAnalysisTool
            
            # Create analysis tool
            analysis_tool = ImageAnalysisTool(self.file_storage_manager)
            
            # Perform analysis
            result = await analysis_tool.analyze_image(file_id, analysis_prompt)
            
            # Convert to dictionary format
            return {
                'success': result.success,
                'description': result.description,
                'objects_detected': result.objects_detected,
                'text_detected': result.text_detected,
                'confidence_scores': result.confidence_scores,
                'error_message': result.error_message,
                'processing_time_ms': result.processing_time_ms,
                'capabilities_used': [cap.value for cap in result.capabilities_used],
                'metadata': result.metadata,
                'user_friendly_summary': result.user_friendly_summary
            }
            
        except Exception as e:
            logger.error(f"Failed to analyze stored image {file_id}: {e}")
            return {
                'success': False,
                'error_message': str(e),
                'file_id': file_id
            }


# ===== ENHANCED FILE INPUT PROCESSING =====

async def process_file_inputs_with_multimodal(
    agent_input,
    file_storage_manager: 'FileStorageManager',
    user_id: str = "default_user",
    session_id: str = "default_session",
    store_files: bool = True,
    include_text_content: bool = True,
    convert_to_markdown: bool = True,
    enable_multimodal_processing: bool = True
):
    """
    Enhanced version of process_file_inputs with multimodal capabilities.
    
    This function extends the standard file processing with multimodal
    image analysis and enhanced metadata extraction.
    
    Args:
        agent_input: The original StructuredAgentInput
        file_storage_manager: File storage manager instance
        user_id: User ID for file storage
        session_id: Session ID for file storage
        store_files: Whether to store files persistently
        include_text_content: Whether to include text file content
        convert_to_markdown: Whether to attempt markdown conversion
        enable_multimodal_processing: Whether to enable multimodal processing
        
    Returns:
        Tuple containing modified input and enhanced file metadata
    """
    if not enable_multimodal_processing:
        # Fall back to standard processing
        from .file_system_management import process_file_inputs
        return await process_file_inputs(
            agent_input, file_storage_manager, user_id, session_id,
            store_files, include_text_content, convert_to_markdown
        )
    
    # Use multimodal integration for enhanced processing
    multimodal_integration = MultimodalFileStorageIntegration(file_storage_manager)
    
    # Process files with multimodal capabilities
    # This would extend the existing process_file_inputs logic
    # For now, we'll delegate to the standard function and add multimodal info
    from .file_system_management import process_file_inputs
    processed_input, uploaded_files = await process_file_inputs(
        agent_input, file_storage_manager, user_id, session_id,
        store_files, include_text_content, convert_to_markdown
    )
    
    # Enhance file metadata with multimodal information
    enhanced_files = []
    for file_info in uploaded_files:
        if 'file_id' in file_info:
            multimodal_info = await multimodal_integration.get_multimodal_file_info(file_info['file_id'])
            file_info.update({
                'multimodal_info': multimodal_info,
                'has_visual_content': multimodal_info.get('has_visual_content', False),
                'multimodal_capabilities': multimodal_info.get('available_capabilities', [])
            })
        enhanced_files.append(file_info)
    
    return processed_input, enhanced_files


def get_multimodal_file_processing_summary(uploaded_files: list) -> str:
    """
    Generate a human-readable summary of processed files with multimodal information.
    
    Args:
        uploaded_files: List of enhanced file metadata dictionaries
        
    Returns:
        Formatted string summarizing the files and their multimodal capabilities
    """
    if not uploaded_files:
        return "No files uploaded."
    
    total_size = sum(f.get('size_bytes', 0) for f in uploaded_files)
    file_types = set(f.get('mime_type', 'unknown') for f in uploaded_files)
    
    # Count multimodal files
    multimodal_files = sum(1 for f in uploaded_files if f.get('has_visual_content', False))
    
    summary_parts = [
        f"📁 {len(uploaded_files)} file(s) uploaded",
        f"📊 Total size: {total_size:,} bytes",
        f"🏷️ Types: {', '.join(sorted(file_types))}"
    ]
    
    if multimodal_files > 0:
        summary_parts.append(f"🖼️ {multimodal_files} file(s) with visual content")
    
    if len(uploaded_files) <= 5:  # Show individual files if not too many
        file_list = []
        for f in uploaded_files:
            file_desc = f"• {f['filename']} ({f.get('size_bytes', 0):,} bytes)"
            if f.get('has_visual_content', False):
                file_desc += " [Visual Content]"
            if f.get('conversion_success', False):
                file_desc += " [Markdown ✓]"
            file_list.append(file_desc)
        summary_parts.extend(["", "Files:"] + file_list)
    
    return "\n".join(summary_parts)