"""
File Storage Implementations

Comprehensive file storage system with multiple backend implementations:
- Abstract interface for extensible storage backends
- Local filesystem storage
- AWS S3 storage
- MinIO storage

v 0.1.9 - Consolidated module
"""

import os
import json
import logging
from abc import ABC, abstractmethod
from typing import Optional, List, Dict, Any, Union, Tuple
from dataclasses import dataclass, field
from datetime import datetime
from pathlib import Path
from io import BytesIO
import uuid

import aiofiles

logger = logging.getLogger(__name__)


# ===== ENHANCED ERROR HANDLING INTEGRATION =====

# Import comprehensive error handling system
from agent_framework.monitoring.error_handling import (
    ErrorHandler,
    FileProcessingError,
    FileProcessingErrorType,
    ErrorSeverity,
    StorageError,
    ConversionError,
    MultimodalProcessingError,
    ValidationError,
    ProcessingIssue,
    ErrorHandlingResult,
    UserFeedbackGenerator,
    GracefulDegradationHandler
)


# ===== FILE METADATA MODEL =====

@dataclass
class FileMetadata:
    """Enhanced file metadata with multimodal processing, AI generation tracking, and dual file storage"""
    
    # Core file identification and basic metadata
    file_id: str
    filename: str
    mime_type: Optional[str]
    size_bytes: int
    created_at: datetime
    updated_at: datetime
    user_id: str
    session_id: Optional[str]
    agent_id: Optional[str]
    is_generated: bool  # True if file was generated by agent
    tags: List[str] = field(default_factory=list)
    custom_metadata: Dict[str, Any] = field(default_factory=dict)
    storage_backend: str = ""  # Which storage system contains this file
    storage_path: str = ""     # Backend-specific path/key
    
    # Enhanced markdown conversion fields for dual file storage
    markdown_content: Optional[str] = None  # Converted markdown content (for quick access)
    markdown_file_id: Optional[str] = None  # ID of stored markdown file version
    conversion_status: str = "not_attempted"  # success/failed/not_supported/not_attempted
    conversion_timestamp: Optional[datetime] = None  # When conversion was attempted
    conversion_error: Optional[str] = None  # Error message if conversion failed
    
    # Multimodal processing fields
    has_visual_content: bool = False  # True if file contains visual content (images, etc.)
    image_analysis_result: Optional[Dict[str, Any]] = None  # Results from image analysis
    multimodal_processing_status: str = "not_attempted"  # success/failed/not_supported/not_attempted
    
    # Enhanced processing status tracking
    processing_errors: List[str] = field(default_factory=list)  # List of processing errors
    processing_warnings: List[str] = field(default_factory=list)  # List of processing warnings
    total_processing_time_ms: Optional[float] = None  # Total time spent processing file
    
    # AI generation tracking fields
    generation_model: Optional[str] = None  # Model used to generate content (if is_generated=True)
    generation_prompt: Optional[str] = None  # Prompt used for generation
    generation_parameters: Optional[Dict[str, Any]] = None  # Parameters used for generation


# ===== ABSTRACT STORAGE INTERFACE =====

class FileStorageInterface(ABC):
    """Abstract interface for file storage backends"""
    
    @abstractmethod
    async def initialize(self) -> bool:
        """Initialize the storage backend"""
        pass
    
    @abstractmethod
    async def store_file(self, 
                        content: bytes, 
                        filename: str, 
                        metadata: FileMetadata) -> str:
        """Store file and return file_id"""
        pass
    
    @abstractmethod
    async def retrieve_file(self, file_id: str) -> tuple[bytes, FileMetadata]:
        """Retrieve file content and metadata"""
        pass
    
    @abstractmethod
    async def delete_file(self, file_id: str) -> bool:
        """Delete file"""
        pass
    
    @abstractmethod
    async def list_files(self, 
                        user_id: str, 
                        session_id: Optional[str] = None,
                        agent_id: Optional[str] = None,
                        is_generated: Optional[bool] = None) -> List[FileMetadata]:
        """List files with filtering"""
        pass
    
    @abstractmethod
    async def update_metadata(self, file_id: str, metadata: Dict[str, Any]) -> bool:
        """Update file metadata"""
        pass
    
    @abstractmethod
    async def file_exists(self, file_id: str) -> bool:
        """Check if file exists"""
        pass
    
    @abstractmethod
    async def get_file_metadata(self, file_id: str) -> Optional[FileMetadata]:
        """Get file metadata without content"""
        pass
    
    @abstractmethod
    async def convert_file_to_markdown(self, file_id: str) -> Optional[str]:
        """Convert file to markdown and return the content"""
        pass
    
    @abstractmethod
    async def analyze_image(self, file_id: str) -> Optional[Dict[str, Any]]:
        """Analyze image content using multimodal capabilities"""
        pass
    
    @abstractmethod
    async def get_processing_status(self, file_id: str) -> Dict[str, Any]:
        """Get comprehensive processing status for a file"""
        pass
    
    @abstractmethod
    async def store_markdown_version(self, original_file_id: str, markdown_content: str) -> Optional[str]:
        """Store markdown version of a file as a separate file"""
        pass
    
    @abstractmethod
    async def retrieve_markdown_version(self, original_file_id: str) -> Optional[Tuple[str, 'FileMetadata']]:
        """Retrieve markdown version of a file"""
        pass


# ===== LOCAL FILE STORAGE IMPLEMENTATION =====

class LocalFileStorage(FileStorageInterface):
    """Local filesystem storage implementation"""
    
    def __init__(self, base_path: str = "./file_storage"):
        self.base_path = Path(base_path)
        self.files_dir = self.base_path / "files"
        self.metadata_file = self.base_path / "metadata.json"
        self._metadata_cache: Dict[str, FileMetadata] = {}
        self._metadata_lock = None  # Will be initialized in initialize()
    
    async def initialize(self) -> bool:
        """Create storage directories and load metadata"""
        try:
            # Initialize the metadata lock for concurrent write protection
            import asyncio
            self._metadata_lock = asyncio.Lock()
            
            # Create directories
            self.base_path.mkdir(parents=True, exist_ok=True)
            self.files_dir.mkdir(exist_ok=True)
            
            # Load existing metadata
            await self._load_metadata()
            
            logger.info(f"Initialized LocalFileStorage at {self.base_path}")
            return True
        except Exception as e:
            logger.error(f"Failed to initialize LocalFileStorage: {e}")
            return False
    
    async def store_file(self, content: bytes, filename: str, metadata: FileMetadata) -> str:
        """Store file locally and update metadata with comprehensive error handling"""
        error_handler = ErrorHandler()
        
        try:
            # Validate inputs
            if not content:
                raise ValidationError(
                    error_type=FileProcessingErrorType.FILE_EMPTY,
                    severity=ErrorSeverity.ERROR,
                    message="Cannot store empty file",
                    user_message=f"File {filename} is empty and cannot be stored"
                )
            
            if len(content) > 100 * 1024 * 1024:  # 100MB limit
                raise ValidationError(
                    error_type=FileProcessingErrorType.FILE_TOO_LARGE,
                    severity=ErrorSeverity.ERROR,
                    message=f"File {filename} exceeds size limit",
                    user_message=f"File {filename} is too large (max 100MB allowed)"
                )
            
            # Create storage filename with format: id_original_name.ext
            original_name = Path(filename)
            storage_filename = f"{metadata.file_id}_{original_name.stem}{original_name.suffix}"
            file_path = self.files_dir / storage_filename
            
            # Check disk space before writing
            try:
                available_space = os.statvfs(self.base_path).f_bavail * os.statvfs(self.base_path).f_frsize
                if len(content) > available_space:
                    raise StorageError(
                        error_type=FileProcessingErrorType.DISK_SPACE_FULL,
                        severity=ErrorSeverity.ERROR,
                        message="Insufficient disk space",
                        user_message="Not enough storage space available",
                        backend_name="local"
                    )
            except AttributeError:
                # statvfs not available on Windows, skip check
                pass
            
            # Write file content with error handling
            try:
                async with aiofiles.open(file_path, 'wb') as f:
                    await f.write(content)
            except PermissionError as e:
                raise StorageError(
                    error_type=FileProcessingErrorType.STORAGE_PERMISSION_DENIED,
                    severity=ErrorSeverity.ERROR,
                    message=f"Permission denied writing to {file_path}",
                    user_message="Storage permission denied - contact administrator",
                    backend_name="local",
                    technical_details=str(e)
                )
            except OSError as e:
                if "No space left" in str(e):
                    raise StorageError(
                        error_type=FileProcessingErrorType.DISK_SPACE_FULL,
                        severity=ErrorSeverity.ERROR,
                        message="Disk space full",
                        user_message="Storage space is full",
                        backend_name="local",
                        technical_details=str(e)
                    )
                else:
                    raise StorageError(
                        error_type=FileProcessingErrorType.STORAGE_WRITE_FAILED,
                        severity=ErrorSeverity.ERROR,
                        message=f"Failed to write file: {str(e)}",
                        user_message="File storage failed due to system error",
                        backend_name="local",
                        technical_details=str(e)
                    )
            
            # Update metadata with actual size and storage path
            metadata.storage_path = str(file_path)
            metadata.size_bytes = len(content)
            metadata.storage_backend = "local"
            metadata.updated_at = datetime.now()
            
            # Update metadata cache and persist
            try:
                await self._update_metadata_cache(metadata)
            except Exception as e:
                # File was stored but metadata update failed - this is a warning
                logger.warning(f"File stored but metadata update failed for {metadata.file_id}: {e}")
                # Don't raise here as the file was successfully stored
            
            logger.debug(f"✅ Stored file {metadata.file_id} ({filename}) locally")
            return metadata.file_id
            
        except (StorageError, ValidationError):
            # Re-raise our structured errors
            raise
        except Exception as e:
            # Handle unexpected errors
            structured_error = error_handler.handle_exception(
                exception=e,
                operation="file_storage",
                filename=filename,
                context={
                    'file_id': metadata.file_id,
                    'backend': 'local',
                    'file_size': len(content) if content else 0
                }
            )
            logger.error(f"❌ Unexpected error storing file {filename}: {structured_error}")
            raise structured_error

    async def store_markdown_version(self, original_file_id: str, markdown_content: str) -> Optional[str]:
        """Store markdown version of a file as a separate file"""
        try:
            if original_file_id not in self._metadata_cache:
                logger.error(f"Original file {original_file_id} not found for markdown storage")
                return None
            
            original_metadata = self._metadata_cache[original_file_id]
            
            # Create new file ID for markdown version
            markdown_file_id = str(uuid.uuid4())
            
            # Create markdown filename with format: id_original_name.md
            base_name = Path(original_metadata.filename).stem
            markdown_filename = f"{markdown_file_id}_{base_name}.md"
            
            # Store markdown content as bytes
            markdown_bytes = markdown_content.encode('utf-8')
            markdown_file_path = self.files_dir / markdown_filename
            
            async with aiofiles.open(markdown_file_path, 'wb') as f:
                await f.write(markdown_bytes)
            
            # Create metadata for markdown file
            from datetime import datetime
            markdown_metadata = FileMetadata(
                file_id=markdown_file_id,
                filename=markdown_filename,
                mime_type="text/markdown",
                size_bytes=len(markdown_bytes),
                created_at=datetime.now(),
                updated_at=datetime.now(),
                user_id=original_metadata.user_id,
                session_id=original_metadata.session_id,
                agent_id=original_metadata.agent_id,
                is_generated=True,  # Markdown version is generated
                tags=original_metadata.tags + ["markdown-conversion", "auto-generated"],
                custom_metadata={
                    **original_metadata.custom_metadata,
                    "original_file_id": original_file_id,
                    "conversion_source": original_metadata.filename
                },
                storage_backend="local",
                storage_path=str(markdown_file_path),
                conversion_status="success",
                conversion_timestamp=datetime.now()
            )
            
            # Update metadata cache for markdown file
            await self._update_metadata_cache(markdown_metadata)
            
            # Update original file metadata to reference markdown version
            original_metadata.markdown_file_id = markdown_file_id
            original_metadata.markdown_content = markdown_content  # Keep for quick access
            original_metadata.conversion_status = "success"
            original_metadata.conversion_timestamp = datetime.now()
            original_metadata.conversion_error = None
            original_metadata.updated_at = datetime.now()
            
            # Save updated original metadata
            await self._update_metadata_cache(original_metadata)
            
            logger.info(f"✅ Stored markdown version of {original_metadata.filename} as separate file {markdown_file_id}")
            return markdown_file_id
            
        except Exception as e:
            logger.error(f"Failed to store markdown version for file {original_file_id}: {e}")
            return None

    async def retrieve_markdown_version(self, original_file_id: str) -> Optional[Tuple[str, 'FileMetadata']]:
        """Retrieve markdown version of a file"""
        try:
            if original_file_id not in self._metadata_cache:
                return None
            
            original_metadata = self._metadata_cache[original_file_id]
            
            if not original_metadata.markdown_file_id:
                return None
            
            # Retrieve markdown file
            markdown_content, markdown_metadata = await self.retrieve_file(original_metadata.markdown_file_id)
            markdown_text = markdown_content.decode('utf-8')
            
            return markdown_text, markdown_metadata
            
        except Exception as e:
            logger.error(f"Failed to retrieve markdown version for file {original_file_id}: {e}")
            return None
    
    async def retrieve_file(self, file_id: str) -> tuple[bytes, FileMetadata]:
        """Retrieve file content and metadata with comprehensive error handling"""
        error_handler = ErrorHandler()
        
        try:
            # Validate file_id
            if not file_id or not file_id.strip():
                raise ValidationError(
                    error_type=FileProcessingErrorType.FILENAME_INVALID,
                    severity=ErrorSeverity.ERROR,
                    message="Invalid file ID provided",
                    user_message="File ID is required for retrieval"
                )
            
            # Check metadata cache
            if file_id not in self._metadata_cache:
                raise StorageError(
                    error_type=FileProcessingErrorType.STORAGE_READ_FAILED,
                    severity=ErrorSeverity.ERROR,
                    message=f"File {file_id} not found in metadata",
                    user_message="File not found in storage system",
                    backend_name="local",
                    context={'file_id': file_id}
                )
            
            metadata = self._metadata_cache[file_id]
            file_path = Path(metadata.storage_path)
            
            # Check if file exists on disk
            if not file_path.exists():
                raise StorageError(
                    error_type=FileProcessingErrorType.STORAGE_READ_FAILED,
                    severity=ErrorSeverity.ERROR,
                    message=f"File {file_id} not found on disk at {file_path}",
                    user_message="File not found on storage disk - may have been deleted",
                    backend_name="local",
                    context={'file_id': file_id, 'expected_path': str(file_path)}
                )
            
            # Read file content with error handling
            try:
                async with aiofiles.open(file_path, 'rb') as f:
                    content = await f.read()
            except PermissionError as e:
                raise StorageError(
                    error_type=FileProcessingErrorType.STORAGE_PERMISSION_DENIED,
                    severity=ErrorSeverity.ERROR,
                    message=f"Permission denied reading {file_path}",
                    user_message="Storage permission denied - contact administrator",
                    backend_name="local",
                    technical_details=str(e),
                    context={'file_id': file_id, 'file_path': str(file_path)}
                )
            except OSError as e:
                if "corrupted" in str(e).lower() or "invalid" in str(e).lower():
                    raise StorageError(
                        error_type=FileProcessingErrorType.FILE_CORRUPTED,
                        severity=ErrorSeverity.ERROR,
                        message=f"File {file_id} appears to be corrupted",
                        user_message="File is corrupted and cannot be read",
                        backend_name="local",
                        technical_details=str(e),
                        context={'file_id': file_id}
                    )
                else:
                    raise StorageError(
                        error_type=FileProcessingErrorType.STORAGE_READ_FAILED,
                        severity=ErrorSeverity.ERROR,
                        message=f"Failed to read file: {str(e)}",
                        user_message="File read failed due to system error",
                        backend_name="local",
                        technical_details=str(e),
                        context={'file_id': file_id}
                    )
            
            # Validate content
            if content is None:
                raise StorageError(
                    error_type=FileProcessingErrorType.FILE_CORRUPTED,
                    severity=ErrorSeverity.ERROR,
                    message=f"File {file_id} returned null content",
                    user_message="File appears to be corrupted",
                    backend_name="local",
                    context={'file_id': file_id}
                )
            
            logger.debug(f"✅ Retrieved file {file_id} from local storage ({len(content)} bytes)")
            return content, metadata
            
        except (StorageError, ValidationError):
            # Re-raise our structured errors
            raise
        except Exception as e:
            # Handle unexpected errors
            structured_error = error_handler.handle_exception(
                exception=e,
                operation="file_retrieval",
                context={
                    'file_id': file_id,
                    'backend': 'local'
                }
            )
            logger.error(f"❌ Unexpected error retrieving file {file_id}: {structured_error}")
            raise structured_error
    
    async def delete_file(self, file_id: str) -> bool:
        """Delete file from storage and metadata"""
        try:
            if file_id not in self._metadata_cache:
                return False
            
            metadata = self._metadata_cache[file_id]
            file_path = Path(metadata.storage_path)
            
            # Delete file from disk
            if file_path.exists():
                file_path.unlink()
            
            # Remove from metadata cache
            del self._metadata_cache[file_id]
            await self._save_metadata()
            
            logger.debug(f"Deleted file {file_id} from local storage")
            return True
            
        except Exception as e:
            logger.error(f"Failed to delete file {file_id}: {e}")
            return False
    
    async def list_files(self, 
                        user_id: str, 
                        session_id: Optional[str] = None,
                        agent_id: Optional[str] = None,
                        is_generated: Optional[bool] = None) -> List[FileMetadata]:
        """List files with filtering"""
        try:
            logger.info(f"🔍 LOCAL STORAGE - Listing files with filters: user_id={user_id}, session_id={session_id}, agent_id={agent_id}, is_generated={is_generated}")
            
            # Always refresh metadata from disk to get latest files
            await self._load_metadata()
            
            logger.info(f"📊 LOCAL STORAGE - Total files in cache after refresh: {len(self._metadata_cache)}")
            
            files = []
            for metadata in self._metadata_cache.values():
                logger.info(f"🔍 Checking file: {metadata.filename} (user={metadata.user_id}, session={metadata.session_id}, generated={metadata.is_generated})")
                
                # Filter by user_id
                if metadata.user_id != user_id:
                    logger.info(f"❌ User mismatch: {metadata.user_id} != {user_id}")
                    continue
                
                # Filter by session_id if specified
                if session_id is not None and metadata.session_id != session_id:
                    logger.info(f"❌ Session mismatch: {metadata.session_id} != {session_id}")
                    continue
                
                # Filter by agent_id if specified
                if agent_id is not None and metadata.agent_id != agent_id:
                    logger.info(f"❌ Agent mismatch: {metadata.agent_id} != {agent_id}")
                    continue
                
                # Filter by is_generated if specified
                if is_generated is not None and metadata.is_generated != is_generated:
                    logger.info(f"❌ Generated mismatch: {metadata.is_generated} != {is_generated}")
                    continue
                
                logger.info(f"✅ File passed all filters: {metadata.filename}")
                files.append(metadata)
            
            # Sort by created_at descending
            files.sort(key=lambda x: x.created_at, reverse=True)
            logger.info(f"📁 LOCAL STORAGE - Returning {len(files)} files after filtering and sorting")
            return files
            
        except Exception as e:
            logger.error(f"Failed to list files: {e}")
            return []
    
    async def update_metadata(self, file_id: str, metadata_updates: Dict[str, Any]) -> bool:
        """Update file metadata"""
        try:
            if file_id not in self._metadata_cache:
                return False
            
            metadata = self._metadata_cache[file_id]
            
            # Update allowed fields
            for key, value in metadata_updates.items():
                if hasattr(metadata, key) and key not in ['file_id', 'created_at', 'storage_path', 'storage_backend']:
                    setattr(metadata, key, value)
            
            metadata.updated_at = datetime.now()
            await self._save_metadata()
            
            logger.debug(f"Updated metadata for file {file_id}")
            return True
            
        except Exception as e:
            logger.error(f"Failed to update metadata for file {file_id}: {e}")
            return False
    
    async def file_exists(self, file_id: str) -> bool:
        """Check if file exists"""
        if file_id not in self._metadata_cache:
            return False
        
        metadata = self._metadata_cache[file_id]
        file_path = Path(metadata.storage_path)
        return file_path.exists()
    
    async def get_file_metadata(self, file_id: str) -> Optional[FileMetadata]:
        """Get file metadata without content"""
        logger.info(f"🔍 GET METADATA - Looking for file {file_id}")
        
        # First check cache
        if file_id in self._metadata_cache:
            logger.info(f"✅ GET METADATA - Found file {file_id} in cache")
            return self._metadata_cache[file_id]
        
        logger.info(f"❌ GET METADATA - File {file_id} not in cache, reloading from disk")
        
        # If not in cache, reload metadata from disk
        await self._load_metadata()
        
        if file_id in self._metadata_cache:
            logger.info(f"✅ GET METADATA - Found file {file_id} after reload")
            return self._metadata_cache[file_id]
        else:
            logger.error(f"❌ GET METADATA - File {file_id} still not found after reload")
            return None
    
    async def convert_file_to_markdown(self, file_id: str) -> Optional[str]:
        """Convert file to markdown and return the content"""
        try:
            # Get file content and metadata
            content, metadata = await self.retrieve_file(file_id)
            
            # Import here to avoid circular import
            from agent_framework.processing.markdown_converter import markdown_converter
            
            # Convert to markdown
            markdown_content = await markdown_converter.convert_to_markdown(
                content, metadata.filename, metadata.mime_type or ""
            )
            
            if markdown_content:
                # Update metadata with conversion results
                metadata.markdown_content = markdown_content
                metadata.conversion_status = "success"
                metadata.conversion_timestamp = datetime.now()
                metadata.conversion_error = None
                
                # Save updated metadata
                await self._save_metadata()
                
                logger.info(f"✅ Converted file {file_id} to markdown")
                return markdown_content
            else:
                # Update metadata with failure
                metadata.conversion_status = "failed"
                metadata.conversion_timestamp = datetime.now()
                metadata.conversion_error = "Conversion returned empty content"
                
                # Save updated metadata
                await self._save_metadata()
                
                logger.warning(f"❌ Failed to convert file {file_id} to markdown")
                return None
                
        except Exception as e:
            logger.error(f"❌ Error converting file {file_id} to markdown: {e}")
            
            # Update metadata with error
            if file_id in self._metadata_cache:
                metadata = self._metadata_cache[file_id]
                metadata.conversion_status = "failed"
                metadata.conversion_timestamp = datetime.now()
                metadata.conversion_error = str(e)
                await self._save_metadata()
            
            return None
    
    async def analyze_image(self, file_id: str) -> Optional[Dict[str, Any]]:
        """Analyze image content using multimodal capabilities"""
        try:
            if file_id not in self._metadata_cache:
                logger.warning(f"File {file_id} not found for image analysis")
                return None
            
            metadata = self._metadata_cache[file_id]
            
            # Check if file has visual content
            if not metadata.has_visual_content and not (metadata.mime_type and metadata.mime_type.startswith('image/')):
                logger.warning(f"File {file_id} does not contain visual content")
                return None
            
            # For now, return a placeholder result indicating analysis capability
            # This will be implemented in later tasks with actual multimodal processing
            analysis_result = {
                "status": "not_implemented",
                "message": "Image analysis capability will be implemented in task 4",
                "file_id": file_id,
                "filename": metadata.filename,
                "mime_type": metadata.mime_type,
                "has_visual_content": metadata.has_visual_content
            }
            
            # Update metadata with analysis attempt
            metadata.multimodal_processing_status = "not_implemented"
            await self._save_metadata()
            
            logger.debug(f"Image analysis placeholder for file {file_id}")
            return analysis_result
            
        except Exception as e:
            logger.error(f"Error in image analysis for file {file_id}: {e}")
            return None
    
    async def get_processing_status(self, file_id: str) -> Dict[str, Any]:
        """Get comprehensive processing status for a file"""
        try:
            if file_id not in self._metadata_cache:
                return {
                    "file_id": file_id,
                    "exists": False,
                    "error": "File not found"
                }
            
            metadata = self._metadata_cache[file_id]
            
            # Compile comprehensive processing status
            status = {
                "file_id": file_id,
                "exists": True,
                "filename": metadata.filename,
                "mime_type": metadata.mime_type,
                "size_bytes": metadata.size_bytes,
                "storage_backend": metadata.storage_backend,
                "created_at": metadata.created_at.isoformat(),
                "updated_at": metadata.updated_at.isoformat(),
                
                # Conversion status
                "conversion_status": metadata.conversion_status,
                "conversion_timestamp": metadata.conversion_timestamp.isoformat() if metadata.conversion_timestamp else None,
                "conversion_error": metadata.conversion_error,
                "has_markdown_version": metadata.markdown_file_id is not None,
                "markdown_file_id": metadata.markdown_file_id,
                
                # Multimodal processing status
                "has_visual_content": metadata.has_visual_content,
                "multimodal_processing_status": metadata.multimodal_processing_status,
                "image_analysis_available": metadata.image_analysis_result is not None,
                
                # Processing errors and warnings
                "processing_errors": metadata.processing_errors,
                "processing_warnings": metadata.processing_warnings,
                "total_processing_time_ms": metadata.total_processing_time_ms,
                
                # AI generation info
                "is_generated": metadata.is_generated,
                "generation_model": metadata.generation_model,
                "generation_prompt": metadata.generation_prompt,
                "generation_parameters": metadata.generation_parameters,
                
                # Tags and metadata
                "tags": metadata.tags,
                "custom_metadata": metadata.custom_metadata
            }
            
            logger.debug(f"Retrieved processing status for file {file_id}")
            return status
            
        except Exception as e:
            logger.error(f"Error getting processing status for file {file_id}: {e}")
            return {
                "file_id": file_id,
                "exists": False,
                "error": str(e)
            }
    
    async def _load_metadata(self):
        """Load metadata from JSON file"""
        try:
            logger.info(f"📂 Loading metadata from: {self.metadata_file}")
            if not self.metadata_file.exists():
                logger.warning(f"❌ Metadata file does not exist: {self.metadata_file}")
                self._metadata_cache = {}
                return
            
            logger.info(f"📖 Reading metadata file...")
            async with aiofiles.open(self.metadata_file, 'r') as f:
                content = await f.read()
                metadata_dict = json.loads(content)
            
            logger.info(f"📊 Found {len(metadata_dict)} files in metadata.json")
            
            # Convert dict to FileMetadata objects
            self._metadata_cache = {}
            successful_loads = 0
            for file_id, data in metadata_dict.items():
                try:
                    # Parse datetime strings
                    data['created_at'] = datetime.fromisoformat(data['created_at'])
                    data['updated_at'] = datetime.fromisoformat(data['updated_at'])
                    
                    # Handle optional datetime fields
                    if 'conversion_timestamp' in data and data['conversion_timestamp']:
                        data['conversion_timestamp'] = datetime.fromisoformat(data['conversion_timestamp'])
                    
                    # Provide defaults for new fields to maintain backward compatibility
                    data.setdefault('markdown_file_id', None)
                    data.setdefault('has_visual_content', False)
                    data.setdefault('image_analysis_result', None)
                    data.setdefault('multimodal_processing_status', 'not_attempted')
                    data.setdefault('processing_errors', [])
                    data.setdefault('processing_warnings', [])
                    data.setdefault('total_processing_time_ms', None)
                    data.setdefault('generation_model', None)
                    data.setdefault('generation_prompt', None)
                    data.setdefault('generation_parameters', None)
                    
                    self._metadata_cache[file_id] = FileMetadata(**data)
                    successful_loads += 1
                    
                except Exception as e:
                    logger.error(f"❌ Failed to load metadata for file {file_id}: {e}")
                    continue
            
            logger.info(f"✅ Successfully loaded metadata for {successful_loads}/{len(metadata_dict)} files")
            logger.info(f"📁 Cache now contains {len(self._metadata_cache)} files")
            
        except Exception as e:
            logger.error(f"❌ Failed to load metadata, starting with empty cache: {e}")
            self._metadata_cache = {}
    
    async def _save_metadata(self):
        """Save metadata to JSON file with atomic write and backup protection"""
        # Use lock to prevent concurrent writes that could corrupt the file
        async with self._metadata_lock:
            try:
                # Convert FileMetadata objects to dict
                metadata_dict = {}
                for file_id, metadata in self._metadata_cache.items():
                    data = {
                        'file_id': metadata.file_id,
                        'filename': metadata.filename,
                        'mime_type': metadata.mime_type,
                        'size_bytes': metadata.size_bytes,
                        'created_at': metadata.created_at.isoformat(),
                        'updated_at': metadata.updated_at.isoformat(),
                        'user_id': metadata.user_id,
                        'session_id': metadata.session_id,
                        'agent_id': metadata.agent_id,
                        'is_generated': metadata.is_generated,
                        'tags': metadata.tags,
                        'custom_metadata': metadata.custom_metadata,
                        'storage_backend': metadata.storage_backend,
                        'storage_path': metadata.storage_path,
                        # Enhanced markdown conversion fields
                        'markdown_content': metadata.markdown_content,
                        'markdown_file_id': metadata.markdown_file_id,
                        'conversion_status': metadata.conversion_status,
                        'conversion_timestamp': metadata.conversion_timestamp.isoformat() if metadata.conversion_timestamp else None,
                        'conversion_error': metadata.conversion_error,
                        # Multimodal processing fields
                        'has_visual_content': metadata.has_visual_content,
                        'image_analysis_result': metadata.image_analysis_result,
                        'multimodal_processing_status': metadata.multimodal_processing_status,
                        # Enhanced processing status tracking
                        'processing_errors': metadata.processing_errors,
                        'processing_warnings': metadata.processing_warnings,
                        'total_processing_time_ms': metadata.total_processing_time_ms,
                        # AI generation tracking fields
                        'generation_model': metadata.generation_model,
                        'generation_prompt': metadata.generation_prompt,
                        'generation_parameters': metadata.generation_parameters
                    }
                    metadata_dict[file_id] = data
                
                # Serialize to JSON string
                json_content = json.dumps(metadata_dict, indent=2)
                
                # Validate JSON is parseable before writing
                try:
                    json.loads(json_content)
                except json.JSONDecodeError as e:
                    logger.error(f"❌ Generated invalid JSON before write: {e}")
                    raise ValueError(f"Invalid JSON generated: {e}")
                
                # Create backup of existing file if it exists
                backup_file = self.metadata_file.with_suffix('.json.backup')
                if self.metadata_file.exists():
                    try:
                        # Copy existing file to backup
                        async with aiofiles.open(self.metadata_file, 'r') as src:
                            backup_content = await src.read()
                        async with aiofiles.open(backup_file, 'w') as dst:
                            await dst.write(backup_content)
                        logger.debug(f"📦 Created backup: {backup_file}")
                    except Exception as backup_error:
                        logger.warning(f"⚠️ Failed to create backup: {backup_error}")
                        # Continue anyway - backup is nice to have but not critical
                
                # Write to temporary file first (atomic write pattern)
                temp_file = self.metadata_file.with_suffix('.json.tmp')
                try:
                    async with aiofiles.open(temp_file, 'w') as f:
                        await f.write(json_content)
                    
                    # Validate the written file
                    async with aiofiles.open(temp_file, 'r') as f:
                        validation_content = await f.read()
                        json.loads(validation_content)  # Will raise if invalid
                    
                    # Atomic rename (replaces existing file)
                    temp_file.replace(self.metadata_file)
                    logger.debug(f"✅ Metadata saved successfully ({len(metadata_dict)} files)")
                    
                    # Clean up old backup if write was successful
                    if backup_file.exists():
                        try:
                            backup_file.unlink()
                        except Exception:
                            pass  # Ignore cleanup errors
                    
                except json.JSONDecodeError as e:
                    logger.error(f"❌ Written file failed validation: {e}")
                    # Restore from backup if available
                    if backup_file.exists():
                        try:
                            async with aiofiles.open(backup_file, 'r') as src:
                                restore_content = await src.read()
                            async with aiofiles.open(self.metadata_file, 'w') as dst:
                                await dst.write(restore_content)
                            logger.info(f"♻️ Restored metadata from backup")
                        except Exception as restore_error:
                            logger.error(f"❌ Failed to restore from backup: {restore_error}")
                    raise ValueError(f"Metadata file validation failed: {e}")
                finally:
                    # Clean up temp file if it still exists
                    if temp_file.exists():
                        try:
                            temp_file.unlink()
                        except Exception:
                            pass
                
            except Exception as e:
                logger.error(f"❌ Failed to save metadata: {e}")
                raise
    
    async def _update_metadata_cache(self, metadata: FileMetadata):
        """Update metadata cache and persist"""
        self._metadata_cache[metadata.file_id] = metadata
        await self._save_metadata()


# ===== S3 FILE STORAGE IMPLEMENTATION =====

# S3 availability flag
S3_AVAILABLE = False
try:
    import boto3
    from botocore.exceptions import ClientError, NoCredentialsError
    S3_AVAILABLE = True
except ImportError:
    logger.warning("S3 storage not available: boto3 not installed. Install with 'pip install boto3'")


class S3FileStorage(FileStorageInterface):
    """AWS S3 storage implementation"""
    
    def __init__(self, bucket: str, region: str = "us-east-1", prefix: str = "agent-files/"):
        if not S3_AVAILABLE:
            raise ImportError("S3 storage requires boto3. Install with 'pip install boto3'")
        
        self.bucket = bucket
        self.region = region
        self.prefix = prefix
        self.s3_client = None
        self.metadata_key = f"{prefix}metadata.json"
        self._metadata_cache: Dict[str, FileMetadata] = {}
    
    async def initialize(self) -> bool:
        """Initialize S3 client and verify bucket access"""
        try:
            self.s3_client = boto3.client('s3', region_name=self.region)
            
            # Test bucket access
            self.s3_client.head_bucket(Bucket=self.bucket)
            
            # Load existing metadata
            await self._load_metadata()
            
            logger.info(f"Initialized S3FileStorage for bucket {self.bucket}")
            return True
            
        except (ClientError, NoCredentialsError) as e:
            logger.error(f"Failed to initialize S3 storage: {e}")
            return False
        except Exception as e:
            logger.error(f"Unexpected error initializing S3 storage: {e}")
            return False
    
    async def store_file(self, content: bytes, filename: str, metadata: FileMetadata) -> str:
        """Store file in S3"""
        try:
            # Create storage filename with format: id_original_name.ext
            original_name = Path(filename)
            storage_filename = f"{metadata.file_id}_{original_name.stem}{original_name.suffix}"
            key = f"{self.prefix}{storage_filename}"
            
            # Store file in S3 with metadata
            self.s3_client.put_object(
                Bucket=self.bucket,
                Key=key,
                Body=content,
                ContentType=metadata.mime_type or 'application/octet-stream',
                Metadata={
                    'filename': filename,
                    'mime_type': metadata.mime_type or '',
                    'user_id': metadata.user_id,
                    'session_id': metadata.session_id or '',
                    'agent_id': metadata.agent_id or '',
                    'is_generated': str(metadata.is_generated),
                    'created_at': metadata.created_at.isoformat(),
                    'file_id': metadata.file_id
                }
            )
            
            # Update metadata
            metadata.storage_path = key
            metadata.size_bytes = len(content)
            metadata.storage_backend = "s3"
            metadata.updated_at = datetime.now()
            
            # Update metadata cache and persist
            await self._update_metadata_cache(metadata)
            
            logger.debug(f"Stored file {metadata.file_id} ({filename}) in S3")
            return metadata.file_id
            
        except Exception as e:
            logger.error(f"Failed to store file {metadata.file_id} in S3: {e}")
            raise

    async def store_markdown_version(self, original_file_id: str, markdown_content: str) -> Optional[str]:
        """Store markdown version of a file as a separate file in S3"""
        try:
            if original_file_id not in self._metadata_cache:
                logger.error(f"Original file {original_file_id} not found for markdown storage")
                return None
            
            original_metadata = self._metadata_cache[original_file_id]
            
            # Create new file ID for markdown version
            markdown_file_id = str(uuid.uuid4())
            
            # Create markdown filename with format: id_original_name.md
            base_name = Path(original_metadata.filename).stem
            markdown_filename = f"{markdown_file_id}_{base_name}.md"
            
            # Store markdown content as bytes
            markdown_bytes = markdown_content.encode('utf-8')
            markdown_key = f"{self.prefix}{markdown_filename}"
            
            # Store markdown file in S3
            self.s3_client.put_object(
                Bucket=self.bucket,
                Key=markdown_key,
                Body=markdown_bytes,
                ContentType="text/markdown",
                Metadata={
                    'filename': markdown_filename,
                    'mime_type': 'text/markdown',
                    'user_id': original_metadata.user_id,
                    'session_id': original_metadata.session_id or '',
                    'agent_id': original_metadata.agent_id or '',
                    'is_generated': 'True',
                    'created_at': datetime.now().isoformat(),
                    'file_id': markdown_file_id,
                    'original_file_id': original_file_id
                }
            )
            
            # Create metadata for markdown file
            from datetime import datetime
            markdown_metadata = FileMetadata(
                file_id=markdown_file_id,
                filename=markdown_filename,
                mime_type="text/markdown",
                size_bytes=len(markdown_bytes),
                created_at=datetime.now(),
                updated_at=datetime.now(),
                user_id=original_metadata.user_id,
                session_id=original_metadata.session_id,
                agent_id=original_metadata.agent_id,
                is_generated=True,  # Markdown version is generated
                tags=original_metadata.tags + ["markdown-conversion", "auto-generated"],
                custom_metadata={
                    **original_metadata.custom_metadata,
                    "original_file_id": original_file_id,
                    "conversion_source": original_metadata.filename
                },
                storage_backend="s3",
                storage_path=markdown_key,
                conversion_status="success",
                conversion_timestamp=datetime.now()
            )
            
            # Update metadata cache for markdown file
            await self._update_metadata_cache(markdown_metadata)
            
            # Update original file metadata to reference markdown version
            original_metadata.markdown_file_id = markdown_file_id
            original_metadata.markdown_content = markdown_content  # Keep for quick access
            original_metadata.conversion_status = "success"
            original_metadata.conversion_timestamp = datetime.now()
            original_metadata.conversion_error = None
            original_metadata.updated_at = datetime.now()
            
            # Save updated original metadata
            await self._update_metadata_cache(original_metadata)
            
            logger.info(f"✅ Stored markdown version of {original_metadata.filename} as separate file {markdown_file_id} in S3")
            return markdown_file_id
            
        except Exception as e:
            logger.error(f"Failed to store markdown version for file {original_file_id} in S3: {e}")
            return None

    async def retrieve_markdown_version(self, original_file_id: str) -> Optional[Tuple[str, 'FileMetadata']]:
        """Retrieve markdown version of a file from S3"""
        try:
            if original_file_id not in self._metadata_cache:
                return None
            
            original_metadata = self._metadata_cache[original_file_id]
            
            if not original_metadata.markdown_file_id:
                return None
            
            # Retrieve markdown file
            markdown_content, markdown_metadata = await self.retrieve_file(original_metadata.markdown_file_id)
            markdown_text = markdown_content.decode('utf-8')
            
            return markdown_text, markdown_metadata
            
        except Exception as e:
            logger.error(f"Failed to retrieve markdown version for file {original_file_id} from S3: {e}")
            return None
    
    async def retrieve_file(self, file_id: str) -> tuple[bytes, FileMetadata]:
        """Retrieve file content and metadata from S3"""
        try:
            if file_id not in self._metadata_cache:
                raise FileNotFoundError(f"File {file_id} not found")
            
            metadata = self._metadata_cache[file_id]
            
            # Get file from S3
            response = self.s3_client.get_object(
                Bucket=self.bucket,
                Key=metadata.storage_path
            )
            
            content = response['Body'].read()
            
            logger.debug(f"Retrieved file {file_id} from S3")
            return content, metadata
            
        except ClientError as e:
            if e.response['Error']['Code'] == 'NoSuchKey':
                raise FileNotFoundError(f"File {file_id} not found in S3")
            else:
                logger.error(f"Failed to retrieve file {file_id} from S3: {e}")
                raise
        except Exception as e:
            logger.error(f"Failed to retrieve file {file_id} from S3: {e}")
            raise
    
    async def delete_file(self, file_id: str) -> bool:
        """Delete file from S3 and metadata"""
        try:
            if file_id not in self._metadata_cache:
                return False
            
            metadata = self._metadata_cache[file_id]
            
            # Delete file from S3
            self.s3_client.delete_object(
                Bucket=self.bucket,
                Key=metadata.storage_path
            )
            
            # Remove from metadata cache
            del self._metadata_cache[file_id]
            await self._save_metadata()
            
            logger.debug(f"Deleted file {file_id} from S3")
            return True
            
        except Exception as e:
            logger.error(f"Failed to delete file {file_id} from S3: {e}")
            return False
    
    async def list_files(self, 
                        user_id: str, 
                        session_id: Optional[str] = None,
                        agent_id: Optional[str] = None,
                        is_generated: Optional[bool] = None) -> List[FileMetadata]:
        """List files with filtering"""
        try:
            files = []
            for metadata in self._metadata_cache.values():
                # Filter by user_id
                if metadata.user_id != user_id:
                    continue
                
                # Filter by session_id if specified
                if session_id is not None and metadata.session_id != session_id:
                    continue
                
                # Filter by agent_id if specified
                if agent_id is not None and metadata.agent_id != agent_id:
                    continue
                
                # Filter by is_generated if specified
                if is_generated is not None and metadata.is_generated != is_generated:
                    continue
                
                files.append(metadata)
            
            # Sort by created_at descending
            files.sort(key=lambda x: x.created_at, reverse=True)
            return files
            
        except Exception as e:
            logger.error(f"Failed to list files from S3: {e}")
            return []
    
    async def update_metadata(self, file_id: str, metadata_updates: Dict[str, Any]) -> bool:
        """Update file metadata"""
        try:
            if file_id not in self._metadata_cache:
                return False
            
            metadata = self._metadata_cache[file_id]
            
            # Update allowed fields
            for key, value in metadata_updates.items():
                if hasattr(metadata, key) and key not in ['file_id', 'created_at', 'storage_path', 'storage_backend']:
                    setattr(metadata, key, value)
            
            metadata.updated_at = datetime.now()
            await self._save_metadata()
            
            logger.debug(f"Updated metadata for file {file_id} in S3")
            return True
            
        except Exception as e:
            logger.error(f"Failed to update metadata for file {file_id} in S3: {e}")
            return False
    
    async def file_exists(self, file_id: str) -> bool:
        """Check if file exists in S3"""
        if file_id not in self._metadata_cache:
            return False
        
        metadata = self._metadata_cache[file_id]
        
        try:
            self.s3_client.head_object(
                Bucket=self.bucket,
                Key=metadata.storage_path
            )
            return True
        except ClientError as e:
            if e.response['Error']['Code'] == '404':
                return False
            else:
                logger.error(f"Error checking file existence in S3: {e}")
                return False
    
    async def get_file_metadata(self, file_id: str) -> Optional[FileMetadata]:
        """Get file metadata without content"""
        return self._metadata_cache.get(file_id)
    
    async def convert_file_to_markdown(self, file_id: str) -> Optional[str]:
        """Convert file to markdown and return the content"""
        try:
            # Get file content and metadata
            content, metadata = await self.retrieve_file(file_id)
            
            # Import here to avoid circular import
            from agent_framework.processing.markdown_converter import markdown_converter
            
            # Convert to markdown
            markdown_content = await markdown_converter.convert_to_markdown(
                content, metadata.filename, metadata.mime_type or ""
            )
            
            if markdown_content:
                # Update metadata with conversion results
                metadata.markdown_content = markdown_content
                metadata.conversion_status = "success"
                metadata.conversion_timestamp = datetime.now()
                metadata.conversion_error = None
                
                # Save updated metadata
                await self._save_metadata()
                
                logger.info(f"✅ Converted file {file_id} to markdown")
                return markdown_content
            else:
                # Update metadata with failure
                metadata.conversion_status = "failed"
                metadata.conversion_timestamp = datetime.now()
                metadata.conversion_error = "Conversion returned empty content"
                
                # Save updated metadata
                await self._save_metadata()
                
                logger.warning(f"❌ Failed to convert file {file_id} to markdown")
                return None
                
        except Exception as e:
            logger.error(f"❌ Error converting file {file_id} to markdown: {e}")
            
            # Update metadata with error
            if file_id in self._metadata_cache:
                metadata = self._metadata_cache[file_id]
                metadata.conversion_status = "failed"
                metadata.conversion_timestamp = datetime.now()
                metadata.conversion_error = str(e)
                await self._save_metadata()
            
            return None
    
    async def analyze_image(self, file_id: str) -> Optional[Dict[str, Any]]:
        """Analyze image content using multimodal capabilities"""
        try:
            if file_id not in self._metadata_cache:
                logger.warning(f"File {file_id} not found for image analysis")
                return None
            
            metadata = self._metadata_cache[file_id]
            
            # Check if file has visual content
            if not metadata.has_visual_content and not (metadata.mime_type and metadata.mime_type.startswith('image/')):
                logger.warning(f"File {file_id} does not contain visual content")
                return None
            
            # For now, return a placeholder result indicating analysis capability
            # This will be implemented in later tasks with actual multimodal processing
            analysis_result = {
                "status": "not_implemented",
                "message": "Image analysis capability will be implemented in task 4",
                "file_id": file_id,
                "filename": metadata.filename,
                "mime_type": metadata.mime_type,
                "has_visual_content": metadata.has_visual_content
            }
            
            # Update metadata with analysis attempt
            metadata.multimodal_processing_status = "not_implemented"
            await self._save_metadata()
            
            logger.debug(f"Image analysis placeholder for file {file_id}")
            return analysis_result
            
        except Exception as e:
            logger.error(f"Error in image analysis for file {file_id}: {e}")
            return None
    
    async def get_processing_status(self, file_id: str) -> Dict[str, Any]:
        """Get comprehensive processing status for a file"""
        try:
            if file_id not in self._metadata_cache:
                return {
                    "file_id": file_id,
                    "exists": False,
                    "error": "File not found"
                }
            
            metadata = self._metadata_cache[file_id]
            
            # Compile comprehensive processing status
            status = {
                "file_id": file_id,
                "exists": True,
                "filename": metadata.filename,
                "mime_type": metadata.mime_type,
                "size_bytes": metadata.size_bytes,
                "storage_backend": metadata.storage_backend,
                "created_at": metadata.created_at.isoformat(),
                "updated_at": metadata.updated_at.isoformat(),
                
                # Conversion status
                "conversion_status": metadata.conversion_status,
                "conversion_timestamp": metadata.conversion_timestamp.isoformat() if metadata.conversion_timestamp else None,
                "conversion_error": metadata.conversion_error,
                "has_markdown_version": metadata.markdown_file_id is not None,
                "markdown_file_id": metadata.markdown_file_id,
                
                # Multimodal processing status
                "has_visual_content": metadata.has_visual_content,
                "multimodal_processing_status": metadata.multimodal_processing_status,
                "image_analysis_available": metadata.image_analysis_result is not None,
                
                # Processing errors and warnings
                "processing_errors": metadata.processing_errors,
                "processing_warnings": metadata.processing_warnings,
                "total_processing_time_ms": metadata.total_processing_time_ms,
                
                # AI generation info
                "is_generated": metadata.is_generated,
                "generation_model": metadata.generation_model,
                "generation_prompt": metadata.generation_prompt,
                "generation_parameters": metadata.generation_parameters,
                
                # Tags and metadata
                "tags": metadata.tags,
                "custom_metadata": metadata.custom_metadata
            }
            
            logger.debug(f"Retrieved processing status for file {file_id}")
            return status
            
        except Exception as e:
            logger.error(f"Error getting processing status for file {file_id}: {e}")
            return {
                "file_id": file_id,
                "exists": False,
                "error": str(e)
            }
    
    async def _load_metadata(self):
        """Load metadata from S3 manifest"""
        try:
            response = self.s3_client.get_object(
                Bucket=self.bucket,
                Key=self.metadata_key
            )
            
            content = response['Body'].read().decode('utf-8')
            metadata_dict = json.loads(content)
            
            # Convert dict to FileMetadata objects
            self._metadata_cache = {}
            for file_id, data in metadata_dict.items():
                # Parse datetime strings
                data['created_at'] = datetime.fromisoformat(data['created_at'])
                data['updated_at'] = datetime.fromisoformat(data['updated_at'])
                
                # Handle optional datetime fields
                if 'conversion_timestamp' in data and data['conversion_timestamp']:
                    data['conversion_timestamp'] = datetime.fromisoformat(data['conversion_timestamp'])
                
                # Provide defaults for new fields to maintain backward compatibility
                data.setdefault('markdown_file_id', None)
                data.setdefault('has_visual_content', False)
                data.setdefault('image_analysis_result', None)
                data.setdefault('multimodal_processing_status', 'not_attempted')
                data.setdefault('processing_errors', [])
                data.setdefault('processing_warnings', [])
                data.setdefault('total_processing_time_ms', None)
                data.setdefault('generation_model', None)
                data.setdefault('generation_prompt', None)
                data.setdefault('generation_parameters', None)
                
                self._metadata_cache[file_id] = FileMetadata(**data)
            
            logger.debug(f"Loaded metadata for {len(self._metadata_cache)} files from S3")
            
        except ClientError as e:
            if e.response['Error']['Code'] == 'NoSuchKey':
                logger.debug("No existing metadata found in S3, starting with empty cache")
                self._metadata_cache = {}
            else:
                logger.warning(f"Failed to load metadata from S3: {e}")
                self._metadata_cache = {}
        except Exception as e:
            logger.warning(f"Failed to load metadata from S3: {e}")
            self._metadata_cache = {}
    
    async def _save_metadata(self):
        """Save metadata to S3 manifest"""
        try:
            # Convert FileMetadata objects to dict
            metadata_dict = {}
            for file_id, metadata in self._metadata_cache.items():
                data = {
                    'file_id': metadata.file_id,
                    'filename': metadata.filename,
                    'mime_type': metadata.mime_type,
                    'size_bytes': metadata.size_bytes,
                    'created_at': metadata.created_at.isoformat(),
                    'updated_at': metadata.updated_at.isoformat(),
                    'user_id': metadata.user_id,
                    'session_id': metadata.session_id,
                    'agent_id': metadata.agent_id,
                    'is_generated': metadata.is_generated,
                    'tags': metadata.tags,
                    'custom_metadata': metadata.custom_metadata,
                    'storage_backend': metadata.storage_backend,
                    'storage_path': metadata.storage_path,
                    # Enhanced markdown conversion fields
                    'markdown_content': metadata.markdown_content,
                    'markdown_file_id': metadata.markdown_file_id,
                    'conversion_status': metadata.conversion_status,
                    'conversion_timestamp': metadata.conversion_timestamp.isoformat() if metadata.conversion_timestamp else None,
                    'conversion_error': metadata.conversion_error,
                    # Multimodal processing fields
                    'has_visual_content': metadata.has_visual_content,
                    'image_analysis_result': metadata.image_analysis_result,
                    'multimodal_processing_status': metadata.multimodal_processing_status,
                    # Enhanced processing status tracking
                    'processing_errors': metadata.processing_errors,
                    'processing_warnings': metadata.processing_warnings,
                    'total_processing_time_ms': metadata.total_processing_time_ms,
                    # AI generation tracking fields
                    'generation_model': metadata.generation_model,
                    'generation_prompt': metadata.generation_prompt,
                    'generation_parameters': metadata.generation_parameters
                }
                metadata_dict[file_id] = data
            
            # Save to S3
            self.s3_client.put_object(
                Bucket=self.bucket,
                Key=self.metadata_key,
                Body=json.dumps(metadata_dict, indent=2),
                ContentType='application/json'
            )
            
        except Exception as e:
            logger.error(f"Failed to save metadata to S3: {e}")
            raise
    
    async def _update_metadata_cache(self, metadata: FileMetadata):
        """Update metadata cache and persist"""
        self._metadata_cache[metadata.file_id] = metadata
        await self._save_metadata()


# ===== MINIO FILE STORAGE IMPLEMENTATION =====

# MinIO availability flag
MINIO_AVAILABLE = False
try:
    from minio import Minio
    from minio.error import S3Error
    MINIO_AVAILABLE = True
except ImportError:
    logger.warning("MinIO storage not available: minio not installed. Install with 'pip install minio'")


class MinIOFileStorage(FileStorageInterface):
    """MinIO storage implementation"""
    
    def __init__(self, endpoint: str, access_key: str, secret_key: str, 
                 bucket: str, secure: bool = True, prefix: str = "agent-files/"):
        if not MINIO_AVAILABLE:
            raise ImportError("MinIO storage requires minio package. Install with 'pip install minio'")
        
        self.endpoint = endpoint
        self.access_key = access_key  
        self.secret_key = secret_key
        self.bucket = bucket
        self.secure = secure
        self.prefix = prefix
        self.client = None
        self.metadata_key = f"{prefix}metadata.json"
        self._metadata_cache: Dict[str, FileMetadata] = {}
        
    async def initialize(self) -> bool:
        """Initialize MinIO client"""
        try:
            self.client = Minio(
                self.endpoint,
                access_key=self.access_key,
                secret_key=self.secret_key,
                secure=self.secure
            )
            
            # Ensure bucket exists
            if not self.client.bucket_exists(self.bucket):
                self.client.make_bucket(self.bucket)
                logger.info(f"Created MinIO bucket {self.bucket}")
            
            # Load existing metadata
            await self._load_metadata()
            
            logger.info(f"Initialized MinIOFileStorage for bucket {self.bucket} on {self.endpoint}")
            return True
            
        except Exception as e:
            logger.error(f"Failed to initialize MinIO storage: {e}")
            return False
    
    async def store_file(self, content: bytes, filename: str, metadata: FileMetadata) -> str:
        """Store file in MinIO"""
        try:
            # Create storage filename with format: id_original_name.ext
            original_name = Path(filename)
            storage_filename = f"{metadata.file_id}_{original_name.stem}{original_name.suffix}"
            key = f"{self.prefix}{storage_filename}"
            
            # Prepare metadata for MinIO
            minio_metadata = {
                'filename': filename,
                'mime-type': metadata.mime_type or 'application/octet-stream',
                'user-id': metadata.user_id,
                'session-id': metadata.session_id or '',
                'agent-id': metadata.agent_id or '',
                'is-generated': str(metadata.is_generated),
                'created-at': metadata.created_at.isoformat(),
                'file-id': metadata.file_id
            }
            
            # Store file in MinIO
            self.client.put_object(
                bucket_name=self.bucket,
                object_name=key,
                data=BytesIO(content),
                length=len(content),
                content_type=metadata.mime_type or 'application/octet-stream',
                metadata=minio_metadata
            )
            
            # Update metadata
            metadata.storage_path = key
            metadata.size_bytes = len(content)
            metadata.storage_backend = "minio"
            metadata.updated_at = datetime.now()
            
            # Update metadata cache and persist
            await self._update_metadata_cache(metadata)
            
            logger.debug(f"Stored file {metadata.file_id} ({filename}) in MinIO")
            return metadata.file_id
            
        except Exception as e:
            logger.error(f"Failed to store file {metadata.file_id} in MinIO: {e}")
            raise
    
    async def retrieve_file(self, file_id: str) -> tuple[bytes, FileMetadata]:
        """Retrieve file content and metadata from MinIO"""
        try:
            if file_id not in self._metadata_cache:
                raise FileNotFoundError(f"File {file_id} not found")
            
            metadata = self._metadata_cache[file_id]
            
            # Get file from MinIO
            response = self.client.get_object(
                bucket_name=self.bucket,
                object_name=metadata.storage_path
            )
            
            content = response.read()
            response.close()
            response.release_conn()
            
            logger.debug(f"Retrieved file {file_id} from MinIO")
            return content, metadata
            
        except S3Error as e:
            if e.code == 'NoSuchKey':
                raise FileNotFoundError(f"File {file_id} not found in MinIO")
            else:
                logger.error(f"Failed to retrieve file {file_id} from MinIO: {e}")
                raise
        except Exception as e:
            logger.error(f"Failed to retrieve file {file_id} from MinIO: {e}")
            raise

    async def store_markdown_version(self, original_file_id: str, markdown_content: str) -> Optional[str]:
        """Store markdown version of a file as a separate file in MinIO"""
        try:
            if original_file_id not in self._metadata_cache:
                logger.error(f"Original file {original_file_id} not found for markdown storage")
                return None
            
            original_metadata = self._metadata_cache[original_file_id]
            
            # Create new file ID for markdown version
            markdown_file_id = str(uuid.uuid4())
            
            # Create markdown filename with format: id_original_name.md
            base_name = Path(original_metadata.filename).stem
            markdown_filename = f"{markdown_file_id}_{base_name}.md"
            
            # Store markdown content as bytes
            markdown_bytes = markdown_content.encode('utf-8')
            markdown_key = f"{self.prefix}{markdown_filename}"
            
            # Prepare metadata for MinIO
            minio_metadata = {
                'filename': markdown_filename,
                'mime-type': 'text/markdown',
                'user-id': original_metadata.user_id,
                'session-id': original_metadata.session_id or '',
                'agent-id': original_metadata.agent_id or '',
                'is-generated': 'True',
                'created-at': datetime.now().isoformat(),
                'file-id': markdown_file_id,
                'original-file-id': original_file_id
            }
            
            # Store markdown file in MinIO
            self.client.put_object(
                bucket_name=self.bucket,
                object_name=markdown_key,
                data=BytesIO(markdown_bytes),
                length=len(markdown_bytes),
                content_type="text/markdown",
                metadata=minio_metadata
            )
            
            # Create metadata for markdown file
            from datetime import datetime
            markdown_metadata = FileMetadata(
                file_id=markdown_file_id,
                filename=markdown_filename,
                mime_type="text/markdown",
                size_bytes=len(markdown_bytes),
                created_at=datetime.now(),
                updated_at=datetime.now(),
                user_id=original_metadata.user_id,
                session_id=original_metadata.session_id,
                agent_id=original_metadata.agent_id,
                is_generated=True,  # Markdown version is generated
                tags=original_metadata.tags + ["markdown-conversion", "auto-generated"],
                custom_metadata={
                    **original_metadata.custom_metadata,
                    "original_file_id": original_file_id,
                    "conversion_source": original_metadata.filename
                },
                storage_backend="minio",
                storage_path=markdown_key,
                conversion_status="success",
                conversion_timestamp=datetime.now()
            )
            
            # Update metadata cache for markdown file
            await self._update_metadata_cache(markdown_metadata)
            
            # Update original file metadata to reference markdown version
            original_metadata.markdown_file_id = markdown_file_id
            original_metadata.markdown_content = markdown_content  # Keep for quick access
            original_metadata.conversion_status = "success"
            original_metadata.conversion_timestamp = datetime.now()
            original_metadata.conversion_error = None
            original_metadata.updated_at = datetime.now()
            
            # Save updated original metadata
            await self._update_metadata_cache(original_metadata)
            
            logger.info(f"✅ Stored markdown version of {original_metadata.filename} as separate file {markdown_file_id} in MinIO")
            return markdown_file_id
            
        except Exception as e:
            logger.error(f"Failed to store markdown version for file {original_file_id} in MinIO: {e}")
            return None

    async def retrieve_markdown_version(self, original_file_id: str) -> Optional[Tuple[str, 'FileMetadata']]:
        """Retrieve markdown version of a file from MinIO"""
        try:
            if original_file_id not in self._metadata_cache:
                return None
            
            original_metadata = self._metadata_cache[original_file_id]
            
            if not original_metadata.markdown_file_id:
                return None
            
            # Retrieve markdown file
            markdown_content, markdown_metadata = await self.retrieve_file(original_metadata.markdown_file_id)
            markdown_text = markdown_content.decode('utf-8')
            
            return markdown_text, markdown_metadata
            
        except Exception as e:
            logger.error(f"Failed to retrieve markdown version for file {original_file_id} from MinIO: {e}")
            return None
    
    async def delete_file(self, file_id: str) -> bool:
        """Delete file from MinIO and metadata"""
        try:
            if file_id not in self._metadata_cache:
                return False
            
            metadata = self._metadata_cache[file_id]
            
            # Delete file from MinIO
            self.client.remove_object(
                bucket_name=self.bucket,
                object_name=metadata.storage_path
            )
            
            # Remove from metadata cache
            del self._metadata_cache[file_id]
            await self._save_metadata()
            
            logger.debug(f"Deleted file {file_id} from MinIO")
            return True
            
        except Exception as e:
            logger.error(f"Failed to delete file {file_id} from MinIO: {e}")
            return False
    
    async def list_files(self, 
                        user_id: str, 
                        session_id: Optional[str] = None,
                        agent_id: Optional[str] = None,
                        is_generated: Optional[bool] = None) -> List[FileMetadata]:
        """List files with filtering"""
        try:
            files = []
            for metadata in self._metadata_cache.values():
                # Filter by user_id
                if metadata.user_id != user_id:
                    continue
                
                # Filter by session_id if specified
                if session_id is not None and metadata.session_id != session_id:
                    continue
                
                # Filter by agent_id if specified
                if agent_id is not None and metadata.agent_id != agent_id:
                    continue
                
                # Filter by is_generated if specified
                if is_generated is not None and metadata.is_generated != is_generated:
                    continue
                
                files.append(metadata)
            
            # Sort by created_at descending
            files.sort(key=lambda x: x.created_at, reverse=True)
            return files
            
        except Exception as e:
            logger.error(f"Failed to list files from MinIO: {e}")
            return []
    
    async def update_metadata(self, file_id: str, metadata_updates: Dict[str, Any]) -> bool:
        """Update file metadata"""
        try:
            if file_id not in self._metadata_cache:
                return False
            
            metadata = self._metadata_cache[file_id]
            
            # Update allowed fields
            for key, value in metadata_updates.items():
                if hasattr(metadata, key) and key not in ['file_id', 'created_at', 'storage_path', 'storage_backend']:
                    setattr(metadata, key, value)
            
            metadata.updated_at = datetime.now()
            await self._save_metadata()
            
            logger.debug(f"Updated metadata for file {file_id} in MinIO")
            return True
            
        except Exception as e:
            logger.error(f"Failed to update metadata for file {file_id} in MinIO: {e}")
            return False
    
    async def file_exists(self, file_id: str) -> bool:
        """Check if file exists in MinIO"""
        if file_id not in self._metadata_cache:
            return False
        
        metadata = self._metadata_cache[file_id]
        
        try:
            self.client.stat_object(
                bucket_name=self.bucket,
                object_name=metadata.storage_path
            )
            return True
        except S3Error as e:
            if e.code == 'NoSuchKey':
                return False
            else:
                logger.error(f"Error checking file existence in MinIO: {e}")
                return False
    
    async def get_file_metadata(self, file_id: str) -> Optional[FileMetadata]:
        """Get file metadata without content"""
        return self._metadata_cache.get(file_id)
    
    async def convert_file_to_markdown(self, file_id: str) -> Optional[str]:
        """Convert file to markdown and return the content"""
        try:
            # Get file content and metadata
            content, metadata = await self.retrieve_file(file_id)
            
            # Import here to avoid circular import
            from agent_framework.processing.markdown_converter import markdown_converter
            
            # Convert to markdown
            markdown_content = await markdown_converter.convert_to_markdown(
                content, metadata.filename, metadata.mime_type or ""
            )
            
            if markdown_content:
                # Update metadata with conversion results
                metadata.markdown_content = markdown_content
                metadata.conversion_status = "success"
                metadata.conversion_timestamp = datetime.now()
                metadata.conversion_error = None
                
                # Save updated metadata
                await self._save_metadata()
                
                logger.info(f"✅ Converted file {file_id} to markdown")
                return markdown_content
            else:
                # Update metadata with failure
                metadata.conversion_status = "failed"
                metadata.conversion_timestamp = datetime.now()
                metadata.conversion_error = "Conversion returned empty content"
                
                # Save updated metadata
                await self._save_metadata()
                
                logger.warning(f"❌ Failed to convert file {file_id} to markdown")
                return None
                
        except Exception as e:
            logger.error(f"❌ Error converting file {file_id} to markdown: {e}")
            
            # Update metadata with error
            if file_id in self._metadata_cache:
                metadata = self._metadata_cache[file_id]
                metadata.conversion_status = "failed"
                metadata.conversion_timestamp = datetime.now()
                metadata.conversion_error = str(e)
                await self._save_metadata()
            
            return None
    
    async def analyze_image(self, file_id: str) -> Optional[Dict[str, Any]]:
        """Analyze image content using multimodal capabilities"""
        try:
            if file_id not in self._metadata_cache:
                logger.warning(f"File {file_id} not found for image analysis")
                return None
            
            metadata = self._metadata_cache[file_id]
            
            # Check if file has visual content
            if not metadata.has_visual_content and not (metadata.mime_type and metadata.mime_type.startswith('image/')):
                logger.warning(f"File {file_id} does not contain visual content")
                return None
            
            # For now, return a placeholder result indicating analysis capability
            # This will be implemented in later tasks with actual multimodal processing
            analysis_result = {
                "status": "not_implemented",
                "message": "Image analysis capability will be implemented in task 4",
                "file_id": file_id,
                "filename": metadata.filename,
                "mime_type": metadata.mime_type,
                "has_visual_content": metadata.has_visual_content
            }
            
            # Update metadata with analysis attempt
            metadata.multimodal_processing_status = "not_implemented"
            await self._save_metadata()
            
            logger.debug(f"Image analysis placeholder for file {file_id}")
            return analysis_result
            
        except Exception as e:
            logger.error(f"Error in image analysis for file {file_id}: {e}")
            return None
    
    async def get_processing_status(self, file_id: str) -> Dict[str, Any]:
        """Get comprehensive processing status for a file"""
        try:
            if file_id not in self._metadata_cache:
                return {
                    "file_id": file_id,
                    "exists": False,
                    "error": "File not found"
                }
            
            metadata = self._metadata_cache[file_id]
            
            # Compile comprehensive processing status
            status = {
                "file_id": file_id,
                "exists": True,
                "filename": metadata.filename,
                "mime_type": metadata.mime_type,
                "size_bytes": metadata.size_bytes,
                "storage_backend": metadata.storage_backend,
                "created_at": metadata.created_at.isoformat(),
                "updated_at": metadata.updated_at.isoformat(),
                
                # Conversion status
                "conversion_status": metadata.conversion_status,
                "conversion_timestamp": metadata.conversion_timestamp.isoformat() if metadata.conversion_timestamp else None,
                "conversion_error": metadata.conversion_error,
                "has_markdown_version": metadata.markdown_file_id is not None,
                "markdown_file_id": metadata.markdown_file_id,
                
                # Multimodal processing status
                "has_visual_content": metadata.has_visual_content,
                "multimodal_processing_status": metadata.multimodal_processing_status,
                "image_analysis_available": metadata.image_analysis_result is not None,
                
                # Processing errors and warnings
                "processing_errors": metadata.processing_errors,
                "processing_warnings": metadata.processing_warnings,
                "total_processing_time_ms": metadata.total_processing_time_ms,
                
                # AI generation info
                "is_generated": metadata.is_generated,
                "generation_model": metadata.generation_model,
                "generation_prompt": metadata.generation_prompt,
                "generation_parameters": metadata.generation_parameters,
                
                # Tags and metadata
                "tags": metadata.tags,
                "custom_metadata": metadata.custom_metadata
            }
            
            logger.debug(f"Retrieved processing status for file {file_id}")
            return status
            
        except Exception as e:
            logger.error(f"Error getting processing status for file {file_id}: {e}")
            return {
                "file_id": file_id,
                "exists": False,
                "error": str(e)
            }
    
    async def _load_metadata(self):
        """Load metadata from MinIO manifest"""
        try:
            response = self.client.get_object(
                bucket_name=self.bucket,
                object_name=self.metadata_key
            )
            
            content = response.read().decode('utf-8')
            response.close()
            response.release_conn()
            
            metadata_dict = json.loads(content)
            
            # Convert dict to FileMetadata objects
            self._metadata_cache = {}
            for file_id, data in metadata_dict.items():
                # Parse datetime strings
                data['created_at'] = datetime.fromisoformat(data['created_at'])
                data['updated_at'] = datetime.fromisoformat(data['updated_at'])
                
                # Handle optional datetime fields
                if 'conversion_timestamp' in data and data['conversion_timestamp']:
                    data['conversion_timestamp'] = datetime.fromisoformat(data['conversion_timestamp'])
                
                # Provide defaults for new fields to maintain backward compatibility
                data.setdefault('markdown_file_id', None)
                data.setdefault('has_visual_content', False)
                data.setdefault('image_analysis_result', None)
                data.setdefault('multimodal_processing_status', 'not_attempted')
                data.setdefault('processing_errors', [])
                data.setdefault('processing_warnings', [])
                data.setdefault('total_processing_time_ms', None)
                data.setdefault('generation_model', None)
                data.setdefault('generation_prompt', None)
                data.setdefault('generation_parameters', None)
                
                self._metadata_cache[file_id] = FileMetadata(**data)
            
            logger.debug(f"Loaded metadata for {len(self._metadata_cache)} files from MinIO")
            
        except S3Error as e:
            if e.code == 'NoSuchKey':
                logger.debug("No existing metadata found in MinIO, starting with empty cache")
                self._metadata_cache = {}
            else:
                logger.warning(f"Failed to load metadata from MinIO: {e}")
                self._metadata_cache = {}
        except Exception as e:
            logger.warning(f"Failed to load metadata from MinIO: {e}")
            self._metadata_cache = {}
    
    async def _save_metadata(self):
        """Save metadata to MinIO manifest"""
        try:
            # Convert FileMetadata objects to dict
            metadata_dict = {}
            for file_id, metadata in self._metadata_cache.items():
                data = {
                    'file_id': metadata.file_id,
                    'filename': metadata.filename,
                    'mime_type': metadata.mime_type,
                    'size_bytes': metadata.size_bytes,
                    'created_at': metadata.created_at.isoformat(),
                    'updated_at': metadata.updated_at.isoformat(),
                    'user_id': metadata.user_id,
                    'session_id': metadata.session_id,
                    'agent_id': metadata.agent_id,
                    'is_generated': metadata.is_generated,
                    'tags': metadata.tags,
                    'custom_metadata': metadata.custom_metadata,
                    'storage_backend': metadata.storage_backend,
                    'storage_path': metadata.storage_path,
                    # Enhanced markdown conversion fields
                    'markdown_content': metadata.markdown_content,
                    'markdown_file_id': metadata.markdown_file_id,
                    'conversion_status': metadata.conversion_status,
                    'conversion_timestamp': metadata.conversion_timestamp.isoformat() if metadata.conversion_timestamp else None,
                    'conversion_error': metadata.conversion_error,
                    # Multimodal processing fields
                    'has_visual_content': metadata.has_visual_content,
                    'image_analysis_result': metadata.image_analysis_result,
                    'multimodal_processing_status': metadata.multimodal_processing_status,
                    # Enhanced processing status tracking
                    'processing_errors': metadata.processing_errors,
                    'processing_warnings': metadata.processing_warnings,
                    'total_processing_time_ms': metadata.total_processing_time_ms,
                    # AI generation tracking fields
                    'generation_model': metadata.generation_model,
                    'generation_prompt': metadata.generation_prompt,
                    'generation_parameters': metadata.generation_parameters
                }
                metadata_dict[file_id] = data
            
            # Save to MinIO
            json_data = json.dumps(metadata_dict, indent=2)
            self.client.put_object(
                bucket_name=self.bucket,
                object_name=self.metadata_key,
                data=BytesIO(json_data.encode('utf-8')),
                length=len(json_data.encode('utf-8')),
                content_type='application/json'
            )
            
        except Exception as e:
            logger.error(f"Failed to save metadata to MinIO: {e}")
            raise
    
    async def _update_metadata_cache(self, metadata: FileMetadata):
        """Update metadata cache and persist"""
        self._metadata_cache[metadata.file_id] = metadata
        await self._save_metadata() 