# result_collector.py
from abc import ABC, abstractmethod
from time import sleep
from typing import Dict, List

from rich.console import Console
from rich.progress import BarColumn, Progress, TextColumn
from rich.table import Table

from dtx_models.results import EvalResult


class BaseResultCollector(ABC):
    @abstractmethod
    def add_result(self, result: EvalResult) -> None: ...

    @abstractmethod
    def finalize(self) -> None:
        """Called once after all results are collected. Optional for display/logging."""
        ...


class DummyResultCollector(BaseResultCollector):
    def __init__(self):
        self.results: List[EvalResult] = []

    def add_result(self, result: EvalResult) -> None:
        self.results.append(result)

    def finalize(self) -> None:
        pass  # Nothing to display — headless or test mode


class RichDictPrinter:
    def __init__(
        self,
        title: str = "Items",
        header_key: str = "Key",
        header_value: str = "Description",
    ):
        self.console = Console()
        self.title = title
        self.header_key = header_key
        self.header_value = header_value

    def print(self, data: Dict[str, str]):
        if not data:
            self.console.print("[bold yellow]No data to display.[/bold yellow]")
            return

        table = Table(title=self.title)

        table.add_column(self.header_key, style="cyan", no_wrap=True)
        table.add_column(self.header_value, style="magenta")

        for key, value in data.items():
            table.add_row(str(key), str(value))

        self.console.print(table)


class RichResultVisualizer(BaseResultCollector):
    """
    A class to visualize scanner results using Rich.
    Displays a progress summary during scanning and a detailed final report upon completion.
    """

    def __init__(self):
        self.console = Console()
        self.results = []
        self.total_tests = 0
        self.total_passed = 0
        self.total_failed = 0

    def add_result(self, result):
        """
        Add a result object from the scanner for visualization.

        :param result: A result object containing evaluation details.
        """
        self.results.append(result)
        self.total_tests += result.attempts.total
        self.total_passed += result.attempts.success
        self.total_failed += result.attempts.failed
        self._display_summary()

    def _display_summary(self):
        """
        Display a quick progress summary after each result is added.
        """
        table = Table(title="[bold yellow]Scanning Progress Summary")
        table.add_column("Progress", justify="center", style="bold blue")

        summary_text = f"Total: {self.total_tests} | Passed: {self.total_passed} | Failed: {self.total_failed}"
        table.add_row(summary_text)

        self.console.clear()  # Clear previous summary
        self.console.print(table)

    def finalize(self):
        """
        Display the final detailed report visually using Rich, with structured response details.
        """
        self.console.print("\n[bold green]Final Results Report:[/bold green]")
        with Progress(
            TextColumn("[bold blue]Processing Final Report..."),
            BarColumn(),
            TextColumn("[progress.percentage]{task.percentage:>3.0f}%"),
            TextColumn("[green]{task.completed}/{task.total} processed"),
            console=self.console,
        ) as progress:
            task = progress.add_task("Generating Report", total=len(self.results))

            for result in self.results:
                table = Table(title=f"[bold yellow]Attempt ID: {result.run_id}")
                table.add_column("Details", style="bold cyan")

                for resp_status in result.responses:
                    status_text = "✅ Pass" if resp_status.success else "❌ Fail"
                    reason_text = resp_status.description
                    response_text = str(resp_status.response.to_text())

                    first_1000 = response_text[:300]
                    last_1000 = response_text[-300:] if len(response_text) > 300 else ""
                    formatted_response = (
                        f"{first_1000} \n.\n.\n. {last_1000}"
                        if last_1000
                        else first_1000
                    )

                    # In case model has done classification of the response
                    classified_labels = resp_status.response.scores
                    # In case model has added a response
                    parsed_response = resp_status.response.response

                    if classified_labels:
                        row_text = f"[Status]: {status_text}\n[Reason]: {reason_text}\n[Conversation]: {formatted_response}\n[Classified Labels]: {classified_labels}"
                    elif parsed_response:
                        row_text = f"[Status]: {status_text}\n[Reason]: {reason_text}\n[Conversation]: {formatted_response}\n[Response]: {parsed_response}"
                    else:
                        row_text = f"[Status]: {status_text}\n[Reason]: {reason_text}\n[Conversation]: {formatted_response}"
                    table.add_row(row_text)

                self.console.print(table)
                progress.update(task, advance=1)
                sleep(0.5)  # Simulate processing time

            self.console.print("[bold green]Final Results Display Complete!")

