"""
This module contains functions for evaluating the correctness of model responses using various metrics.
"""

import difflib
import json
import os
import re
import string
from typing import Any, Dict, List

import cohere
import Levenshtein
import weave
from nltk import word_tokenize
from nltk.corpus import wordnet as wn
from nltk.translate import meteor
from nltk.translate.bleu_score import SmoothingFunction, sentence_bleu
from pydantic import BaseModel, field_validator
from rouge import Rouge

from .utils import extract_json_from_markdown, make_cohere_api_call

wn.ensure_loaded()


def normalize_text(text: str) -> str:
    """
    Normalize the input text by lowercasing, removing punctuation, and extra whitespace.

    Args:
        text (str): The input text to normalize.

    Returns:
        str: The normalized text.
    """
    # Convert to lowercase
    if text is None:
        return "no output"
    text = text.lower()
    text = re.sub(r"[^\w\s\d]", " ", text)
    text = text.translate(str.maketrans("", "", string.punctuation))
    text = re.sub(r"\s+", " ", text).strip()
    return text


@weave.op
def compute_diff(model_output: str, answer: str) -> float:
    """
    Compute the similarity ratio between the normalized model output and the expected answer.

    Args:
        model_output (str): The output generated by the model.
        answer (str): The expected answer.

    Returns:
        float: The similarity ratio between the normalized model output and the expected answer.
    """
    norm_output = normalize_text(model_output)
    norm_answer = normalize_text(answer)
    return difflib.SequenceMatcher(None, norm_output, norm_answer).ratio()


@weave.op
def compute_levenshtein(model_output: str, answer: str) -> float:
    """
    Compute the Levenshtein ratio between the normalized model output and the answer.

    Args:
        model_output (str): The output generated by the model.
        answer (str): The expected answer.

    Returns:
        float: The Levenshtein ratio between the normalized model output and the answer.
    """
    norm_output = normalize_text(model_output)
    norm_answer = normalize_text(answer)
    return Levenshtein.ratio(norm_output, norm_answer)


@weave.op
def compute_rouge(model_output: str, answer: str) -> float:
    """
    Compute the ROUGE-L F1 score between the normalized model output and the reference answer.

    Args:
        model_output (str): The model's generated output.
        answer (str): The reference answer.

    Returns:
        float: The ROUGE-L F1 score.
    """
    norm_output = normalize_text(model_output)
    norm_answer = normalize_text(answer)
    rouge = Rouge(metrics=["rouge-l"], stats="f")
    scores = rouge.get_scores(norm_output, norm_answer)
    return scores[0]["rouge-l"]["f"]


@weave.op
def compute_bleu(model_output: str, answer: str) -> float:
    """
    Compute the BLEU score between the normalized model output and the reference answer.

    Args:
        model_output (str): The generated output from the model.
        answer (str): The reference answer.

    Returns:
        float: The BLEU score between the normalized model output and the reference answer.
    """
    chencherry = SmoothingFunction()
    smoothing_function = chencherry.method2

    norm_output = normalize_text(model_output)
    norm_answer = normalize_text(answer)
    reference = word_tokenize(norm_answer)
    candidate = word_tokenize(norm_output)
    score = sentence_bleu([reference], candidate, smoothing_function=smoothing_function)
    return score


@weave.op
def compute_meteor(model_output: str, answer: str) -> float:
    """
    Compute the METEOR score between the normalized model output and the reference answer.

    Args:
        model_output (str): The model's generated output.
        answer (str): The reference answer.

    Returns:
        float: The METEOR score rounded to 4 decimal places.
    """
    norm_output = normalize_text(model_output)
    norm_answer = normalize_text(answer)
    reference = word_tokenize(norm_answer)
    candidate = word_tokenize(norm_output)
    meteor_score = round(meteor([candidate], reference), 4)
    return meteor_score


@weave.op
async def parse_and_validate_response(response_text: str) -> Dict[str, Any]:
    """
    Parse and validate the response text to ensure it meets the expected format.

    Args:
        response_text (str): The response text to be parsed and validated.

    Returns:
        Dict[str, Any]: The validated response as a dictionary.

    Raises:
        ValueError: If the 'final_score' is not in [0, 1, 2] or if the 'decision' is not 'correct' or 'incorrect'.
    """

    class CorrectnessScore(BaseModel):
        reason: str
        final_score: int
        decision: str

        @field_validator("final_score")
        def check_score_range(cls, v):
            if v not in [0, 1, 2]:
                raise ValueError(f"Final score must be 0, 1, or 2. Got {v}")
            return v

        @field_validator("decision")
        def check_decision(cls, v):
            if v not in ["correct", "incorrect"]:
                raise ValueError(f"Decision must be 'correct' or 'incorrect'. Got {v}")
            return v

    cleaned_text = extract_json_from_markdown(response_text)
    parsed_response = json.loads(cleaned_text)
    validated_response = CorrectnessScore(**parsed_response)
    return validated_response.model_dump()


@weave.op
async def call_cohere_with_retry(
    co_client: cohere.AsyncClientV2,
    messages: List[Dict[str, str]],
    max_retries: int = 5,
) -> Dict[str, Any]:
    """
    Call the Cohere API with retry logic.

    Args:
        co_client (cohere.AsyncClient): The Cohere asynchronous client.
        messages (List[Dict[str, str]]): The list of messages to send to the Cohere API.
        max_retries (int, optional): The maximum number of retry attempts. Defaults to 5.

    Returns:
        Dict[str, Any]: The parsed and validated response from the Cohere API.

    Raises:
        Exception: If the maximum number of retries is reached without successful validation.
    """
    response_text = ""

    for attempt in range(max_retries):
        try:
            response_text = await make_cohere_api_call(
                co_client,
                messages,
                model="command-r-plus",
                temperature=0.0,
                max_tokens=250,
            )
            return await parse_and_validate_response(response_text)
        except Exception as e:
            error_message = f"Your previous response resulted in an error:\n{str(e)}"
            error_message = (
                f"{error_message}\nEnsure that:\n1. The 'final_score' is 0, 1, or 2.\n2. The 'decision' "
                f"is either 'correct' or 'incorrect'.\n3. The 'reason' field is included.\n4. The "
                f"response is a valid JSON object, not wrapped in markdown code blocks."
            )

            if attempt == max_retries - 1:
                raise

            messages.extend(
                [
                    {"role": "assistant", "content": response_text},
                    {"role": "user", "content": error_message},
                ]
            )

    raise Exception("Max retries reached without successful validation")


@weave.op
async def evaluate_correctness_using_llm_judge(
    question: str,
    answer: str,
    model_output: str,
    prompt_file: str = "prompts/correctness_eval.json",
) -> Dict[str, Any]:
    """
    Evaluate the correctness of the model output using a language model judge.

    Args:
        question (str): The question posed to the model.
        answer (str): The reference answer.
        model_output (str): The output generated by the model.
        prompt_file (str, optional): The file containing the prompt for the language model. Defaults to "prompts/correctness_eval.json".

    Returns:
        Dict[str, Any]: The evaluation result containing the final score and decision.
    """
    co_client = cohere.AsyncClientV2(api_key=os.environ["COHERE_API_KEY"])
    messages = json.load(open(prompt_file))
    message_template = """<question>\n{question}\n</question><reference_answer>\n
    {reference_answer}\n</reference_answer>\n<generated_answer>\n{generated_answer}\n</generated_answer>"""
    messages.append(
        {
            "role": "user",
            "content": message_template.format(
                question=question,
                reference_answer=answer,
                generated_answer=model_output,
            ),
        }
    )

    return await call_cohere_with_retry(co_client, messages)


@weave.op
async def llm_response_scorer(
    model_output: str, question: str, answer: str
) -> Dict[str, Any]:
    """
    Evaluate the correctness of the model output using a language model judge.

    Args:
        model_output (str): The output generated by the model.
        question (str): The question posed to the model.
        answer (str): The reference answer.

    Returns:
        Dict[str, Any]: A dictionary containing the final score and a boolean indicating correctness.
    """
    evaluation_result = await evaluate_correctness_using_llm_judge(
        question, answer, model_output
    )

    return {
        "score": evaluation_result["final_score"],
        "correct": evaluation_result["decision"] == "correct",
    }


NLP_METRICS = [
    compute_diff,
    compute_levenshtein,
    compute_rouge,
    compute_bleu,
]

LLM_METRICS = [
    llm_response_scorer,
]

ALL_METRICS = NLP_METRICS + LLM_METRICS
