"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerImageAsset = exports.Platform = exports.NetworkMode = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const ecr = require("@aws-cdk/aws-ecr");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line
const assets_1 = require("@aws-cdk/assets");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * networking mode on build time supported by docker
 */
class NetworkMode {
    /**
     * @param mode The networking mode to use for docker build
     */
    constructor(mode) {
        this.mode = mode;
    }
    /**
     * Reuse another container's network stack
     *
     * @param containerId The target container's id or name
     */
    static fromContainer(containerId) {
        return new NetworkMode(`container:${containerId}`);
    }
    /**
     * Used to specify a custom networking mode
     * Use this if the networking mode name is not yet supported by the CDK.
     *
     * @param mode The networking mode to use for docker build
     */
    static custom(mode) {
        return new NetworkMode(mode);
    }
}
exports.NetworkMode = NetworkMode;
_a = JSII_RTTI_SYMBOL_1;
NetworkMode[_a] = { fqn: "@aws-cdk/aws-ecr-assets.NetworkMode", version: "1.202.0" };
/**
 * The default networking mode if omitted, create a network stack on the default Docker bridge
 */
NetworkMode.DEFAULT = new NetworkMode('default');
/**
 * Use the Docker host network stack
 */
NetworkMode.HOST = new NetworkMode('host');
/**
 * Disable the network stack, only the loopback device will be created
 */
NetworkMode.NONE = new NetworkMode('none');
/**
 * platform supported by docker
 */
class Platform {
    /**
     * @param platform The platform to use for docker build
     */
    constructor(platform) {
        this.platform = platform;
    }
    /**
     * Used to specify a custom platform
     * Use this if the platform name is not yet supported by the CDK.
     *
     * @param platform The platform to use for docker build
     */
    static custom(platform) {
        return new Platform(platform);
    }
}
exports.Platform = Platform;
_b = JSII_RTTI_SYMBOL_1;
Platform[_b] = { fqn: "@aws-cdk/aws-ecr-assets.Platform", version: "1.202.0" };
/**
 * Build for linux/amd64
 */
Platform.LINUX_AMD64 = new Platform('linux/amd64');
/**
 * Build for linux/arm64
 */
Platform.LINUX_ARM64 = new Platform('linux/arm64');
/**
 * An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 */
class DockerImageAsset extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ecr_assets_DockerImageAssetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DockerImageAsset);
            }
            throw error;
        }
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        this.dockerfilePath = props.file || 'Dockerfile';
        const file = path.join(dir, this.dockerfilePath);
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        const defaultIgnoreMode = core_1.FeatureFlags.of(this).isEnabled(cxapi.DOCKER_IGNORE_SUPPORT)
            ? core_1.IgnoreMode.DOCKER : core_1.IgnoreMode.GLOB;
        let ignoreMode = props.ignoreMode ?? defaultIgnoreMode;
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            const dockerIgnorePatterns = fs.readFileSync(ignore).toString().split('\n').filter(e => !!e);
            exclude = [
                ...dockerIgnorePatterns,
                ...exclude,
                // Ensure .dockerignore is included no matter what.
                '!.dockerignore',
            ];
        }
        // Ensure the Dockerfile is included no matter what.
        exclude.push('!' + path.basename(file));
        // Ensure the cdk.out folder is not included to avoid infinite loops.
        const cdkout = core_1.Stage.of(this)?.outdir ?? 'cdk.out';
        exclude.push(cdkout);
        if (props.repositoryName) {
            core_1.Annotations.of(this).addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (props.invalidation?.extraHash !== false && props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (props.invalidation?.buildArgs !== false && props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (props.invalidation?.target !== false && props.target) {
            extraHash.target = props.target;
        }
        if (props.invalidation?.file !== false && props.file) {
            extraHash.file = props.file;
        }
        if (props.invalidation?.repositoryName !== false && props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        if (props.invalidation?.networkMode !== false && props.networkMode) {
            extraHash.networkMode = props.networkMode;
        }
        if (props.invalidation?.platform !== false && props.platform) {
            extraHash.platform = props.platform;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new core_1.AssetStaging(this, 'Staging', {
            ...props,
            follow: props.followSymlinks ?? toSymlinkFollow(props.follow),
            exclude,
            ignoreMode,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.assetHash;
        this.assetHash = staging.assetHash;
        const stack = core_1.Stack.of(this);
        this.assetPath = staging.relativeStagedPath(stack);
        this.dockerBuildArgs = props.buildArgs;
        this.dockerBuildTarget = props.target;
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: this.assetPath,
            dockerBuildArgs: this.dockerBuildArgs,
            dockerBuildTarget: this.dockerBuildTarget,
            dockerFile: props.file,
            sourceHash: staging.assetHash,
            networkMode: props.networkMode?.mode,
            platform: props.platform?.platform,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/aws/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     */
    addResourceMetadata(resource, resourceProperty) {
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the resourceProperty of this resource
        // points to a local path and include the path to de dockerfile, docker build args, and target,
        // in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_DOCKERFILE_PATH_KEY] = this.dockerfilePath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_DOCKER_BUILD_ARGS_KEY] = this.dockerBuildArgs;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_DOCKER_BUILD_TARGET_KEY] = this.dockerBuildTarget;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
}
exports.DockerImageAsset = DockerImageAsset;
_c = JSII_RTTI_SYMBOL_1;
DockerImageAsset[_c] = { fqn: "@aws-cdk/aws-ecr-assets.DockerImageAsset", version: "1.202.0" };
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
function toSymlinkFollow(follow) {
    switch (follow) {
        case undefined: return undefined;
        case assets_1.FollowMode.NEVER: return core_1.SymlinkFollowMode.NEVER;
        case assets_1.FollowMode.ALWAYS: return core_1.SymlinkFollowMode.ALWAYS;
        case assets_1.FollowMode.BLOCK_EXTERNAL: return core_1.SymlinkFollowMode.BLOCK_EXTERNAL;
        case assets_1.FollowMode.EXTERNAL: return core_1.SymlinkFollowMode.EXTERNAL;
    }
}
//# sourceMappingURL=data:application/json;base64,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