"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoToApiGatewayToLambda = void 0;
const api = require("@aws-cdk/aws-apigateway");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
class CognitoToApiGatewayToLambda extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the CognitoToApiGatewayToLambda class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        // This Construct requires that the auth type be COGNITO regardless of what is specified in the props
        if (props.apiGatewayProps) {
            if (props.apiGatewayProps.defaultMethodOptions === undefined) {
                props.apiGatewayProps.defaultMethodOptions = {
                    authorizationType: api.AuthorizationType.COGNITO,
                };
            }
            else if (((_a = props.apiGatewayProps) === null || _a === void 0 ? void 0 : _a.defaultMethodOptions.authorizationType) === undefined) {
                props.apiGatewayProps.defaultMethodOptions.authorizationType = api.AuthorizationType.COGNITO;
            }
            else if (((_b = props.apiGatewayProps) === null || _b === void 0 ? void 0 : _b.defaultMethodOptions.authorizationType) !== 'COGNITO_USER_POOLS') {
                defaults.printWarning('Overriding Authorization type to be AuthorizationType.COGNITO');
                props.apiGatewayProps.defaultMethodOptions.authorizationType = api.AuthorizationType.COGNITO;
            }
        }
        if (props.apiGatewayProps && (typeof props.apiGatewayProps.proxy !== 'undefined') && (props.apiGatewayProps.proxy === false)) {
            defaults.printWarning('For non-proxy API, addAuthorizers() method must be called after all the resources and methods for API are fuly defined. Not calling addAuthorizers() will result in API methods NOT protected by Cognito.');
        }
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] =
            defaults.GlobalLambdaRestApi(this, this.lambdaFunction, props.apiGatewayProps, props.logGroupProps);
        this.userPool = defaults.buildUserPool(this, props.cognitoUserPoolProps);
        this.userPoolClient = defaults.buildUserPoolClient(this, this.userPool, props.cognitoUserPoolClientProps);
        this.apiGatewayAuthorizer = new api.CfnAuthorizer(this, 'CognitoAuthorizer', {
            restApiId: this.apiGateway.restApiId,
            type: 'COGNITO_USER_POOLS',
            providerArns: [this.userPool.userPoolArn],
            identitySource: "method.request.header.Authorization",
            name: "authorizer"
        });
        this.addAuthorizers();
    }
    addAuthorizers() {
        this.apiGateway.methods.forEach((apiMethod) => {
            // Leave the authorizer NONE for HTTP OPTIONS method to support CORS, for the rest set it to COGNITO
            const child = apiMethod.node.findChild('Resource');
            if (apiMethod.httpMethod === 'OPTIONS') {
                child.addPropertyOverride('AuthorizationType', 'NONE');
            }
            else {
                child.addPropertyOverride('AuthorizationType', 'COGNITO_USER_POOLS');
                child.addPropertyOverride('AuthorizerId', { Ref: this.apiGatewayAuthorizer.logicalId });
            }
        });
    }
}
exports.CognitoToApiGatewayToLambda = CognitoToApiGatewayToLambda;
//# sourceMappingURL=data:application/json;base64,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