"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const lib_1 = require("../lib");
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const api = require("@aws-cdk/aws-apigateway");
require("@aws-cdk/assert/jest");
function deployNewFunc(stack) {
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler'
    };
    return new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps
    });
}
test('snapshot test CognitoToApiGatewayToLambda default params', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('override cognito properties', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    const cognitoUserPoolProps = {
        userPoolName: 'test',
        userVerification: {}
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        cognitoUserPoolProps
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPool', {
        AdminCreateUserConfig: {
            AllowAdminCreateUserOnly: true
        },
        EmailVerificationMessage: "The verification code to your new account is {####}",
        EmailVerificationSubject: "Verify your new account",
        SmsVerificationMessage: "The verification code to your new account is {####}",
        UserPoolAddOns: {
            AdvancedSecurityMode: "ENFORCED"
        },
        UserPoolName: "test",
        VerificationMessageTemplate: {
            DefaultEmailOption: "CONFIRM_WITH_CODE",
            EmailMessage: "The verification code to your new account is {####}",
            EmailSubject: "Verify your new account",
            SmsMessage: "The verification code to your new account is {####}"
        }
    });
});
test('check exception for Missing existingObj from props', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.userPool !== null);
    expect(construct.userPoolClient !== null);
    expect(construct.apiGateway !== null);
    expect(construct.lambdaFunction !== null);
    expect(construct.apiGatewayCloudWatchRole !== null);
    expect(construct.apiGatewayLogGroup !== null);
    expect(construct.apiGatewayAuthorizer !== null);
});
test('override cognito cognitoUserPoolClientProps', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    const cognitoUserPoolClientProps = {
        authFlows: {
            userSrp: true
        }
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        cognitoUserPoolClientProps
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
        ExplicitAuthFlows: [
            "ALLOW_USER_SRP_AUTH",
            "ALLOW_REFRESH_TOKEN_AUTH"
        ],
    });
});
test('Check for default Cognito Auth Type', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('override Auth Type to COGNITO', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.COGNITO
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Try to override Auth Type to NONE', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Override apiGatewayProps', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                operationName: 'foo-bar'
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        OperationName: "foo-bar",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        OperationName: "foo-bar",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Override apiGatewayProps to support CORS', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultCorsPreflightOptions: {
                allowOrigins: api.Cors.ALL_ORIGINS,
                allowMethods: api.Cors.ALL_METHODS
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "ANY",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "ANY",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "NONE"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "NONE"
    });
});
test('Override apiGatewayProps with proxy = false and add POST method', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    const c = new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            proxy: false
        }
    });
    const r = c.apiGateway.root.addResource('foo');
    r.addMethod('POST');
    // Super imporant to call this method to Apply the Cognito Authorizers
    c.addAuthorizers();
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "POST",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApifoo89ACA437"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Resource', {
        PathPart: "foo",
    });
});
test('Override apiGatewayProps with proxy = false and add OPTIONS method', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    const c = new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            proxy: false
        }
    });
    const r = c.apiGateway.root.addResource('foo');
    r.addMethod('OPTIONS');
    // Mandatory to call this method to Apply the Cognito Authorizers
    c.addAuthorizers();
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApifoo89ACA437"
        },
        AuthorizationType: "NONE",
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Resource', {
        PathPart: "foo",
    });
});
//# sourceMappingURL=data:application/json;base64,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